/*
 * Decompiled with CFR 0.152.
 */
package com.procurer.managers;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;
import net.kyori.adventure.text.Component;
import net.kyori.adventure.text.format.NamedTextColor;
import net.kyori.adventure.text.format.TextColor;
import org.bukkit.Location;
import org.bukkit.Material;
import org.bukkit.Particle;
import org.bukkit.Sound;
import org.bukkit.World;
import org.bukkit.block.Block;
import org.bukkit.block.BlockState;
import org.bukkit.block.Container;
import org.bukkit.block.data.BlockData;
import org.bukkit.block.data.type.Chest;
import org.bukkit.entity.Entity;
import org.bukkit.entity.Player;
import org.bukkit.inventory.Inventory;
import org.bukkit.inventory.InventoryHolder;
import org.bukkit.inventory.ItemStack;
import org.bukkit.plugin.Plugin;
import org.bukkit.plugin.java.JavaPlugin;
import org.bukkit.scheduler.BukkitRunnable;
import org.bukkit.scheduler.BukkitTask;

public class BlockRegenerationManager {
    private final JavaPlugin plugin;
    private final Map<Location, BlockRegenData> regenerationQueue;
    private final List<RegenerationSession> activeSessions;
    private BukkitTask regenerationTask;
    private static final long REGEN_START_DELAY = 5000L;
    private static final long REGEN_DURATION = 5000L;
    private static final long TASK_INTERVAL = 2L;

    public BlockRegenerationManager(JavaPlugin plugin) {
        this.plugin = plugin;
        this.regenerationQueue = new ConcurrentHashMap<Location, BlockRegenData>();
        this.activeSessions = new ArrayList<RegenerationSession>();
        this.startRegenerationTask();
    }

    public void storeBlocksForRegeneration(Collection<Block> blocks) {
        long currentTime = System.currentTimeMillis();
        long regenStartTime = currentTime + 5000L;
        HashMap<Integer, List<BlockRegenData>> blocksByLayer = new HashMap<Integer, List<BlockRegenData>>();
        for (Block block : blocks) {
            InventoryHolder inventoryHolder;
            Inventory inventory;
            if (block.getType() == Material.AIR) continue;
            Location loc = block.getLocation().clone();
            BlockData blockDataObj = block.getBlockData().clone();
            Object inventoryContents = null;
            BlockState blockState = block.getState();
            if (blockState instanceof InventoryHolder && (inventory = (inventoryHolder = (InventoryHolder)blockState).getInventory()) != null && inventory.getSize() > 0) {
                Chest chestData;
                inventoryContents = block.getType() == Material.CHEST || block.getType() == Material.TRAPPED_CHEST ? ((chestData = (Chest)block.getBlockData()).getType() != Chest.Type.SINGLE ? (chestData.getType() == Chest.Type.LEFT ? (ItemStack[])inventory.getContents().clone() : null) : (ItemStack[])inventory.getContents().clone()) : (ItemStack[])inventory.getContents().clone();
                if (inventoryContents != null) {
                    inventory.clear();
                }
            }
            BlockRegenData regenData = new BlockRegenData(loc, block.getType(), blockDataObj, (ItemStack[])inventoryContents);
            int layer = loc.getBlockY();
            blocksByLayer.computeIfAbsent(layer, k -> new ArrayList()).add(regenData);
        }
        if (!blocksByLayer.isEmpty()) {
            RegenerationSession session = new RegenerationSession(regenStartTime, blocksByLayer);
            this.activeSessions.add(session);
            this.plugin.getLogger().info("Stored " + blocks.size() + " blocks in " + blocksByLayer.size() + " layers for regeneration");
        }
    }

    public void storeBlockForRegeneration(Block block) {
        this.storeBlocksForRegeneration(Collections.singletonList(block));
    }

    private void startRegenerationTask() {
        this.regenerationTask = new BukkitRunnable(){

            public void run() {
                BlockRegenerationManager.this.processRegeneration();
            }
        }.runTaskTimer((Plugin)this.plugin, 0L, 2L);
    }

    private void processRegeneration() {
        long currentTime = System.currentTimeMillis();
        Iterator<RegenerationSession> sessionIterator = this.activeSessions.iterator();
        while (sessionIterator.hasNext()) {
            RegenerationSession session = sessionIterator.next();
            if (currentTime < session.startTime) continue;
            if (session.isComplete()) {
                sessionIterator.remove();
                continue;
            }
            long elapsedSinceStart = currentTime - session.startTime;
            double progress = Math.min(1.0, (double)elapsedSinceStart / 5000.0);
            int totalBlocks = session.getTotalBlocks();
            int targetBlocksPlaced = (int)((double)totalBlocks * progress);
            int currentBlocksPlaced = session.getTotalBlocksPlaced();
            int blocksToPlace = targetBlocksPlaced - currentBlocksPlaced;
            for (int i = 0; i < blocksToPlace && !session.isComplete(); ++i) {
                BlockRegenData blockData = session.getNextBlock();
                if (blockData == null) continue;
                this.restoreBlock(blockData);
                this.teleportPlayersIfNeeded(blockData.location);
            }
        }
    }

    private void teleportPlayersIfNeeded(Location blockLocation) {
        World world = blockLocation.getWorld();
        if (world == null) {
            return;
        }
        Location checkLocation = blockLocation.clone();
        Location aboveLocation = blockLocation.clone().add(0.0, 1.0, 0.0);
        for (Entity entity : world.getNearbyEntities(checkLocation, 0.5, 1.0, 0.5)) {
            Player player;
            int minimumY;
            Location safeLocation;
            if (!(entity instanceof Player) || (safeLocation = this.findSafeLocationAbove(blockLocation, minimumY = (player = (Player)entity).getLocation().getBlockY())) == null) continue;
            player.teleport(safeLocation);
            player.sendMessage(Component.text((String)"Teleported to safety due to block regeneration!").color((TextColor)NamedTextColor.YELLOW));
            player.playSound(player.getLocation(), Sound.ENTITY_ENDERMAN_TELEPORT, 1.0f, 1.0f);
        }
    }

    private Location findSafeLocationAbove(Location blockLocation, int minimumY) {
        World world = blockLocation.getWorld();
        if (world == null) {
            return null;
        }
        Location directAbove = this.findSafeLocationDirectlyAbove(blockLocation, minimumY);
        if (directAbove != null) {
            return directAbove;
        }
        for (int radius = 5; radius <= 25; radius += 5) {
            for (int x = -radius; x <= radius; ++x) {
                for (int z = -radius; z <= radius; ++z) {
                    Location surfaceLocation;
                    if (Math.abs(x) != radius && Math.abs(z) != radius || (surfaceLocation = this.findSurfaceLocation(blockLocation, x, z, minimumY)) == null) continue;
                    return surfaceLocation;
                }
            }
        }
        for (int y = Math.max(blockLocation.getBlockY() + 10, minimumY); y < Math.min(world.getMaxHeight(), blockLocation.getBlockY() + 50); ++y) {
            Location checkLocation = new Location(world, (double)blockLocation.getBlockX() + 0.5, (double)y, (double)blockLocation.getBlockZ() + 0.5);
            Block feetBlock = checkLocation.getBlock();
            Block headBlock = checkLocation.clone().add(0.0, 1.0, 0.0).getBlock();
            if (feetBlock.getType() != Material.AIR || headBlock.getType() != Material.AIR) continue;
            return checkLocation;
        }
        return this.findHighestSafePoint(blockLocation, minimumY);
    }

    private Location findSafeLocationDirectlyAbove(Location blockLocation, int minimumY) {
        int startY;
        World world = blockLocation.getWorld();
        if (world == null) {
            return null;
        }
        for (int y = startY = Math.max(blockLocation.getBlockY() + 2, minimumY); y < Math.min(world.getMaxHeight(), blockLocation.getBlockY() + 20); ++y) {
            Location checkLocation = new Location(world, (double)blockLocation.getBlockX() + 0.5, (double)y, (double)blockLocation.getBlockZ() + 0.5);
            Block feetBlock = checkLocation.getBlock();
            Block headBlock = checkLocation.clone().add(0.0, 1.0, 0.0).getBlock();
            Block groundBlock = checkLocation.clone().add(0.0, -1.0, 0.0).getBlock();
            if (feetBlock.getType() != Material.AIR || headBlock.getType() != Material.AIR || !groundBlock.getType().isSolid()) continue;
            return checkLocation;
        }
        return null;
    }

    private Location findSurfaceLocation(Location blockLocation, int xOffset, int zOffset, int minimumY) {
        World world = blockLocation.getWorld();
        if (world == null) {
            return null;
        }
        int x = blockLocation.getBlockX() + xOffset;
        int z = blockLocation.getBlockZ() + zOffset;
        for (int y = Math.min(world.getMaxHeight() - 2, blockLocation.getBlockY() + 10); y >= minimumY; --y) {
            Location checkLocation = new Location(world, (double)x + 0.5, (double)y, (double)z + 0.5);
            Block feetBlock = checkLocation.getBlock();
            Block headBlock = checkLocation.clone().add(0.0, 1.0, 0.0).getBlock();
            Block groundBlock = checkLocation.clone().add(0.0, -1.0, 0.0).getBlock();
            if (feetBlock.getType() != Material.AIR || headBlock.getType() != Material.AIR || !groundBlock.getType().isSolid() || this.isLiquid(groundBlock.getType())) continue;
            return checkLocation;
        }
        return null;
    }

    private Location findHighestSafePoint(Location blockLocation, int minimumY) {
        World world = blockLocation.getWorld();
        if (world == null) {
            return null;
        }
        int highestY = minimumY - 1;
        Location bestLocation = null;
        for (int x = blockLocation.getBlockX() - 10; x <= blockLocation.getBlockX() + 10; ++x) {
            block1: for (int z = blockLocation.getBlockZ() - 10; z <= blockLocation.getBlockZ() + 10; ++z) {
                for (int y = world.getMaxHeight() - 1; y >= minimumY; --y) {
                    Block block = world.getBlockAt(x, y, z);
                    if (!block.getType().isSolid() || this.isLiquid(block.getType())) continue;
                    Block above1 = world.getBlockAt(x, y + 1, z);
                    Block above2 = world.getBlockAt(x, y + 2, z);
                    if (above1.getType() != Material.AIR || above2.getType() != Material.AIR || y <= highestY) continue block1;
                    highestY = y;
                    bestLocation = new Location(world, (double)x + 0.5, (double)(y + 1), (double)z + 0.5);
                    continue block1;
                }
            }
        }
        return bestLocation;
    }

    private boolean isLiquid(Material material) {
        return material == Material.WATER || material == Material.LAVA || material.name().contains("WATER") || material.name().contains("LAVA");
    }

    private void restoreBlock(final BlockRegenData regenData) {
        final Block block = regenData.location.getBlock();
        block.setType(regenData.originalMaterial);
        block.setBlockData(regenData.originalBlockData);
        if (regenData.inventoryContents != null) {
            new BukkitRunnable(this){
                final /* synthetic */ BlockRegenerationManager this$0;
                {
                    this.this$0 = this$0;
                }

                public void run() {
                    InventoryHolder inventoryHolder;
                    Inventory inventory;
                    BlockState blockState = block.getState();
                    if (blockState instanceof InventoryHolder && (inventory = (inventoryHolder = (InventoryHolder)blockState).getInventory()) != null) {
                        if (block.getType() == Material.CHEST || block.getType() == Material.TRAPPED_CHEST) {
                            Chest chestData = (Chest)block.getBlockData();
                            if (chestData.getType() == Chest.Type.LEFT) {
                                this.this$0.restoreInventoryContents(inventory, regenData.inventoryContents);
                            } else if (chestData.getType() == Chest.Type.SINGLE) {
                                ItemStack[] singleChestContents = new ItemStack[27];
                                System.arraycopy(regenData.inventoryContents, 0, singleChestContents, 0, Math.min(regenData.inventoryContents.length, 27));
                                this.this$0.restoreInventoryContents(inventory, singleChestContents);
                            }
                        } else {
                            this.this$0.restoreInventoryContents(inventory, regenData.inventoryContents);
                        }
                        BlockState blockState2 = block.getState();
                        if (blockState2 instanceof Container) {
                            Container container = (Container)blockState2;
                            container.update();
                        }
                        this.this$0.scheduleInventoryVerification(block, regenData.inventoryContents);
                    }
                }
            }.runTaskLater((Plugin)this.plugin, 1L);
        }
        Location particleLocation = regenData.location.clone().add(0.5, 0.5, 0.5);
        regenData.location.getWorld().spawnParticle(Particle.CLOUD, particleLocation, 3, 0.2, 0.2, 0.2, 0.0);
        regenData.location.getWorld().playSound(regenData.location, Sound.BLOCK_STONE_PLACE, 0.5f, 1.0f);
    }

    private void restoreInventoryContents(Inventory inventory, ItemStack[] contents) {
        try {
            inventory.clear();
            int maxSlots = Math.min(inventory.getSize(), contents.length);
            for (int i = 0; i < maxSlots; ++i) {
                if (contents[i] == null) continue;
                inventory.setItem(i, contents[i].clone());
            }
            InventoryHolder inventoryHolder = inventory.getHolder();
            if (inventoryHolder instanceof BlockState) {
                BlockState blockState = (BlockState)inventoryHolder;
                blockState.update(true, false);
            }
        }
        catch (Exception e) {
            this.plugin.getLogger().warning("Failed to restore inventory contents: " + e.getMessage());
            try {
                for (int i = 0; i < Math.min(inventory.getSize(), contents.length); ++i) {
                    if (contents[i] == null || inventory.getItem(i) != null) continue;
                    inventory.setItem(i, contents[i]);
                }
            }
            catch (Exception e2) {
                this.plugin.getLogger().severe("Critical failure in inventory restoration: " + e2.getMessage());
            }
        }
    }

    public void clearRegenerationArea(Location center, double radius) {
        this.activeSessions.removeIf(session -> session.blocksByLayer.values().stream().anyMatch(blocks -> blocks.stream().anyMatch(blockData -> {
            Location loc = blockData.location;
            return loc.getWorld().equals((Object)center.getWorld()) && loc.distance(center) <= radius;
        })));
    }

    private void scheduleInventoryVerification(final Block block, final ItemStack[] expectedContents) {
        new BukkitRunnable(this){
            final /* synthetic */ BlockRegenerationManager this$0;
            {
                this.this$0 = this$0;
            }

            public void run() {
                BlockState blockState = block.getState();
                if (blockState instanceof InventoryHolder) {
                    InventoryHolder inventoryHolder = (InventoryHolder)blockState;
                    Inventory inventory = inventoryHolder.getInventory();
                    int expectedItemCount = 0;
                    for (ItemStack item : expectedContents) {
                        if (item == null) continue;
                        ++expectedItemCount;
                    }
                    int actualItemCount = 0;
                    for (ItemStack item : inventory.getContents()) {
                        if (item == null) continue;
                        ++actualItemCount;
                    }
                    if (actualItemCount < expectedItemCount) {
                        this.this$0.plugin.getLogger().warning("Inventory verification failed for " + String.valueOf(block.getType()) + " at " + String.valueOf(block.getLocation()) + ". Expected " + expectedItemCount + " items, found " + actualItemCount + ". Retrying restoration...");
                        this.this$0.restoreInventoryContents(inventory, expectedContents);
                        BlockState blockState2 = block.getState();
                        if (blockState2 instanceof Container) {
                            Container container = (Container)blockState2;
                            container.update(true, false);
                        }
                    } else {
                        this.this$0.plugin.getLogger().info("Successfully verified inventory restoration for " + String.valueOf(block.getType()) + " at " + String.valueOf(block.getLocation()));
                    }
                }
            }
        }.runTaskLater((Plugin)this.plugin, 10L);
    }

    public int getQueuedBlockCount() {
        return this.activeSessions.stream().mapToInt(RegenerationSession::getTotalBlocks).sum();
    }

    public boolean isQueuedForRegeneration(Location location) {
        return this.activeSessions.stream().anyMatch(session -> session.blocksByLayer.values().stream().anyMatch(blocks -> blocks.stream().anyMatch(blockData -> blockData.location.equals((Object)location))));
    }

    public void shutdown() {
        if (this.regenerationTask != null) {
            this.regenerationTask.cancel();
        }
        this.regenerationQueue.clear();
        this.activeSessions.clear();
    }

    private static class BlockRegenData {
        final Location location;
        final Material originalMaterial;
        final BlockData originalBlockData;
        final ItemStack[] inventoryContents;

        BlockRegenData(Location location, Material originalMaterial, BlockData originalBlockData, ItemStack[] inventoryContents) {
            this.location = location;
            this.originalMaterial = originalMaterial;
            this.originalBlockData = originalBlockData;
            this.inventoryContents = inventoryContents;
        }
    }

    private static class RegenerationSession {
        final long startTime;
        final Map<Integer, List<BlockRegenData>> blocksByLayer;
        final List<Integer> layerOrder;
        int currentLayerIndex = 0;
        int currentBlockInLayer = 0;
        int totalBlocksPlaced = 0;

        RegenerationSession(long startTime, Map<Integer, List<BlockRegenData>> blocksByLayer) {
            this.startTime = startTime;
            this.blocksByLayer = blocksByLayer;
            this.layerOrder = blocksByLayer.keySet().stream().sorted().collect(Collectors.toList());
            blocksByLayer.values().forEach(Collections::shuffle);
        }

        boolean isComplete() {
            return this.currentLayerIndex >= this.layerOrder.size();
        }

        BlockRegenData getNextBlock() {
            if (this.isComplete()) {
                return null;
            }
            int currentY = this.layerOrder.get(this.currentLayerIndex);
            List<BlockRegenData> currentLayerBlocks = this.blocksByLayer.get(currentY);
            if (this.currentBlockInLayer >= currentLayerBlocks.size()) {
                ++this.currentLayerIndex;
                this.currentBlockInLayer = 0;
                return this.getNextBlock();
            }
            BlockRegenData block = currentLayerBlocks.get(this.currentBlockInLayer);
            ++this.currentBlockInLayer;
            ++this.totalBlocksPlaced;
            return block;
        }

        int getTotalBlocks() {
            return this.blocksByLayer.values().stream().mapToInt(List::size).sum();
        }

        int getTotalBlocksPlaced() {
            return this.totalBlocksPlaced;
        }
    }
}

