package dev.zenfyr.andromeda.common.config;

import com.google.common.collect.Maps;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import dev.zenfyr.andromeda.bootstrap.Module;
import dev.zenfyr.andromeda.bootstrap.ModuleManager;
import dev.zenfyr.andromeda.bootstrap.config.BaseConfig;
import dev.zenfyr.andromeda.bootstrap.config.ConfigDefinition;
import dev.zenfyr.andromeda.common.Andromeda;
import dev.zenfyr.andromeda.common.config.handler.GameConfigHandler;
import dev.zenfyr.andromeda.common.util.IdentifiedJsonDataLoader;
import dev.zenfyr.andromeda.util.Util;
import dev.zenfyr.pulsar.resources.ReloaderType;
import dev.zenfyr.pulsar.resources.ServerReloadersEvent;
import it.unimi.dsi.fastutil.objects.ReferenceLinkedOpenHashSet;
import it.unimi.dsi.fastutil.objects.ReferenceOpenHashSet;
import java.lang.reflect.Field;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import net.fabricmc.fabric.api.event.lifecycle.v1.ServerLifecycleEvents;
import net.minecraft.class_1937;
import net.minecraft.class_2960;
import net.minecraft.class_3218;
import net.minecraft.class_3300;
import net.minecraft.class_3695;
import net.minecraft.server.MinecraftServer;
// Loads and applies custom config overrides from data packs
public final class DataConfigs extends IdentifiedJsonDataLoader {
  private static final org.apache.logging.log4j.Logger log = dev.zenfyr.andromeda.util.Util.logger();
  public static final class_2960 DEFAULT = Andromeda.id("default");
  public static final ReloaderType<DataConfigs> RELOADER = ReloaderType.create(Andromeda.id("scoped_config"));

  public static DataConfigs get(MinecraftServer server) {
    return server.pulsar$getReloader(RELOADER);
  }

  private final ModuleManager moduleManager;
  public Map<class_2960, Map<Module, Set<Data>>> configs;
  public Map<Module, Set<Data>> defaultConfigs;

  public DataConfigs(ModuleManager moduleManager) {
    super(RELOADER.location());
    this.moduleManager = moduleManager;
  }

  @Override
  protected void apply(Map<class_2960, JsonElement> data, class_3300 manager, class_3695 profiler) {
    Map<class_2960, Map<Module, Set<Data>>> parsed = new HashMap<>();
    for (var entry : Maps.transformValues(data, JsonElement::getAsJsonObject).entrySet()) {
      class_2960 id = entry.getKey();
      JsonObject json = entry.getValue();
      // Modules must be loaded to apply their configs.
      var module = this.moduleManager.get(id.method_12832()).orElseThrow(() -> new IllegalStateException("Invalid module path \'%s\'! The module must be enabled!".formatted(id.method_12832())));
      var type = Andromeda.GAME.getDefinition(module).supplier().get();
      Maps.transformValues(json.asMap(), JsonElement::getAsJsonObject).forEach((string, value) -> {
        var dimension = new class_2960(string);
        var cfg = Andromeda.GAME.gson().fromJson(value, type);
        // Parse the fields that must be modified during `apply`
        Set<Field> overrides = new ReferenceOpenHashSet<>();
        for (String field : value.keySet()) {
          try {
            overrides.add(type.getField(field));
          } catch (NoSuchFieldException e) {
            throw Util.wrap("No such field \'%s\' for module %s".formatted(field, id), e);
          }
        }
        // We store configs grouped by dimension.
        parsed.computeIfAbsent(dimension, i_ -> new HashMap<>()).computeIfAbsent(module, m_ -> new ReferenceLinkedOpenHashSet<>()).add(new Data(overrides, cfg));
      });
    }
    this.defaultConfigs = parsed.remove(DEFAULT);
    this.configs = parsed;
  }

  public void applyConfigs(AttachmentGetter getter, class_2960 dimension) {
    Objects.requireNonNull(configs);
    var handler = getter.andromeda$getConfigs();
    handler.loadAll();
    handler.forEach((module, baseConfig) -> this.applyDataPacks(baseConfig, module, dimension));
  }

  void applyDataPacks(BaseConfig config, Module module, class_2960 dimension) {
    if (defaultConfigs != null) {
      var forModule = defaultConfigs.get(module);
      if (forModule != null) for (Data data : forModule) this.apply(config, data);
    }
    if (dimension.equals(DEFAULT)) return;
    var overrides = Objects.requireNonNull(configs).get(dimension);
    if (overrides != null) {
      var forModule = overrides.get(module);
      if (forModule != null) for (Data data1 : forModule) apply(config, data1);
    }
  }

  private void apply(BaseConfig config, Data data) {
    data.cFields().forEach(field -> {
      try {
        field.set(config, field.get(data.config()));
      } catch (IllegalAccessException e) {
        throw new RuntimeException("Failed to apply config data for module \'%s\'".formatted(config.getClass().getSimpleName()), e);
      }
    });
  }


  public record Data(Set<Field> cFields, BaseConfig config) {
  }


  public interface WorldExtension {
    default <T extends BaseConfig> T am$get(ConfigDefinition<T> definition) {
      log.error("Scoped configs requested on client in world \'{}\'! Returning un-scoped!", ((class_1937) this).method_27983().method_29177());
      return Andromeda.MAIN.get(definition); // Stub implementation. DNI
    }
  }


  public interface AttachmentGetter {
    GameConfigHandler andromeda$getConfigs();
  }

  public static void init(ModuleManager manager) {
    ServerReloadersEvent.EVENT.register(context -> context.register(new DataConfigs(manager)));
    ServerLifecycleEvents.END_DATA_PACK_RELOAD.register((server, resourceManager, success) -> {
      if (!success) return;
      var configs = DataConfigs.get(server);
      for (class_3218 world : server.method_3738()) {
        configs.applyConfigs((AttachmentGetter) world, world.method_27983().method_29177());
      }
    });
  }
}
