package dev.zenfyr.andromeda.util;

import java.io.IOException;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.*;

public final class ClassPath {
  private final Set<Path> scanned = new HashSet<>();
  private final Set<Info> infos = new TreeSet<>(Comparator.comparing(info -> info.name));

  public ClassPath() {
  }

  public Set<Info> getTopLevelRecursive(String pckg) {
    String s = pckg.replace('/', '.');
    Set<Info> set = new TreeSet<>(Comparator.comparing(info -> info.name));
    for (Info info : this.infos) {
      if (info.getName().startsWith(s)) set.add(info);
    }
    return Collections.unmodifiableSet(set);
  }

  public void addPaths(Collection<Path> paths) {
    if (paths == null || paths.isEmpty()) return;
    for (Path path : paths) {
      if (this.scanned.contains(path)) return;
      if (Files.isDirectory(path)) {
        scan(path);
      } else {
        scanJar(path);
      }
      synchronized (this.scanned) {
        this.scanned.add(path);
      }
    }
  }

  private void scanJar(Path path) {
    try {
      try (var delegate = FileSystemUtil.getJarFileSystem(path, false)) {
        scan(delegate.get().getRootDirectories().iterator().next());
      }
    } catch (final java.lang.Throwable $ex) {
      throw lombok.Lombok.sneakyThrow($ex);
    }
  }

  private void scan(Path path) {
    try {
      Files.walkFileTree(path, new SimpleFileVisitor<>() {
        @Override
        public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) {
          String name = path.relativize(file).toString();
          if (name.endsWith(".class") && !name.contains("$")) {
            String clsName = name.substring(0, name.length() - ".class".length()).replace('\\', '.').replace('/', '.');
            synchronized (ClassPath.this.infos) {
              ClassPath.this.infos.add(new Info(file, clsName));
            }
          }
          return FileVisitResult.CONTINUE;
        }
      });
    } catch (final java.lang.Throwable $ex) {
      throw lombok.Lombok.sneakyThrow($ex);
    }
  }


  public static class Info {
    private final Path path;
    private final String name;

    Info(Path path, String name) {
      this.path = path;
      this.name = name;
    }

    public byte[] readAllBytes() throws IOException {
      return Files.readAllBytes(path);
    }

    public String packageName() {
      int c = name.lastIndexOf('.');
      return c < 0 ? "" : name.substring(0, c);
    }

    public Path getPath() {
      return this.path;
    }

    public String getName() {
      return this.name;
    }
  }

  @Override
  public String toString() {
    return "ClassPath(scanned=" + this.scanned + ", infos=" + this.infos + ")";
  }
}
