package com.github.tartaricacid.touhoulittlemaid.inventory.handler;

import cn.sh1rocu.touhoulittlemaid.util.itemhandler.ItemStackHandler;
import com.github.tartaricacid.touhoulittlemaid.api.bauble.IMaidBauble;
import com.github.tartaricacid.touhoulittlemaid.entity.passive.EntityMaid;
import com.github.tartaricacid.touhoulittlemaid.item.bauble.BaubleManager;
import com.google.common.collect.Sets;
import it.unimi.dsi.fastutil.ints.Int2ObjectRBTreeMap;
import it.unimi.dsi.fastutil.ints.Int2ObjectSortedMap;
import org.jetbrains.annotations.ApiStatus;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_2371;
import net.minecraft.class_2487;
import net.minecraft.class_2520;
import java.util.Set;
import java.util.function.BiPredicate;
import java.util.stream.IntStream;

public class BaubleItemHandler extends ItemStackHandler {
    /**
     * 存储 IMaidBauble 对象的数组，该数组和饰品栏不同等大小
     */
    private final Int2ObjectSortedMap<IMaidBauble> baubles = new Int2ObjectRBTreeMap<>();

    /**
     * 存储所有物品的缓存集合，用于提高查询效率
     */
    private final Set<class_1792> baubleItemsCache = Sets.newHashSet();

    /**
     * 构建默认大小（1 格）的饰品栏
     */
    public BaubleItemHandler() {
        this(1);
    }

    /**
     * 构建 size 大小的饰品栏
     *
     * @param size 饰品栏大小
     */
    public BaubleItemHandler(int size) {
        super(size);
    }

    /**
     * 通过输入的 NonNullList<ItemStack> 构建饰品栏大小
     *
     * @param stacks 输入的 NonNullList<ItemStack>
     */
    public BaubleItemHandler(class_2371<class_1799> stacks) {
        super(stacks);
        IntStream.range(0, getSlots()).forEach(this::onContentsChanged);
    }

    /**
     * 设定指定格子的物品的 IMaidBauble 对象
     *
     * @param slot   指定的格子
     * @param bauble 设定的 IMaidBauble 对象
     */
    private void setBaubleInSlot(int slot, @Nullable IMaidBauble bauble) {
        validateSlotIndex(slot);
        if (bauble == null) {
            baubles.remove(slot);
        } else {
            baubles.put(slot, bauble);
        }
    }

    /**
     * 获取指定格子的 IMaidBauble 对象
     *
     * @param slot 指定的格子
     * @return 获取的 IMaidBauble 对象
     */
    @Nullable
    public IMaidBauble getBaubleInSlot(int slot) {
        class_1799 stack = getStackInSlot(slot);
        if (stack.method_7960()) {
            return null;
        } else {
            return baubles.get(slot);
        }
    }

    /**
     * 当内容改变时触发的方法
     *
     * @param slot 触发的格子
     */
    @Override
    protected void onContentsChanged(int slot) {
        // 更新饰品信息
        this.updateBaubles(slot);
        // 更新物品缓存
        this.updateBaublesCache();
    }

    /**
     * 更新指定格子的饰品信息
     *
     * @param slot 指定的格子
     */
    protected void updateBaubles(int slot) {
        class_1799 stack = getStackInSlot(slot);
        if (stack.method_7960()) {
            setBaubleInSlot(slot, null);
        } else {
            setBaubleInSlot(slot, BaubleManager.getBauble(stack));
        }
    }

    protected void updateBaublesCache() {
        baubleItemsCache.clear();
        for (int baubleSlot : baubles.keySet()) {
            class_1799 stack = getStackInSlot(baubleSlot);
            if (!stack.method_7960()) {
                baubleItemsCache.add(stack.method_7909());
            }
        }
    }

    /**
     * 物品是否合法
     *
     * @param slot  格子
     * @param stack 传入的物品堆
     * @return 物品是否合法
     */
    @Override
    public boolean isItemValid(int slot, @Nonnull class_1799 stack) {
        return BaubleManager.getBauble(stack) != null;
    }

    /**
     * 插入物品时的逻辑
     */
    @Override
    @Nonnull
    public class_1799 insertItem(int slot, @Nonnull class_1799 stack, boolean simulate) {
        if (isItemValid(slot, stack)) {
            return super.insertItem(slot, stack, simulate);
        } else {
            return stack;
        }
    }

    /**
     * 处理反序列化时的饰品加载
     */
    @Override
    protected void onLoad() {
        IntStream.range(0, getSlots()).forEach(this::updateBaubles);
        this.updateBaublesCache();
    }

    public boolean fireEvent(BiPredicate<IMaidBauble, class_1799> function) {
        var iterator = baubles.int2ObjectEntrySet().iterator();
        while (iterator.hasNext()) {
            var entry = iterator.next();
            int slot = entry.getIntKey();

            IMaidBauble bauble = entry.getValue();
            class_1799 stack = getStackInSlot(slot);

            if (stack.method_7960()) {
                // 删除不存在物品的映射
                iterator.remove();
                continue;
            }

            if (function.test(bauble, stack)) {
                return true;
            }
        }
        return false;
    }

    public int getBaubleSlot(IMaidBauble bauble) {
        for (var entry : baubles.int2ObjectEntrySet()) {
            if (entry.getValue() == bauble) {
                return entry.getIntKey();
            }
        }
        return -1;
    }

    @ApiStatus.AvailableSince("1.4.3")
    public boolean containsItem(class_1792 item) {
        return baubleItemsCache.contains(item);
    }

    @Override
    public void readFromNbt(class_2487 tag) {
        if (tag.method_10545(TAG_INVENTORY)) {
            class_2487 baubleTag = tag.method_10562(TAG_INVENTORY);
            if (baubleTag.method_10573("Size", class_2520.field_33253)) {
                // 1.4.2 版本起，饰品栏拓展了数量，需要在这里进行修正
                int oldSize = baubleTag.method_10550("Size");
                if (oldSize < EntityMaid.BAUBLE_INV_SIZE) {
                    baubleTag.method_10569("Size", EntityMaid.BAUBLE_INV_SIZE);
                }
            }
            this.deserializeNBT(baubleTag);
        }
    }
}