package com.github.tartaricacid.touhoulittlemaid.client.resource;

import com.github.tartaricacid.simplebedrockmodel.client.bedrock.pojo.BedrockModelPOJO;
import com.github.tartaricacid.simplebedrockmodel.client.bedrock.pojo.BedrockVersion;
import com.github.tartaricacid.simplebedrockmodel.client.bedrock.pojo.CubesItem;
import com.github.tartaricacid.touhoulittlemaid.TouhouLittleMaid;
import com.github.tartaricacid.touhoulittlemaid.client.animation.CustomJsAnimationManger;
import com.github.tartaricacid.touhoulittlemaid.client.animation.gecko.condition.ConditionManager;
import com.github.tartaricacid.touhoulittlemaid.client.gui.entity.cache.CacheIconManager;
import com.github.tartaricacid.touhoulittlemaid.client.model.bedrock.BedrockModel;
import com.github.tartaricacid.touhoulittlemaid.client.renderer.texture.FilePackTexture;
import com.github.tartaricacid.touhoulittlemaid.client.renderer.texture.ZipPackTexture;
import com.github.tartaricacid.touhoulittlemaid.client.resource.models.ChairModels;
import com.github.tartaricacid.touhoulittlemaid.client.resource.models.MaidModels;
import com.github.tartaricacid.touhoulittlemaid.client.resource.pojo.ChairModelInfo;
import com.github.tartaricacid.touhoulittlemaid.client.resource.pojo.CustomModelPack;
import com.github.tartaricacid.touhoulittlemaid.client.resource.pojo.MaidModelInfo;
import com.github.tartaricacid.touhoulittlemaid.client.sound.CustomSoundLoader;
import com.github.tartaricacid.touhoulittlemaid.entity.item.EntityChair;
import com.github.tartaricacid.touhoulittlemaid.geckolib3.file.AnimationFile;
import com.github.tartaricacid.touhoulittlemaid.util.ZipFileCheck;
import com.google.common.collect.Sets;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonSyntaxException;
import com.google.gson.reflect.TypeToken;
import net.fabricmc.api.EnvType;
import net.fabricmc.api.Environment;
import net.minecraft.class_1308;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.Marker;
import org.apache.logging.log4j.MarkerManager;

import javax.annotation.Nullable;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Enumeration;
import java.util.List;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

import static com.github.tartaricacid.touhoulittlemaid.TouhouLittleMaid.LOGGER;

@Environment(EnvType.CLIENT)
public class CustomPackLoader {
    public static final Gson GSON = new GsonBuilder()
            .registerTypeAdapter(class_2960.class, new class_2960.class_2961())
            .registerTypeAdapter(CubesItem.class, new CubesItem.Deserializer())
            .create();
    public static final MaidModels MAID_MODELS = MaidModels.getInstance();
    public static final ChairModels CHAIR_MODELS = ChairModels.getInstance();
    private static final Set<class_2960> TMP_REGISTER_TEXTURE = Sets.newHashSet();
    private static final String CUSTOM_PACK_DIR_NAME = "tlm_custom_pack";
    public static final Path PACK_FOLDER = Paths.get(class_310.method_1551().field_1697.toURI()).resolve(CUSTOM_PACK_DIR_NAME);
    private static final Marker MARKER = MarkerManager.getMarker("CustomPackLoader");
    private static final Pattern DOMAIN = Pattern.compile("^assets/([\\w.]+)/$");

    public static void reloadPacks() {
        // 清除
        CustomJsAnimationManger.clearAll();
        ConditionManager.clear();
        MAID_MODELS.clearAll();
        CHAIR_MODELS.clearAll();
        TMP_REGISTER_TEXTURE.clear();
        LanguageLoader.clear();
        CustomSoundLoader.clear();
        CacheIconManager.clearCache();

        // 读取
        loadPacks(PACK_FOLDER.toFile());
        LanguageLoader.loadDownloadInfoLanguages();

        // 对读取的列表进行排序，把默认模型包排在最前面
        // 其他模型包按照 namespace 字典排序
        MAID_MODELS.sortPackList();
        CHAIR_MODELS.sortPackList();
        CustomSoundLoader.sortSoundPack();
    }

    private static void loadPacks(File packFolder) {
        File[] files = packFolder.listFiles((dir, name) -> true);
        if (files == null) {
            return;
        }
        for (File file : files) {
            if (file.isFile() && file.getName().endsWith(".zip")) {
                try {
                    if (ZipFileCheck.isZipFile(file)) {
                        readModelFromZipFile(file);
                    } else {
                        TouhouLittleMaid.LOGGER.error("{} file is corrupt and cannot be loaded.", file.getName());
                    }
                } catch (IOException ioException) {
                    ioException.printStackTrace();
                }
            }
            if (file.isDirectory()) {
                readModelFromFolder(file);
            }
        }
    }

    public static void readModelFromFolder(File root) {
        try {
            File[] domainFiles = root.toPath().resolve("assets").toFile().listFiles((dir, name) -> true);
            if (domainFiles == null) {
                return;
            }
            for (File domainDir : domainFiles) {
                if (domainDir.isDirectory()) {
                    Path rootPath = root.toPath();
                    String domain = domainDir.getName();
                    loadMaidModelPack(rootPath, domain);
                    loadChairModelPack(rootPath, domain);
                    LanguageLoader.readLanguageFile(rootPath, domain);
                    CustomSoundLoader.loadSoundPack(rootPath, domain);
                }
            }
        } catch (IOException ioException) {
            ioException.printStackTrace();
        }
    }

    public static void readModelFromZipFile(File file) {
        try (ZipFile zipFile = new ZipFile(file)) {
            Enumeration<? extends ZipEntry> iteration = zipFile.entries();
            while (iteration.hasMoreElements()) {
                String filePath = iteration.nextElement().getName();
                Matcher matcher = DOMAIN.matcher(filePath);
                if (matcher.find()) {
                    String domain = matcher.group(1);
                    loadMaidModelPack(zipFile, domain);
                    loadChairModelPack(zipFile, domain);
                    CustomSoundLoader.loadSoundPack(zipFile, domain);
                    continue;
                }
                // 语言文件单独加载
                LanguageLoader.readLanguageFile(zipFile, filePath);
            }
        } catch (IOException ioException) {
            ioException.printStackTrace();
        }
    }

    private static void loadMaidModelPack(Path rootPath, String domain) {
        LOGGER.debug(MARKER, "Touhou little maid mod's model is loading...");
        File file = rootPath.resolve("assets").resolve(domain).resolve(MAID_MODELS.getJsonFileName()).toFile();
        if (!file.isFile()) {
            return;
        }
        try (InputStream stream = Files.newInputStream(file.toPath())) {
            CustomModelPack<MaidModelInfo> pack = GSON.fromJson(new InputStreamReader(stream, StandardCharsets.UTF_8),
                    new TypeToken<CustomModelPack<MaidModelInfo>>() {
                    }.getType());
            pack.decorate(domain);
            // 加载图标贴图
            if (pack.getIcon() != null) {
                registerFilePackTexture(rootPath, pack.getIcon());
            }
            for (MaidModelInfo maidModelItem : pack.getModelList()) {
                if (maidModelItem.isGeckoModel()) {
                    loadGeckoMaidModelElement(rootPath, maidModelItem);
                } else {
                    loadMaidModelElement(rootPath, maidModelItem);
                }
            }
            MAID_MODELS.addPack(pack);
        } catch (IOException e) {
            e.printStackTrace();
        } catch (JsonSyntaxException e) {
            LOGGER.warn(MARKER, "Fail to parse model pack in domain {}", domain);
            e.printStackTrace();
        }
        LOGGER.debug(MARKER, "Touhou little maid mod's model is loaded");
    }

    private static void loadMaidModelElement(Path rootPath, MaidModelInfo maidModelItem) {
        // 尝试加载模型
        BedrockModel<class_1308> modelJson = loadMaidModel(rootPath, maidModelItem.getModel());
        // 加载贴图
        registerFilePackTexture(rootPath, maidModelItem.getTexture());
        // 加载动画
        @Nullable List<Object> animations = CustomJsAnimationManger.getCustomAnimation(rootPath, maidModelItem);
        if (modelJson != null) {
            // 加载彩蛋，彩蛋不允许为空
            if (maidModelItem.getEasterEgg() != null && StringUtils.isNotBlank(maidModelItem.getEasterEgg().getTag())) {
                putMaidEasterEggData(maidModelItem, modelJson, animations);
            } else {
                putMaidModelData(maidModelItem, modelJson, animations);
            }
            // 打印日志
            LOGGER.debug(MARKER, "Loaded model: {}", maidModelItem.getModel());
        }
    }

    private static void loadGeckoMaidModelElement(Path rootPath, MaidModelInfo maidModelItem) throws IOException {
        class_2960 uid = maidModelItem.getModelId();
        // 尝试加载模型
        class_2960 modelLocation = maidModelItem.getModel();
        File modelFile = rootPath.resolve("assets").resolve(modelLocation.method_12836()).resolve(modelLocation.method_12832()).toFile();
        if (!modelFile.isFile()) {
            return;
        }
        try (InputStream fileInputStream = Files.newInputStream(modelFile.toPath())) {
            GeckoModelLoader.registerGeo(uid, fileInputStream);
        }
        // 加载贴图
        registerFilePackTexture(rootPath, maidModelItem.getTexture());
        // 加载动画
        List<class_2960> animation = maidModelItem.getAnimation();
        if (animation == null || animation.isEmpty()) {
            return;
        }
        AnimationFile animationData = new AnimationFile();
        for (class_2960 animationPath : animation) {
            if (animationPath.equals(GeckoModelLoader.DEFAULT_MAID_ANIMATION)) {
                break;
            }
            File animationFile = rootPath.resolve("assets").resolve(animationPath.method_12836()).resolve(animationPath.method_12832()).toFile();
            if (!animationFile.isFile()) {
                continue;
            }
            try (InputStream fileInputStream = Files.newInputStream(animationFile.toPath())) {
                GeckoModelLoader.mergeAnimationFile(fileInputStream, animationData);
            }
        }
        GeckoModelLoader.registerMaidAnimations(uid, animationData);
        if (maidModelItem.getEasterEgg() != null && StringUtils.isNotBlank(maidModelItem.getEasterEgg().getTag())) {
            putMaidEasterEggData(maidModelItem, null, null);
        } else {
            MAID_MODELS.putInfo(uid.toString(), maidModelItem);
        }
        // 打印日志
        LOGGER.debug(MARKER, "Loaded model: {}", maidModelItem.getModel());
    }

    private static void loadGeckoChairModelElement(Path rootPath, ChairModelInfo chairModelItem) throws IOException {
        class_2960 uid = chairModelItem.getModelId();
        // 尝试加载模型
        class_2960 modelLocation = chairModelItem.getModel();
        File modelFile = rootPath.resolve("assets").resolve(modelLocation.method_12836()).resolve(modelLocation.method_12832()).toFile();
        if (!modelFile.isFile()) {
            return;
        }
        try (InputStream fileInputStream = Files.newInputStream(modelFile.toPath())) {
            GeckoModelLoader.registerGeo(uid, fileInputStream);
        }
        // 加载贴图
        registerFilePackTexture(rootPath, chairModelItem.getTexture());
        // 加载动画
        List<class_2960> animation = chairModelItem.getAnimation();
        if (animation == null || animation.isEmpty()) {
            return;
        }
        AnimationFile animationData = new AnimationFile();
        for (class_2960 animationPath : animation) {
            if (animationPath.equals(GeckoModelLoader.DEFAULT_CHAIR_ANIMATION)) {
                break;
            }
            File animationFile = rootPath.resolve("assets").resolve(animationPath.method_12836()).resolve(animationPath.method_12832()).toFile();
            if (!animationFile.isFile()) {
                continue;
            }
            try (InputStream fileInputStream = Files.newInputStream(animationFile.toPath())) {
                GeckoModelLoader.mergeAnimationFile(fileInputStream, animationData);
            }
        }
        GeckoModelLoader.registerChairAnimations(uid, animationData);
        CHAIR_MODELS.putInfo(uid.toString(), chairModelItem);
        // 打印日志
        LOGGER.debug(MARKER, "Loaded model: {}", chairModelItem.getModel());
    }


    private static void loadMaidModelPack(ZipFile zipFile, String domain) {
        LOGGER.debug(MARKER, "Touhou little maid mod's model is loading...");
        ZipEntry entry = zipFile.getEntry(String.format("assets/%s/%s", domain, MAID_MODELS.getJsonFileName()));
        if (entry == null) {
            return;
        }
        try (InputStream stream = zipFile.getInputStream(entry)) {
            CustomModelPack<MaidModelInfo> pack = GSON.fromJson(new InputStreamReader(stream, StandardCharsets.UTF_8),
                    new TypeToken<CustomModelPack<MaidModelInfo>>() {
                    }.getType());
            pack.decorate(domain);
            // 加载图标贴图
            if (pack.getIcon() != null) {
                registerZipPackTexture(zipFile.getName(), pack.getIcon());
            }
            for (MaidModelInfo maidModelItem : pack.getModelList()) {
                if (maidModelItem.isGeckoModel()) {
                    loadGeckoMaidModelElement(zipFile, maidModelItem);
                } else {
                    loadMaidModelElement(zipFile, maidModelItem);
                }
            }
            MAID_MODELS.addPack(pack);
        } catch (IOException e) {
            e.printStackTrace();
        } catch (JsonSyntaxException e) {
            LOGGER.warn(MARKER, "Fail to parse model pack in domain {}", domain);
            e.printStackTrace();
        }
        LOGGER.debug(MARKER, "Touhou little maid mod's model is loaded");
    }

    private static void loadMaidModelElement(ZipFile zipFile, MaidModelInfo maidModelItem) {
        // 尝试加载模型
        BedrockModel<class_1308> modelJson = loadMaidModel(zipFile, maidModelItem.getModel());
        // 加载贴图
        registerZipPackTexture(zipFile.getName(), maidModelItem.getTexture());
        // 加载动画
        @Nullable List<Object> animations = CustomJsAnimationManger.getCustomAnimation(zipFile, maidModelItem);
        if (modelJson != null) {
            // 加载彩蛋，彩蛋不允许为空
            if (maidModelItem.getEasterEgg() != null && StringUtils.isNotBlank(maidModelItem.getEasterEgg().getTag())) {
                putMaidEasterEggData(maidModelItem, modelJson, animations);
            } else {
                putMaidModelData(maidModelItem, modelJson, animations);
            }
            // 打印日志
            LOGGER.debug(MARKER, "Loaded model: {}", maidModelItem.getModel());
        }
    }

    private static void loadGeckoMaidModelElement(ZipFile zipFile, MaidModelInfo maidModelItem) throws IOException {
        class_2960 uid = maidModelItem.getModelId();
        // 尝试加载模型
        class_2960 modelLocation = maidModelItem.getModel();
        String path = String.format("assets/%s/%s", modelLocation.method_12836(), modelLocation.method_12832());
        ZipEntry modelZipEntry = zipFile.getEntry(path);
        if (modelZipEntry == null) {
            return;
        }
        try (InputStream zipFileInputStream = zipFile.getInputStream(modelZipEntry)) {
            GeckoModelLoader.registerGeo(uid, zipFileInputStream);
        }
        // 加载贴图
        registerZipPackTexture(zipFile.getName(), maidModelItem.getTexture());
        // 加载动画
        List<class_2960> animation = maidModelItem.getAnimation();
        if (animation == null || animation.isEmpty()) {
            return;
        }
        AnimationFile animationData = new AnimationFile();
        for (class_2960 animationPath : animation) {
            if (animationPath.equals(GeckoModelLoader.DEFAULT_MAID_ANIMATION)) {
                break;
            }
            ZipEntry animationZipEntry = zipFile.getEntry(String.format("assets/%s/%s", animationPath.method_12836(), animationPath.method_12832()));
            if (animationZipEntry == null) {
                continue;
            }
            try (InputStream zipFileInputStream = zipFile.getInputStream(animationZipEntry)) {
                GeckoModelLoader.mergeAnimationFile(zipFileInputStream, animationData);
            }
        }
        GeckoModelLoader.registerMaidAnimations(uid, animationData);
        if (maidModelItem.getEasterEgg() != null && StringUtils.isNotBlank(maidModelItem.getEasterEgg().getTag())) {
            putMaidEasterEggData(maidModelItem, null, null);
        } else {
            MAID_MODELS.putInfo(uid.toString(), maidModelItem);
        }
        // 打印日志
        LOGGER.debug(MARKER, "Loaded model: {}", maidModelItem.getModel());
    }

    private static void loadGeckoChairModelElement(ZipFile zipFile, ChairModelInfo chairModelItem) throws IOException {
        class_2960 uid = chairModelItem.getModelId();
        // 尝试加载模型
        class_2960 modelLocation = chairModelItem.getModel();
        String path = String.format("assets/%s/%s", modelLocation.method_12836(), modelLocation.method_12832());
        ZipEntry modelZipEntry = zipFile.getEntry(path);
        if (modelZipEntry == null) {
            return;
        }
        try (InputStream zipFileInputStream = zipFile.getInputStream(modelZipEntry)) {
            GeckoModelLoader.registerGeo(uid, zipFileInputStream);
        }
        // 加载贴图
        registerZipPackTexture(zipFile.getName(), chairModelItem.getTexture());
        // 加载动画
        List<class_2960> animation = chairModelItem.getAnimation();
        if (animation == null || animation.isEmpty()) {
            return;
        }
        AnimationFile animationData = new AnimationFile();
        for (class_2960 animationPath : animation) {
            if (animationPath.equals(GeckoModelLoader.DEFAULT_CHAIR_ANIMATION)) {
                break;
            }
            ZipEntry animationZipEntry = zipFile.getEntry(String.format("assets/%s/%s", animationPath.method_12836(), animationPath.method_12832()));
            if (animationZipEntry == null) {
                continue;
            }
            try (InputStream zipFileInputStream = zipFile.getInputStream(animationZipEntry)) {
                GeckoModelLoader.mergeAnimationFile(zipFileInputStream, animationData);
            }
        }
        GeckoModelLoader.registerChairAnimations(uid, animationData);
        CHAIR_MODELS.putInfo(uid.toString(), chairModelItem);
        // 打印日志
        LOGGER.debug(MARKER, "Loaded model: {}", chairModelItem.getModel());
    }

    @SuppressWarnings("all")
    private static void putMaidEasterEggData(MaidModelInfo maidModelItem, @Nullable BedrockModel<class_1308> modelJson, @Nullable List<Object> animations) {
        MaidModelInfo.EasterEgg easterEgg = maidModelItem.getEasterEgg();
        MaidModels.ModelData data = new MaidModels.ModelData(modelJson, maidModelItem, animations);
        if (easterEgg.isEncrypt()) {
            MAID_MODELS.putEasterEggEncryptTagModel(easterEgg.getTag(), data);
        } else {
            MAID_MODELS.putEasterEggNormalTagModel(easterEgg.getTag(), data);
        }
    }

    private static void putMaidModelData(MaidModelInfo maidModelItem, BedrockModel<class_1308> modelJson, List<Object> animations) {
        String id = maidModelItem.getModelId().toString();
        // 如果加载的模型不为空
        MAID_MODELS.putModel(id, modelJson);
        MAID_MODELS.putInfo(id, maidModelItem);
        if (animations != null && !animations.isEmpty()) {
            MAID_MODELS.putAnimation(id, animations);
        }
    }

    private static void loadChairModelPack(Path rootPath, String domain) {
        LOGGER.debug(MARKER, "Touhou little maid mod's model is loading...");
        File file = rootPath.resolve("assets").resolve(domain).resolve(CHAIR_MODELS.getJsonFileName()).toFile();
        if (!file.isFile()) {
            return;
        }
        try (InputStream stream = Files.newInputStream(file.toPath())) {
            // 将其转换为 pojo 对象
            // 这个 pojo 是二次修饰的过的对象，所以一部分数据异常已经进行了处理或者抛出
            CustomModelPack<ChairModelInfo> pack = GSON.fromJson(new InputStreamReader(stream, StandardCharsets.UTF_8),
                    new TypeToken<CustomModelPack<ChairModelInfo>>() {
                    }.getType());
            pack.decorate(domain);
            // 加载图标贴图
            if (pack.getIcon() != null) {
                registerFilePackTexture(rootPath, pack.getIcon());
            }
            for (ChairModelInfo chairModelItem : pack.getModelList()) {
                if (chairModelItem.isGeckoModel()) {
                    loadGeckoChairModelElement(rootPath, chairModelItem);
                } else {
                    loadChairModelElement(rootPath, chairModelItem);
                }
            }
            CHAIR_MODELS.addPack(pack);
        } catch (IOException ignore) {
            // 忽略错误，因为资源域很多
        } catch (JsonSyntaxException e) {
            LOGGER.warn(MARKER, "Fail to parse model pack in domain {}", domain);
            e.printStackTrace();
        }
        LOGGER.debug(MARKER, "Touhou little maid mod's model is loaded");
    }

    private static void loadChairModelElement(Path rootPath, ChairModelInfo chairModelItem) {
        // 尝试加载模型
        BedrockModel<EntityChair> modelJson = loadChairModel(rootPath, chairModelItem.getModel());
        // 加载贴图
        registerFilePackTexture(rootPath, chairModelItem.getTexture());
        // 加载动画
        @Nullable List<Object> animations = CustomJsAnimationManger.getCustomAnimation(rootPath, chairModelItem);
        if (modelJson != null) {
            String id = chairModelItem.getModelId().toString();
            // 如果加载的模型不为空
            CHAIR_MODELS.putModel(id, modelJson);
            CHAIR_MODELS.putInfo(id, chairModelItem);
            if (animations != null && !animations.isEmpty()) {
                CHAIR_MODELS.putAnimation(id, animations);
            }
            // 打印日志
            LOGGER.debug(MARKER, "Loaded model: {}", chairModelItem.getModel());
        }
    }

    private static void loadChairModelPack(ZipFile zipFile, String domain) {
        LOGGER.debug(MARKER, "Touhou little maid mod's model is loading...");
        ZipEntry entry = zipFile.getEntry(String.format("assets/%s/%s", domain, CHAIR_MODELS.getJsonFileName()));
        if (entry == null) {
            return;
        }
        try (InputStream stream = zipFile.getInputStream(entry)) {
            // 将其转换为 pojo 对象
            // 这个 pojo 是二次修饰的过的对象，所以一部分数据异常已经进行了处理或者抛出
            CustomModelPack<ChairModelInfo> pack = GSON.fromJson(new InputStreamReader(stream, StandardCharsets.UTF_8),
                    new TypeToken<CustomModelPack<ChairModelInfo>>() {
                    }.getType());
            pack.decorate(domain);
            // 加载图标贴图
            if (pack.getIcon() != null) {
                registerZipPackTexture(zipFile.getName(), pack.getIcon());
            }
            for (ChairModelInfo chairModelItem : pack.getModelList()) {
                if (chairModelItem.isGeckoModel()) {
                    loadGeckoChairModelElement(zipFile, chairModelItem);
                } else {
                    loadChairModelElement(zipFile, chairModelItem);
                }
            }
            CHAIR_MODELS.addPack(pack);
        } catch (IOException ignore) {
            // 忽略错误，因为资源域很多
        } catch (JsonSyntaxException e) {
            LOGGER.warn(MARKER, "Fail to parse model pack in domain {}", domain);
            e.printStackTrace();
        }
        LOGGER.debug(MARKER, "Touhou little maid mod's model is loaded");
    }

    private static void loadChairModelElement(ZipFile zipFile, ChairModelInfo chairModelItem) {
        // 尝试加载模型
        BedrockModel<EntityChair> modelJson = loadChairModel(zipFile, chairModelItem.getModel());
        // 加载贴图
        registerZipPackTexture(zipFile.getName(), chairModelItem.getTexture());
        // 加载动画
        @Nullable List<Object> animations = CustomJsAnimationManger.getCustomAnimation(zipFile, chairModelItem);
        if (modelJson != null) {
            String id = chairModelItem.getModelId().toString();
            // 如果加载的模型不为空
            CHAIR_MODELS.putModel(id, modelJson);
            CHAIR_MODELS.putInfo(id, chairModelItem);
            if (animations != null && !animations.isEmpty()) {
                CHAIR_MODELS.putAnimation(id, animations);
            }
            // 打印日志
            LOGGER.debug(MARKER, "Loaded model: {}", chairModelItem.getModel());
        }
    }

    @Nullable
    public static BedrockModel<class_1308> loadMaidModel(Path rootPath, class_2960 modelLocation) {
        File file = rootPath.resolve("assets").resolve(modelLocation.method_12836()).resolve(modelLocation.method_12832()).toFile();
        if (!file.isFile()) {
            return null;
        }
        try (InputStream stream = Files.newInputStream(file.toPath())) {
            BedrockModelPOJO pojo = GSON.fromJson(new InputStreamReader(stream, StandardCharsets.UTF_8), BedrockModelPOJO.class);
            // 先判断是不是 1.10.0 版本基岩版模型文件
            if (BedrockVersion.isLegacyVersion(pojo)) {
                // 如果 model 字段不为空
                if (pojo.getGeometryModelLegacy() != null) {
                    return new BedrockModel<>(pojo, BedrockVersion.LEGACY);
                } else {
                    // 否则日志给出提示
                    LOGGER.warn(MARKER, "{} model file don't have model field", modelLocation);
                    return null;
                }
            }

            // 判定是不是 1.12.0 版本基岩版模型文件
            if (BedrockVersion.isNewVersion(pojo)) {
                // 如果 model 字段不为空
                if (pojo.getGeometryModelNew() != null) {
                    return new BedrockModel<>(pojo, BedrockVersion.NEW);
                } else {
                    // 否则日志给出提示
                    LOGGER.warn(MARKER, "{} model file don't have model field", modelLocation);
                    return null;
                }
            }

            LOGGER.warn(MARKER, "{} model version is not 1.10.0 or 1.12.0", modelLocation);
        } catch (IOException ioe) {
            // 可能用来判定错误，打印下
            LOGGER.warn(MARKER, "Failed to load model: {}", modelLocation);
            ioe.printStackTrace();
        }
        // 如果前面出了错，返回 Null
        return null;
    }

    @Nullable
    public static BedrockModel<class_1308> loadMaidModel(ZipFile zipFile, class_2960 modelLocation) {
        String path = String.format("assets/%s/%s", modelLocation.method_12836(), modelLocation.method_12832());
        ZipEntry entry = zipFile.getEntry(path);
        if (entry == null) {
            return null;
        }
        try (InputStream stream = zipFile.getInputStream(entry)) {
            BedrockModelPOJO pojo = GSON.fromJson(new InputStreamReader(stream, StandardCharsets.UTF_8), BedrockModelPOJO.class);
            // 先判断是不是 1.10.0 版本基岩版模型文件
            if (BedrockVersion.isLegacyVersion(pojo)) {
                // 如果 model 字段不为空
                if (pojo.getGeometryModelLegacy() != null) {
                    return new BedrockModel<>(pojo, BedrockVersion.LEGACY);
                } else {
                    // 否则日志给出提示
                    LOGGER.warn(MARKER, "{} model file don't have model field", modelLocation);
                    return null;
                }
            }

            // 判定是不是 1.12.0 版本基岩版模型文件
            if (BedrockVersion.isNewVersion(pojo)) {
                // 如果 model 字段不为空
                if (pojo.getGeometryModelNew() != null) {
                    return new BedrockModel<>(pojo, BedrockVersion.NEW);
                } else {
                    // 否则日志给出提示
                    LOGGER.warn(MARKER, "{} model file don't have model field", modelLocation);
                    return null;
                }
            }

            LOGGER.warn(MARKER, "{} model version is not 1.10.0 or 1.12.0", modelLocation);
        } catch (IOException ioe) {
            // 可能用来判定错误，打印下
            LOGGER.warn(MARKER, "Failed to load model: {}", modelLocation);
            ioe.printStackTrace();
        }
        // 如果前面出了错，返回 Null
        return null;
    }

    @Nullable
    public static BedrockModel<EntityChair> loadChairModel(Path rootPath, class_2960 modelLocation) {
        File file = rootPath.resolve("assets").resolve(modelLocation.method_12836()).resolve(modelLocation.method_12832()).toFile();
        if (!file.isFile()) {
            return null;
        }
        try (InputStream stream = Files.newInputStream(file.toPath())) {
            BedrockModelPOJO pojo = GSON.fromJson(new InputStreamReader(stream, StandardCharsets.UTF_8), BedrockModelPOJO.class);
            // 先判断是不是 1.10.0 版本基岩版模型文件
            if (BedrockVersion.isLegacyVersion(pojo)) {
                // 如果 model 字段不为空
                if (pojo.getGeometryModelLegacy() != null) {
                    return new BedrockModel<>(pojo, BedrockVersion.LEGACY);
                } else {
                    // 否则日志给出提示
                    LOGGER.warn(MARKER, "{} model file don't have model field", modelLocation);
                    return null;
                }
            }

            // 判定是不是 1.12.0 版本基岩版模型文件
            if (BedrockVersion.isNewVersion(pojo)) {
                // 如果 model 字段不为空
                if (pojo.getGeometryModelNew() != null) {
                    return new BedrockModel<>(pojo, BedrockVersion.NEW);
                } else {
                    // 否则日志给出提示
                    LOGGER.warn(MARKER, "{} model file don't have model field", modelLocation);
                    return null;
                }
            }

            LOGGER.warn(MARKER, "{} model version is not 1.10.0 or 1.12.0", modelLocation);
        } catch (IOException ioe) {
            // 可能用来判定错误，打印下
            LOGGER.warn(MARKER, "Failed to load model: {}", modelLocation);
            ioe.printStackTrace();
        }
        // 如果前面出了错，返回 Null
        return null;
    }

    @Nullable
    public static BedrockModel<EntityChair> loadChairModel(ZipFile zipFile, class_2960 modelLocation) {
        String path = String.format("assets/%s/%s", modelLocation.method_12836(), modelLocation.method_12832());
        ZipEntry entry = zipFile.getEntry(path);
        if (entry == null) {
            return null;
        }
        try (InputStream stream = zipFile.getInputStream(entry)) {
            BedrockModelPOJO pojo = GSON.fromJson(new InputStreamReader(stream, StandardCharsets.UTF_8), BedrockModelPOJO.class);
            // 先判断是不是 1.10.0 版本基岩版模型文件
            if (BedrockVersion.isLegacyVersion(pojo)) {
                // 如果 model 字段不为空
                if (pojo.getGeometryModelLegacy() != null) {
                    return new BedrockModel<>(pojo, BedrockVersion.LEGACY);
                } else {
                    // 否则日志给出提示
                    LOGGER.warn(MARKER, "{} model file don't have model field", modelLocation);
                    return null;
                }
            }

            // 判定是不是 1.12.0 版本基岩版模型文件
            if (BedrockVersion.isNewVersion(pojo)) {
                // 如果 model 字段不为空
                if (pojo.getGeometryModelNew() != null) {
                    return new BedrockModel<>(pojo, BedrockVersion.NEW);
                } else {
                    // 否则日志给出提示
                    LOGGER.warn(MARKER, "{} model file don't have model field", modelLocation);
                    return null;
                }
            }

            LOGGER.warn(MARKER, "{} model version is not 1.10.0 or 1.12.0", modelLocation);
        } catch (IOException ioe) {
            // 可能用来判定错误，打印下
            LOGGER.warn(MARKER, "Failed to load model: {}", modelLocation);
            ioe.printStackTrace();
        }
        // 如果前面出了错，返回 Null
        return null;
    }

    public static void registerFilePackTexture(Path rootPath, class_2960 texturePath) {
        if (!TMP_REGISTER_TEXTURE.contains(texturePath)) {
            FilePackTexture filePackTexture = new FilePackTexture(rootPath, texturePath);
            if (filePackTexture.isExist()) {
                class_310.method_1551().method_1531().method_4616(texturePath, filePackTexture);
                TMP_REGISTER_TEXTURE.add(texturePath);
            }
        }
    }

    public static void registerZipPackTexture(String zipFilePath, class_2960 texturePath) {
        if (!TMP_REGISTER_TEXTURE.contains(texturePath)) {
            ZipPackTexture zipPackTexture = new ZipPackTexture(zipFilePath, texturePath);
            if (zipPackTexture.isExist()) {
                class_310.method_1551().method_1531().method_4616(texturePath, zipPackTexture);
                TMP_REGISTER_TEXTURE.add(texturePath);
            }
        }
    }
}