package com.github.tartaricacid.touhoulittlemaid.world.backups;

import cn.sh1rocu.touhoulittlemaid.mixin.accessor.DimensionDataStorageAccessor;
import com.github.tartaricacid.touhoulittlemaid.config.ServerConfig;
import com.github.tartaricacid.touhoulittlemaid.entity.passive.EntityMaid;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_2487;
import net.minecraft.class_2507;
import net.minecraft.class_2512;
import net.minecraft.class_2561;
import net.minecraft.class_3218;
import net.minecraft.class_3222;
import net.minecraft.server.MinecraftServer;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import static com.github.tartaricacid.touhoulittlemaid.TouhouLittleMaid.LOGGER;

/**
 * 女仆数据备份管理器
 * <p>
 * 存储结构：
 * <pre>
 * maid_backups/
 * ├── owner_uuid/
 * │    ├── index.dat                    # 索引文件，存储所有女仆的基本信息
 * │    ├── maid_uuid/
 * │    │    ├── 2025-09-08-20-25-30.dat # 女仆数据备份文件
 * │    │    ├── 2025-09-08-20-30-30.dat
 * │    │    └── ...
 * │    ├── maid_uuid/
 * │    └── ...
 * ├── owner_uuid/
 * └── ...
 * </pre>
 *
 * @author TartaricAcid
 */
public final class MaidBackupsManager {
    private static final String BACKUPS_FOLDER_NAME = "maid_backups";
    private static final String INDEX_FILE_NAME = "index.dat";
    private static final String BACKUP_FILE_EXTENSION = ".dat";

    private static final ExecutorService EXECUTOR_SERVICE = Executors.newSingleThreadExecutor(r -> {
        Thread thread = new Thread(r, "Maid-Backups-Thread");
        thread.setDaemon(true);
        thread.setUncaughtExceptionHandler((t, e) ->
                LOGGER.error("Uncaught exception in maid backup thread: {}", t.getName(), e));
        return thread;
    });

    private static final DateTimeFormatter BACKUP_TIME_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd-HH-mm-ss");

    // 私有构造函数，防止实例化
    private MaidBackupsManager() {
    }

    /**
     * 保存女仆数据备份
     *
     * @param server 服务器实例
     * @param maid   要备份的女仆实体
     */
    public static void save(@NotNull MinecraftServer server, @NotNull EntityMaid maid) {
        UUID ownerId = maid.method_6139();
        if (ownerId == null) {
            return;
        }

        class_3218 overWorld = server.method_3847(class_1937.field_25179);
        if (overWorld == null) {
            LOGGER.error("Cannot access overworld for maid backup: {}", maid.method_5845());
            return;
        }

        try {
            BackupData backupData = createBackupData(maid, overWorld, ownerId);
            if (backupData != null) {
                executeBackupAsync(backupData);
            }
        } catch (Exception e) {
            LOGGER.error("Failed to initiate backup for maid: {}", maid.method_5845(), e);
        }
    }

    /**
     * 获取玩家的女仆索引数据
     */
    public static class_2487 getMaidIndex(class_3222 player) {
        UUID ownerId = player.method_5667();
        MinecraftServer server = player.method_5682();
        if (server == null) {
            return new class_2487();
        }
        class_3218 overWorld = server.method_3847(class_1937.field_25179);
        if (overWorld == null) {
            LOGGER.error("Cannot access overworld for maid index retrieval: {}", ownerId);
            return new class_2487();
        }
        Path ownerFolder = ((DimensionDataStorageAccessor) overWorld.method_17983()).tlm$getDataFolder().toPath()
                .resolve(BACKUPS_FOLDER_NAME)
                .resolve(ownerId.toString());
        File indexFile = ownerFolder.resolve(INDEX_FILE_NAME).toFile();
        return loadExistingIndexData(indexFile);
    }

    /**
     * 创建备份数据
     */
    @Nullable
    private static BackupData createBackupData(@NotNull EntityMaid maid, @NotNull class_3218 overWorld, @NotNull UUID ownerId) {
        try {
            Path saveFolder = buildBackupFolderPath(maid, overWorld, ownerId);
            String saveFileName = generateBackupFileName();

            class_2487 entityData = new class_2487();
            boolean saveResult = maid.method_5786(entityData);

            if (!saveResult) {
                LOGGER.warn("Failed to serialize maid data for backup: {}", maid.method_5845());
                return null;
            }

            class_2487 indexData = createIndexData(maid);
            return new BackupData(saveFolder, saveFileName, entityData, indexData, maid.method_5845());

        } catch (Exception e) {
            LOGGER.error("Error creating backup data for maid: {}", maid.method_5845(), e);
            return null;
        }
    }

    /**
     * 创建索引数据
     */
    @NotNull
    private static class_2487 createIndexData(@NotNull EntityMaid maid) {
        class_2487 indexData = new class_2487();
        indexData.method_10582("Name", class_2561.class_2562.method_10867(maid.method_5477()));
        indexData.method_10582("Dimension", maid.field_6002.method_27983().method_29177().toString());
        indexData.method_10566("Pos", class_2512.method_10692(maid.method_24515()));
        indexData.method_10544("Timestamp", System.currentTimeMillis());
        return indexData;
    }

    public static Map<UUID, IndexData> getMaidIndexMap(class_3222 player) {
        class_2487 indexTag = getMaidIndex(player);
        Map<UUID, IndexData> map = Maps.newHashMap();
        for (String s : indexTag.method_10541()) {
            class_2487 maidTag = indexTag.method_10562(s);
            UUID maidUuid = UUID.fromString(s);
            class_2561 name = class_2561.class_2562.method_10877(maidTag.method_10558("Name"));
            class_2338 pos = class_2512.method_10691(maidTag.method_10562("Pos"));
            String dimension = maidTag.method_10558("Dimension");
            long timestamp = maidTag.method_10537("Timestamp");
            map.put(maidUuid, new IndexData(name, pos, dimension, timestamp));
        }
        return map;
    }

    public static List<String> getMaidBackupFiles(class_3222 player, UUID maidUuid) {
        List<String> backupFiles = Lists.newArrayList();
        Path folderPath = ((DimensionDataStorageAccessor) player.method_51469().method_17983()).tlm$getDataFolder().toPath()
                .resolve(BACKUPS_FOLDER_NAME)
                .resolve(player.method_5667().toString())
                .resolve(maidUuid.toString());
        if (!Files.isDirectory(folderPath)) {
            return backupFiles;
        }
        File[] files = folderPath.toFile().listFiles((dir, name) -> name.endsWith(BACKUP_FILE_EXTENSION));
        if (files == null) {
            return backupFiles;
        }
        for (File file : files) {
            backupFiles.add(file.getName());
        }
        backupFiles.sort(Comparator.reverseOrder());
        return backupFiles;
    }

    public static class_2487 getMaidBackFile(class_3222 player, UUID maidUuid, String fileName) {
        Path filePath = ((DimensionDataStorageAccessor) player.method_51469().method_17983()).tlm$getDataFolder().toPath()
                .resolve(BACKUPS_FOLDER_NAME)
                .resolve(player.method_5667().toString())
                .resolve(maidUuid.toString())
                .resolve(fileName);
        File file = filePath.toFile();
        if (!file.exists() || !file.isFile()) {
            return new class_2487();
        }
        try {
            return class_2507.method_30613(file);
        } catch (IOException e) {
            LOGGER.error("Failed to read maid backup file: {}", filePath, e);
            return new class_2487();
        }
    }


    /**
     * 异步执行备份
     */
    private static void executeBackupAsync(@NotNull BackupData backupData) {
        CompletableFuture
                .runAsync(() -> performBackup(backupData), EXECUTOR_SERVICE)
                .exceptionally(error -> {
                    LOGGER.error("Async backup failed for maid: {}", backupData.maidUuid, error);
                    return null;
                });
    }

    /**
     * 执行实际的备份操作
     */
    private static void performBackup(@NotNull BackupData backupData) {
        try {
            // 1. 确保目录存在
            if (!ensureDirectoryExists(backupData.saveFolder)) {
                return;
            }

            // 2. 更新索引文件
            if (!updateIndexFile(backupData)) {
                return;
            }

            // 3. 保存实体数据
            saveEntityData(backupData);

        } catch (Exception e) {
            LOGGER.error("Backup operation failed for maid: {}", backupData.maidUuid, e);
        }
    }

    /**
     * 确保目录存在
     */
    private static boolean ensureDirectoryExists(@NotNull Path directory) {
        if (Files.isDirectory(directory)) {
            return true;
        }

        try {
            Files.createDirectories(directory);
            return true;
        } catch (IOException e) {
            LOGGER.error("Failed to create backup directory: {}", directory, e);
            return false;
        }
    }

    /**
     * 更新索引文件
     */
    private static boolean updateIndexFile(@NotNull BackupData backupData) {
        File indexFile = backupData.saveFolder.getParent().resolve(INDEX_FILE_NAME).toFile();

        try {
            class_2487 allIndexData = loadExistingIndexData(indexFile);
            allIndexData.method_10566(backupData.saveFolder.getFileName().toString(), backupData.indexData);

            class_2507.method_30614(allIndexData, indexFile);
            return true;

        } catch (IOException e) {
            LOGGER.error("Failed to update index file: {}", indexFile, e);
            return false;
        }
    }

    /**
     * 加载现有的索引数据
     */
    @NotNull
    private static class_2487 loadExistingIndexData(@NotNull File indexFile) {
        if (!indexFile.exists()) {
            return new class_2487();
        }

        try {
            return class_2507.method_30613(indexFile);
        } catch (IOException e) {
            LOGGER.warn("Failed to read existing index file, creating new one: {}", indexFile, e);
            return new class_2487();
        }
    }

    /**
     * 保存实体数据
     */
    private static void saveEntityData(@NotNull BackupData backupData) {
        File backupFile = backupData.saveFolder.resolve(backupData.saveFileName).toFile();

        try {
            class_2512.method_48310(backupData.entityData);
            class_2507.method_30614(backupData.entityData, backupFile);

            // 删除旧备份
            removeOldBackups(backupData.saveFolder, ServerConfig.MAID_BACKUP_MAX_COUNT.get());

        } catch (IOException e) {
            LOGGER.error("Failed to save entity data to: {}", backupFile, e);
        }
    }

    private static void removeOldBackups(@NotNull Path backupFolder, int maxBackups) {
        try {
            File[] backupFiles = backupFolder.toFile().listFiles((dir, name) -> name.endsWith(BACKUP_FILE_EXTENSION));
            if (backupFiles == null || backupFiles.length <= maxBackups) {
                return;
            }

            // 按文件名排序，文件名包含时间戳
            Arrays.sort(backupFiles, Comparator.comparing(File::getName));

            // 删除多余的备份文件
            for (int i = 0; i < backupFiles.length - maxBackups; i++) {
                if (!backupFiles[i].delete()) {
                    LOGGER.warn("Failed to delete old backup file: {}", backupFiles[i].getAbsolutePath());
                }
            }
        } catch (Exception e) {
            LOGGER.error("Error while removing old backups in folder: {}", backupFolder, e);
        }
    }

    /**
     * 构建备份文件夹路径
     */
    @NotNull
    private static Path buildBackupFolderPath(@NotNull EntityMaid maid, @NotNull class_3218 level, @NotNull UUID ownerId) {
        return ((DimensionDataStorageAccessor) level.method_17983()).tlm$getDataFolder().toPath()
                .resolve(BACKUPS_FOLDER_NAME)
                .resolve(ownerId.toString())
                .resolve(maid.method_5845());
    }

    /**
     * 生成备份文件名
     */
    @NotNull
    public static String generateBackupFileName() {
        return LocalDateTime.now().format(BACKUP_TIME_FORMATTER) + BACKUP_FILE_EXTENSION;
    }

    public record IndexData(
            class_2561 name,
            class_2338 pos,
            String dimension,
            long timestamp) {
    }

    /**
     * 备份数据容器类
     */
    private record BackupData(
            Path saveFolder,
            String saveFileName,
            class_2487 entityData,
            class_2487 indexData,
            String maidUuid) {
    }
}