package com.github.tartaricacid.touhoulittlemaid.ai.manager.setting.papi;

import cn.sh1rocu.touhoulittlemaid.util.itemhandler.RangedWrapper;
import com.github.tartaricacid.touhoulittlemaid.entity.passive.EntityMaid;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.core.lookup.StrSubstitutor;

import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.function.BiFunction;
import net.minecraft.class_1304;
import net.minecraft.class_1309;
import net.minecraft.class_1799;
import net.minecraft.class_1937;
import net.minecraft.class_1959;
import net.minecraft.class_2960;
import net.minecraft.class_5321;
import net.minecraft.class_7924;

import static com.github.tartaricacid.touhoulittlemaid.ai.manager.setting.papi.StringConstant.*;

/**
 * 因为现在的大语言模型基本都有多语言支持，故直接用中文写设定文件
 * <p>
 * 虽然看着很不舒服，但是为了方便使用，还是这样吧
 */
public class PapiReplacer {
    private static final Map<String, BiFunction<EntityMaid, String, String>> PAPI_MAP = Maps.newLinkedHashMap();

    static {
        registerContext("game_time", PapiReplacer::getTime);
        registerContext("weather", PapiReplacer::getWeather);
        registerContext("dimension", PapiReplacer::getDimension);
        registerContext("mainhand_item", (maid, lang) -> getSlotItemName(class_1304.field_6173, maid));
        registerContext("offhand_item", (maid, lang) -> getSlotItemName(class_1304.field_6171, maid));
        registerContext("inventory_items", PapiReplacer::getInventoryItems);
        registerContext("chat_language", (maid, lang) -> language(lang));
        registerContext("tts_language", ((maid, lang) -> language(maid.getAiChatManager().getTTSLanguage())));
        registerContext("healthy", PapiReplacer::getHealthyInfo);
        registerContext("owner_healthy", PapiReplacer::getOwnerHealthyInfo);
        registerContext("armor_items", PapiReplacer::getArmorItems);
        registerContext("effects", PapiReplacer::getEffects);
        registerContext("biome", PapiReplacer::getBiome);
        registerContext("owner_name", PapiReplacer::getOwnerName);
    }

    public static String replace(String input, EntityMaid maid, String language) {
        Map<String, String> valueMap = Maps.newHashMap();
        valueMap.put("main_setting", input);
        for (String key : PAPI_MAP.keySet()) {
            valueMap.put(key, PAPI_MAP.get(key).apply(maid, language));
        }
        String base = new StrSubstitutor(valueMap).replace(FULL_SETTING);
        if (language.equals(maid.getAiChatManager().getTTSLanguage())) {
            base += new StrSubstitutor(valueMap).replace(OUTPUT_FORMAT_REQUIREMENTS_SAME_LANGUAGES);
        } else {
            base += new StrSubstitutor(valueMap).replace(OUTPUT_FORMAT_REQUIREMENTS_DIFFERENT_LANGUAGES);
        }
        return base;
    }

    public static void registerContext(String key, BiFunction<EntityMaid, String, String> function) {
        PAPI_MAP.put(key, function);
    }

    /**
     * 不能调用 LanguageManager，那个是客户端方法
     */
    private static String language(String languageTag) {
        // 将语言代码转换为 Locale 所需的格式，例如 zh_cn -> zh-CN
        String[] parts = languageTag.split("_");
        if (parts.length == 2) {
            languageTag = parts[0] + "-" + parts[1].toUpperCase(Locale.ENGLISH);
        }
        Locale locale = Locale.forLanguageTag(languageTag);
        return LANGUAGE_FORMAT.formatted(locale.getDisplayLanguage(), locale.getDisplayCountry());
    }

    private static String getBiome(EntityMaid maid, String chatLanguage) {
        class_1959 biome = maid.field_6002.method_23753(maid.method_24515()).comp_349();
        class_2960 key = maid.field_6002.method_30349().method_30530(class_7924.field_41236).method_10221(biome);
        return key == null ? UNKNOWN_BIOME : key.toString();
    }

    private static String getEffects(EntityMaid maid, String chatLanguage) {
        List<String> names = new ArrayList<>();
        maid.method_6026().forEach(i -> {
            names.add(i.toString());
        });
        if (names.isEmpty()) {
            return NONE;
        }
        return StringUtils.join(names, LIST_SEPARATORS);
    }

    private static String getArmorItems(EntityMaid maid, String chatLanguage) {
        List<String> names = new ArrayList<>();
        maid.method_5661().forEach(stack -> {
            if (!stack.method_7960()) {
                String itemName = stack.method_7954().getString();
                int count = stack.method_7947();
                names.add(ITEM_AND_COUNT_FORMAT.formatted(itemName, count));
            }
        });
        if (names.isEmpty()) {
            return EMPTY;
        }
        return StringUtils.join(names, LIST_SEPARATORS);
    }

    private static String getHealthyInfo(EntityMaid maid, String chatLanguage) {
        float maxHealth = maid.method_6063();
        float health = maid.method_6032();
        return HEALTHY_FORMAT.formatted(health, maxHealth);
    }

    private static String getOwnerHealthyInfo(EntityMaid maid, String chatLanguage) {
        class_1309 owner = maid.method_35057();
        if (owner != null) {
            float maxHealth = owner.method_6063();
            float health = owner.method_6032();
            return HEALTHY_FORMAT.formatted(health, maxHealth);
        }
        return StringUtils.EMPTY;
    }

    private static String getOwnerName(EntityMaid maid, String chatLanguage) {
        String ownerName = maid.getAiChatManager().ownerName;
        if (StringUtils.isBlank(ownerName)) {
            return DEFAULT_OWNER_NAME;
        }
        return ownerName;
    }

    private static String getWeather(EntityMaid maid, String chatLanguage) {
        class_1937 level = maid.field_6002;
        if (level.method_8546()) {
            return THUNDERING;
        }
        if (level.method_8419()) {
            return RAINING;
        }
        return SUNNY;
    }

    private static String getTime(EntityMaid maid, String chatLanguage) {
        long time = maid.field_6002.method_8532();
        long hours = (time / 1000 + 6) % 24;
        long minutes = (time % 1000) / (50 / 3);
        return TIME_FORMAT.formatted(hours, minutes);
    }

    private static String getDimension(EntityMaid maid, String chatLanguage) {
        class_5321<class_1937> dimension = maid.field_6002.method_27983();
        if (dimension == class_1937.field_25179) {
            return OVERWORLD;
        }
        if (dimension == class_1937.field_25180) {
            return NETHER;
        }
        if (dimension == class_1937.field_25181) {
            return END;
        }
        return dimension.method_29177().toString();
    }

    private static String getSlotItemName(class_1304 slot, EntityMaid maid) {
        class_1799 stack = maid.method_6118(slot);
        if (stack.method_7960()) {
            return EMPTY;
        }
        String itemName = stack.method_7954().getString();
        int count = stack.method_7947();
        return ITEM_AND_COUNT_FORMAT.formatted(itemName, count);
    }

    private static String getInventoryItems(EntityMaid maid, String chatLanguage) {
        List<String> names = Lists.newArrayList();
        RangedWrapper backpack = maid.getAvailableBackpackInv();
        for (int i = 0; i < backpack.getSlots(); i++) {
            class_1799 stack = backpack.getStackInSlot(i);
            if (!stack.method_7960()) {
                String itemName = stack.method_7954().getString();
                int count = stack.method_7947();
                names.add(ITEM_AND_COUNT_FORMAT.formatted(itemName, count));
            }
        }
        if (names.isEmpty()) {
            return EMPTY;
        }
        return StringUtils.join(names, LIST_SEPARATORS);
    }
}