package com.jowhjy.hidecoords;

import org.jetbrains.annotations.NotNull;

import java.util.*;
import net.minecraft.class_2338;
import net.minecraft.class_2596;
import net.minecraft.class_2784;
import net.minecraft.class_3218;
import net.minecraft.class_3222;
import net.minecraft.class_5889;
import net.minecraft.class_5895;
import net.minecraft.class_5896;
import net.minecraft.class_5897;
import net.minecraft.class_9094;

/**
 * Adapted from jt_prince
 * World border packets require special handling, since applying a plain offset would run into two problems:
 * <ul>
 *     <li>The player can intercept the packet and use it to derive their offset.</li>
 *     <li>World border coordinates apply world scaling for some reason (the only packets that seem to...)</li>
 * </ul>
 * <a href="https://github.com/joshuaprince/CoordinateOffset/wiki/Implications-and-Limitations#world-border">Wiki</a>
 */
public class WorldBorderObfuscator {
    private static final double BASELINE_SIZE = 60_000_000;

    private static EnumSet<Wall> visibleBorders(class_2338 location, class_3218 world) {
        double viewDistanceBlocks = world.method_8503().method_3760().method_14568() * 16;

        class_2784 realBorder = world.method_8621();
        double xMax = realBorder.method_11964() + realBorder.method_11965() / 2;
        double xMin = realBorder.method_11964() - realBorder.method_11965() / 2;
        double zMax = realBorder.method_11980() + realBorder.method_11965() / 2;
        double zMin = realBorder.method_11980() - realBorder.method_11965() / 2;

        EnumSet<Wall> seen = EnumSet.noneOf(Wall.class);
        if (xMax - location.method_10263() < viewDistanceBlocks) {
            seen.add(Wall.X_POSITIVE);
        }
        if (location.method_10263() - xMin < viewDistanceBlocks) {
            seen.add(Wall.X_NEGATIVE);
        }
        if (zMax - location.method_10260() < viewDistanceBlocks) {
            seen.add(Wall.Z_POSITIVE);
        }
        if (location.method_10260() - zMin < viewDistanceBlocks) {
            seen.add(Wall.Z_NEGATIVE);
        }

        return seen;
    }

    public static class_2596<?> translate(@NotNull class_2596<?> packet, Offset offset, class_3222 player) {

        class_3218 world = player.method_51469();

        /*
         * For reasons I cannot fathom, the Minecraft protocol applies the world's coordinate scaling to the world
         * border center location. (e.g. if I wanted to center a border at Nether coordinates (100,100), I would need to
         * send a packet containing (800, 800) as the center.)
         *
         * This could cause problems if the server is running a custom world with a different coordinateScale (which is
         * only accessible through NMS as DimensionType::coordinateScale). For now, just checking environment should be
         * enough.
         */
        double scaleFactor = world.method_8597().comp_646();

        //dummy world border used for packet creation
        class_2784 dummyWorldBorder = new class_2784();

        EnumSet<Wall> seenWalls = visibleBorders(player.method_24515(), world);
        if ((seenWalls.contains(Wall.X_POSITIVE) && seenWalls.contains(Wall.X_NEGATIVE)) ||
                (seenWalls.contains(Wall.Z_POSITIVE) && seenWalls.contains(Wall.Z_NEGATIVE))) {
            // If the player can see opposing walls, or obfuscation is disabled, we should just send the complete
            // offsetted border. No diameter change.
            if (packet.method_65080().equals(class_9094.field_48021)) {
                class_5889 typedPacket = (class_5889)(packet);
                long l = typedPacket.method_34128();
                if (l > 0L) {
                    dummyWorldBorder.method_11957(typedPacket.method_34127(), typedPacket.method_34126(), l);
                } else {
                    dummyWorldBorder.method_11969(typedPacket.method_34126());
                }
                dummyWorldBorder.method_11973(typedPacket.method_34129());
                dummyWorldBorder.method_11967(typedPacket.method_34131());
                dummyWorldBorder.method_11975(typedPacket.method_34130());
                dummyWorldBorder.method_11978((typedPacket.method_34124() + offset.getX()) * scaleFactor, (typedPacket.method_34125() + offset.getZ()) * scaleFactor);
                return new class_5889(dummyWorldBorder);
            } else if (packet.method_65080().equals(class_9094.field_48082)) {
                class_5895 typedPacket = (class_5895)(packet);
                dummyWorldBorder.method_11978((typedPacket.method_34158() + offset.getX()) * scaleFactor, (typedPacket.method_34157() + offset.getZ()) * scaleFactor);
                return new class_5895(dummyWorldBorder);
            }
            return packet;
        }

        //real border
        final class_2784 border = world.method_8621();

        double centerX = 0.0, centerZ = 0.0;
        final double diameter = BASELINE_SIZE;

        if (!seenWalls.isEmpty()) {
            // The player can see one wall, or two walls that are on different axes, adjust the border such that the
            // walls they can't see are a constant and large distance away.
            if (seenWalls.contains(Wall.X_POSITIVE)) {
                double realXMax = border.method_11964() + border.method_11965() / 2;
                centerX = realXMax - (BASELINE_SIZE / 2);
                centerX += offset.getX();
            }
            if (seenWalls.contains(Wall.X_NEGATIVE)) {
                double realXMin = border.method_11964() - border.method_11965() / 2;
                centerX = realXMin + (BASELINE_SIZE / 2);
                centerX += offset.getX();
            }
            if (seenWalls.contains(Wall.Z_POSITIVE)) {
                double realZMax = border.method_11980() + border.method_11965() / 2;
                centerZ = realZMax - (BASELINE_SIZE / 2);
                centerZ += offset.getZ();
            }
            if (seenWalls.contains(Wall.Z_NEGATIVE)) {
                double realZMin = border.method_11980() - border.method_11965() / 2;
                centerZ = realZMin + (BASELINE_SIZE / 2);
                centerZ += offset.getZ();
            }
        } else {
            // The player cannot see any walls. Fully obfuscate the worldborder.
            centerX = centerZ = 0.0;
        }

        if (packet.method_65080().equals(class_9094.field_48021)) {
            var typedPacket = (class_5889)(packet);
            long l = typedPacket.method_34128();
            if (l > 0L) {
                dummyWorldBorder.method_11957(diameter, diameter, l);
            } else {
                dummyWorldBorder.method_11969(diameter);
            }
            dummyWorldBorder.method_11973(typedPacket.method_34129());
            dummyWorldBorder.method_11967(typedPacket.method_34131());
            dummyWorldBorder.method_11975(typedPacket.method_34130());
            dummyWorldBorder.method_11978(centerX * scaleFactor,  centerZ * scaleFactor);
            return new class_5889(dummyWorldBorder);
        }
        if (packet.method_65080().equals(class_9094.field_48082)) {
            dummyWorldBorder.method_11978(centerX * scaleFactor,  centerZ * scaleFactor);
            return new class_5895(dummyWorldBorder);
        }
        if (packet.method_65080().equals(class_9094.field_48083)) {
            var typedPacket = (class_5896)(packet);
            long l = typedPacket.method_34162();
            if (l > 0L) {
                dummyWorldBorder.method_11957(diameter, diameter, l);
            } else {
                dummyWorldBorder.method_11969(diameter);
            }
            return new class_5896(dummyWorldBorder);
        }
        if (packet.method_65080().equals(class_9094.field_48084)) {
            dummyWorldBorder.method_11969(diameter);
            return new class_5897(dummyWorldBorder);
        }
        return packet;
    }

    enum Wall {
        X_POSITIVE,
        X_NEGATIVE,
        Z_POSITIVE,
        Z_NEGATIVE
    }
}
