/*
 * Decompiled with CFR 0.152.
 */
package world.landfall.landfallessentials.database;

import java.sql.SQLException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import world.landfall.landfallessentials.database.MarketDAO;
import world.landfall.landfallessentials.database.RegionDAO;
import world.landfall.landfallessentials.database.RentalDAO;
import world.landfall.landfallessentials.regions.Region;
import world.landfall.landfallessentials.regions.RegionRental;

public class AsyncDatabaseWriter {
    private static final Logger LOGGER = LoggerFactory.getLogger(AsyncDatabaseWriter.class);
    private static final int FLUSH_INTERVAL_SECONDS = 5;
    private static ScheduledExecutorService executor;
    private static final Map<String, Region> pendingRegionWrites;
    private static final Map<String, RegionRental.RentalInfo> pendingRentalWrites;
    private static final Set<String> pendingRegionDeletes;
    private static final Set<String> pendingRentalDeletes;
    private static final Map<String, String> pendingMarketRegionAdds;
    private static final Set<String> pendingMarketRegionRemoves;
    private static volatile boolean initialized;
    private static volatile boolean shutdownInProgress;

    public static synchronized void initialize() {
        if (initialized) {
            LOGGER.warn("AsyncDatabaseWriter already initialized, skipping");
            return;
        }
        executor = Executors.newSingleThreadScheduledExecutor(r -> {
            Thread thread = new Thread(r, "LandfallEssentials-DBWriter");
            thread.setDaemon(false);
            return thread;
        });
        executor.scheduleAtFixedRate(() -> {
            try {
                AsyncDatabaseWriter.flush();
            }
            catch (Exception e) {
                LOGGER.error("Error during periodic database flush", (Throwable)e);
            }
        }, 5L, 5L, TimeUnit.SECONDS);
        initialized = true;
        LOGGER.info("AsyncDatabaseWriter initialized with {}s flush interval", (Object)5);
    }

    public static void queueRegionSave(Region region) {
        if (!initialized) {
            throw new IllegalStateException("AsyncDatabaseWriter not initialized. Call initialize() first.");
        }
        if (region == null || region.getName() == null) {
            LOGGER.warn("Attempted to queue null region or region with null name");
            return;
        }
        pendingRegionWrites.put(region.getName(), region);
        pendingRegionDeletes.remove(region.getName());
    }

    public static void queueRentalSave(RegionRental.RentalInfo rental) {
        if (!initialized) {
            throw new IllegalStateException("AsyncDatabaseWriter not initialized. Call initialize() first.");
        }
        if (rental == null || rental.getRegionName() == null) {
            LOGGER.warn("Attempted to queue null rental or rental with null region name");
            return;
        }
        pendingRentalWrites.put(rental.getRegionName(), rental);
        pendingRentalDeletes.remove(rental.getRegionName());
    }

    public static void queueRegionDelete(String regionName) {
        if (!initialized) {
            throw new IllegalStateException("AsyncDatabaseWriter not initialized. Call initialize() first.");
        }
        if (regionName == null) {
            LOGGER.warn("Attempted to queue null region name for deletion");
            return;
        }
        pendingRegionDeletes.add(regionName);
        pendingRegionWrites.remove(regionName);
    }

    public static void queueRentalDelete(String regionName) {
        if (!initialized) {
            throw new IllegalStateException("AsyncDatabaseWriter not initialized. Call initialize() first.");
        }
        if (regionName == null) {
            LOGGER.warn("Attempted to queue null region name for rental deletion");
            return;
        }
        pendingRentalDeletes.add(regionName);
        pendingRentalWrites.remove(regionName);
    }

    public static void queueMarketRegionAdd(String regionName, String marketName) {
        if (!initialized) {
            throw new IllegalStateException("AsyncDatabaseWriter not initialized. Call initialize() first.");
        }
        if (regionName == null || marketName == null) {
            LOGGER.warn("Attempted to queue null region or market name for market mapping");
            return;
        }
        pendingMarketRegionAdds.put(regionName, marketName);
        pendingMarketRegionRemoves.remove(regionName);
    }

    public static void queueMarketRegionRemove(String regionName) {
        if (!initialized) {
            throw new IllegalStateException("AsyncDatabaseWriter not initialized. Call initialize() first.");
        }
        if (regionName == null) {
            LOGGER.warn("Attempted to queue null region name for market mapping removal");
            return;
        }
        pendingMarketRegionRemoves.add(regionName);
        pendingMarketRegionAdds.remove(regionName);
    }

    public static void flush() {
        if (!initialized) {
            return;
        }
        HashMap<String, Region> regionsToSave = new HashMap<String, Region>(pendingRegionWrites);
        HashMap<String, RegionRental.RentalInfo> rentalsToSave = new HashMap<String, RegionRental.RentalInfo>(pendingRentalWrites);
        HashSet<String> regionsToDelete = new HashSet<String>(pendingRegionDeletes);
        HashSet<String> rentalsToDelete = new HashSet<String>(pendingRentalDeletes);
        HashMap<String, String> marketRegionsToAdd = new HashMap<String, String>(pendingMarketRegionAdds);
        HashSet<String> marketRegionsToRemove = new HashSet<String>(pendingMarketRegionRemoves);
        pendingRegionWrites.clear();
        pendingRentalWrites.clear();
        pendingRegionDeletes.clear();
        pendingRentalDeletes.clear();
        pendingMarketRegionAdds.clear();
        pendingMarketRegionRemoves.clear();
        if (regionsToSave.isEmpty() && rentalsToSave.isEmpty() && regionsToDelete.isEmpty() && rentalsToDelete.isEmpty() && marketRegionsToAdd.isEmpty() && marketRegionsToRemove.isEmpty()) {
            return;
        }
        long startTime = System.currentTimeMillis();
        int totalOperations = regionsToSave.size() + rentalsToSave.size() + regionsToDelete.size() + rentalsToDelete.size() + marketRegionsToAdd.size() + marketRegionsToRemove.size();
        try {
            for (String string : regionsToDelete) {
                try {
                    RegionDAO.deleteRegion(string);
                }
                catch (SQLException e) {
                    LOGGER.error("Failed to delete region '{}' from database", (Object)string, (Object)e);
                }
            }
            for (String string : rentalsToDelete) {
                try {
                    RentalDAO.deleteRental(string);
                }
                catch (SQLException e) {
                    LOGGER.error("Failed to delete rental for region '{}' from database", (Object)string, (Object)e);
                }
            }
            if (!regionsToSave.isEmpty()) {
                try {
                    RegionDAO.saveRegions(regionsToSave.values());
                }
                catch (SQLException e) {
                    LOGGER.error("Failed to batch save regions to database", (Throwable)e);
                    for (Region region : regionsToSave.values()) {
                        try {
                            RegionDAO.saveRegion(region);
                        }
                        catch (SQLException e2) {
                            LOGGER.error("Failed to save region '{}' to database", (Object)region.getName(), (Object)e2);
                        }
                    }
                }
            }
            if (!rentalsToSave.isEmpty()) {
                try {
                    RentalDAO.saveRentals(rentalsToSave.values());
                }
                catch (SQLException e) {
                    LOGGER.error("Failed to batch save rentals to database", (Throwable)e);
                    for (RegionRental.RentalInfo rental : rentalsToSave.values()) {
                        try {
                            RentalDAO.saveRental(rental);
                        }
                        catch (SQLException e2) {
                            LOGGER.error("Failed to save rental for region '{}' to database", (Object)rental.getRegionName(), (Object)e2);
                        }
                    }
                }
            }
            for (String string : marketRegionsToRemove) {
                try {
                    MarketDAO.removeRegionFromMarket(string);
                }
                catch (SQLException e) {
                    LOGGER.error("Failed to remove region '{}' from market in database", (Object)string, (Object)e);
                }
            }
            for (Map.Entry entry : marketRegionsToAdd.entrySet()) {
                try {
                    MarketDAO.addRegionToMarket((String)entry.getValue(), (String)entry.getKey());
                }
                catch (SQLException e) {
                    LOGGER.error("Failed to add region '{}' to market '{}' in database", new Object[]{entry.getKey(), entry.getValue(), e});
                }
            }
            long duration = System.currentTimeMillis() - startTime;
            if (duration > 100L) {
                LOGGER.info("Database flush completed: {} operations in {}ms", (Object)totalOperations, (Object)duration);
            }
        }
        catch (Exception e) {
            LOGGER.error("Unexpected error during database flush", (Throwable)e);
        }
    }

    public static synchronized void shutdown() {
        if (!initialized || shutdownInProgress) {
            return;
        }
        shutdownInProgress = true;
        LOGGER.info("Shutting down AsyncDatabaseWriter...");
        try {
            AsyncDatabaseWriter.flush();
            executor.shutdown();
            if (!executor.awaitTermination(30L, TimeUnit.SECONDS)) {
                LOGGER.warn("Executor did not terminate in 30 seconds, forcing shutdown");
                executor.shutdownNow();
            }
            initialized = false;
            LOGGER.info("AsyncDatabaseWriter shut down successfully");
        }
        catch (InterruptedException e) {
            LOGGER.error("Interrupted while shutting down AsyncDatabaseWriter", (Throwable)e);
            executor.shutdownNow();
            Thread.currentThread().interrupt();
        }
        finally {
            shutdownInProgress = false;
        }
    }

    public static boolean isInitialized() {
        return initialized;
    }

    public static int getPendingRegionWriteCount() {
        return pendingRegionWrites.size();
    }

    public static int getPendingRentalWriteCount() {
        return pendingRentalWrites.size();
    }

    public static void saveRegionNow(Region region) throws SQLException {
        RegionDAO.saveRegion(region);
    }

    public static void saveRentalNow(RegionRental.RentalInfo rental) throws SQLException {
        RentalDAO.saveRental(rental);
    }

    static {
        pendingRegionWrites = new ConcurrentHashMap<String, Region>();
        pendingRentalWrites = new ConcurrentHashMap<String, RegionRental.RentalInfo>();
        pendingRegionDeletes = ConcurrentHashMap.newKeySet();
        pendingRentalDeletes = ConcurrentHashMap.newKeySet();
        pendingMarketRegionAdds = new ConcurrentHashMap<String, String>();
        pendingMarketRegionRemoves = ConcurrentHashMap.newKeySet();
        initialized = false;
        shutdownInProgress = false;
    }
}

