/*
 * Decompiled with CFR 0.152.
 */
package world.landfall.landfallessentials.regions;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonSyntaxException;
import com.google.gson.reflect.TypeToken;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Reader;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import net.minecraft.core.BlockPos;
import net.minecraft.network.chat.Component;
import net.minecraft.network.protocol.Packet;
import net.minecraft.network.protocol.game.ClientboundStopSoundPacket;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.server.MinecraftServer;
import net.minecraft.server.level.ServerPlayer;
import net.minecraft.sounds.SoundEvent;
import net.minecraft.sounds.SoundSource;
import net.minecraft.world.damagesource.DamageSource;
import net.minecraft.world.entity.player.Player;
import net.minecraft.world.level.GameType;
import net.neoforged.bus.api.SubscribeEvent;
import net.neoforged.fml.common.EventBusSubscriber;
import net.neoforged.fml.loading.FMLPaths;
import net.neoforged.neoforge.event.tick.ServerTickEvent;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import world.landfall.landfallessentials.event.MobSpawnHandler;
import world.landfall.landfallessentials.regions.PermissionUtil;
import world.landfall.landfallessentials.regions.Region;
import world.landfall.landfallessentials.regions.RegionBlockManager;
import world.landfall.landfallessentials.regions.RegionRental;
import world.landfall.landfallessentials.regions.RegionVisualizer;
import world.landfall.landfallessentials.regions.RegionVisualizerNew;

@EventBusSubscriber(modid="landfallessentials")
public class RegionManager {
    private static final Logger LOGGER = LoggerFactory.getLogger(RegionManager.class);
    private static final Gson GSON = new GsonBuilder().excludeFieldsWithoutExposeAnnotation().setPrettyPrinting().create();
    private static final File REGIONS_FILE = new File(FMLPaths.CONFIGDIR.get().toFile(), "landfallessentials/regions.json");
    private static final Map<String, Region> regions = new ConcurrentHashMap<String, Region>();
    private static final Map<String, Set<String>> playerRegions = new ConcurrentHashMap<String, Set<String>>();
    private static final Map<String, String> playerPriorityRegion = new ConcurrentHashMap<String, String>();
    private static final Map<String, Long> playerDamageTimers = new ConcurrentHashMap<String, Long>();
    private static final Map<String, String> playerFlightGrantedBy = new ConcurrentHashMap<String, String>();
    private static final Map<String, Boolean> playerOriginalFlightState = new ConcurrentHashMap<String, Boolean>();
    private static final Set<String> adminBypassPlayers = ConcurrentHashMap.newKeySet();
    private static final Map<String, Map<String, Boolean>> permissionCache = new ConcurrentHashMap<String, Map<String, Boolean>>();
    private static final Map<String, Long> permissionCacheTimestamps = new ConcurrentHashMap<String, Long>();
    private static final int DAMAGE_INTERVAL_TICKS = 20;
    private static final float DAMAGE_AMOUNT = 4.0f;
    private static final int PERMISSION_CACHE_DURATION_TICKS = 40;
    private static final int REGION_CHECK_INTERVAL_TICKS = 5;
    private static final int CACHE_CLEANUP_INTERVAL_TICKS = 1200;
    private static final int MAX_CACHE_SIZE = 1000;
    private static long lastCacheCleanup = 0L;
    private static final ScheduledExecutorService cleanupExecutor = Executors.newSingleThreadScheduledExecutor(r -> {
        Thread t = new Thread(r, "RegionManager-Cleanup");
        t.setDaemon(true);
        return t;
    });

    public static void loadRegions() {
        regions.clear();
        playerRegions.clear();
        playerPriorityRegion.clear();
        playerDamageTimers.clear();
        playerFlightGrantedBy.clear();
        playerOriginalFlightState.clear();
        permissionCache.clear();
        permissionCacheTimestamps.clear();
        if (REGIONS_FILE.exists()) {
            try (FileReader reader = new FileReader(REGIONS_FILE);){
                Type type = new TypeToken<Map<String, Region>>(){}.getType();
                Map loadedRegions = (Map)GSON.fromJson((Reader)reader, type);
                if (loadedRegions != null) {
                    regions.putAll(loadedRegions);
                    LOGGER.info("Loaded {} regions from {}", (Object)regions.size(), (Object)REGIONS_FILE.getAbsolutePath());
                }
            }
            catch (JsonSyntaxException | IOException e) {
                LOGGER.error("Failed to load regions from file: {}", (Object)REGIONS_FILE.getAbsolutePath(), (Object)e);
            }
        } else {
            LOGGER.info("Regions file not found, starting with no regions. Will be created on save: {}", (Object)REGIONS_FILE.getAbsolutePath());
            REGIONS_FILE.getParentFile().mkdirs();
        }
    }

    public static void saveRegions() {
        REGIONS_FILE.getParentFile().mkdirs();
        try (FileWriter writer = new FileWriter(REGIONS_FILE);){
            GSON.toJson(regions, (Appendable)writer);
            LOGGER.info("Saved {} regions to {}", (Object)regions.size(), (Object)REGIONS_FILE.getAbsolutePath());
            RegionVisualizer.onRegionsChanged();
            RegionVisualizerNew.onRegionsChanged();
            RegionBlockManager.clearPermissionCache();
            MobSpawnHandler.updateSpawnControlFlags();
        }
        catch (IOException e) {
            LOGGER.error("Failed to save regions to file: {}", (Object)REGIONS_FILE.getAbsolutePath(), (Object)e);
        }
    }

    public static boolean addRegion(Region region) {
        if (region == null || regions.containsKey(region.getName())) {
            LOGGER.warn("Attempted to add null region or region with duplicate name: {}", (Object)(region == null ? "null" : region.getName()));
            return false;
        }
        regions.put(region.getName(), region);
        RegionManager.saveRegions();
        LOGGER.info("Added region: {}", (Object)region.getName());
        return true;
    }

    public static boolean removeRegion(String name) {
        Region removed = regions.remove(name);
        if (removed != null) {
            RegionManager.saveRegions();
            LOGGER.info("Removed region: {}", (Object)name);
            playerRegions.values().forEach(set -> {
                if (set != null) {
                    set.remove(name);
                }
            });
            playerPriorityRegion.entrySet().removeIf(entry -> name.equals(entry.getValue()));
            return true;
        }
        LOGGER.warn("Attempted to remove non-existent region: {}", (Object)name);
        return false;
    }

    public static Region getRegion(String name) {
        return regions.get(name);
    }

    public static Set<String> getRegionNames() {
        return regions.keySet();
    }

    public static Map<String, Region> getAllRegions() {
        return new ConcurrentHashMap<String, Region>(regions);
    }

    public static List<Region> getRegionsAt(BlockPos pos, String dimension) {
        ArrayList<Region> result = new ArrayList<Region>();
        for (Region region : regions.values()) {
            if (!region.getDimension().equals(dimension) || !region.contains(pos)) continue;
            result.add(region);
        }
        return result;
    }

    public static Set<String> getPlayerRegions(String playerUUID) {
        return new HashSet<String>(playerRegions.getOrDefault(playerUUID, ConcurrentHashMap.newKeySet()));
    }

    public static String getPlayerPriorityRegion(String playerUUID) {
        return playerPriorityRegion.get(playerUUID);
    }

    public static String getPlayerRegionInfo(String playerUUID) {
        Set<String> allRegions = RegionManager.getPlayerRegions(playerUUID);
        String priorityRegion = RegionManager.getPlayerPriorityRegion(playerUUID);
        if (allRegions.isEmpty()) {
            return "Player is not in any regions.";
        }
        StringBuilder info = new StringBuilder();
        info.append("Player is in ").append(allRegions.size()).append(" region(s):\n");
        for (String regionName : allRegions) {
            Region region = RegionManager.getRegion(regionName);
            if (region == null) continue;
            info.append("  - ").append(regionName);
            info.append(" (volume: ").append(region.getVolume()).append(" blocks)");
            if (regionName.equals(priorityRegion)) {
                info.append(" \u2190 PRIORITY");
            }
            info.append("\n");
        }
        if (priorityRegion != null) {
            info.append("Priority region: ").append(priorityRegion).append(" (smallest volume)");
        }
        return info.toString();
    }

    public static void cleanupPlayerData(String playerUUID) {
        playerRegions.remove(playerUUID);
        playerPriorityRegion.remove(playerUUID);
        playerDamageTimers.remove(playerUUID);
        playerFlightGrantedBy.remove(playerUUID);
        playerOriginalFlightState.remove(playerUUID);
        permissionCache.remove(playerUUID);
        permissionCacheTimestamps.remove(playerUUID);
        adminBypassPlayers.remove(playerUUID);
        try {
            UUID uuid = UUID.fromString(playerUUID);
            for (Region region : regions.values()) {
                region.cleanupPlayerCache(uuid);
            }
        }
        catch (IllegalArgumentException e) {
            LOGGER.warn("Invalid UUID format for player cleanup: {}", (Object)playerUUID);
        }
        LOGGER.debug("Cleaned up region data for player UUID: {}", (Object)playerUUID);
    }

    public static boolean enableAdminBypass(String playerUUID) {
        boolean added = adminBypassPlayers.add(playerUUID);
        if (added) {
            LOGGER.info("Enabled admin bypass for player UUID: {}", (Object)playerUUID);
        }
        return added;
    }

    public static boolean disableAdminBypass(String playerUUID) {
        boolean removed = adminBypassPlayers.remove(playerUUID);
        if (removed) {
            LOGGER.info("Disabled admin bypass for player UUID: {}", (Object)playerUUID);
        }
        return removed;
    }

    public static boolean toggleAdminBypass(String playerUUID) {
        if (adminBypassPlayers.contains(playerUUID)) {
            RegionManager.disableAdminBypass(playerUUID);
            return false;
        }
        RegionManager.enableAdminBypass(playerUUID);
        return true;
    }

    public static boolean hasAdminBypass(String playerUUID) {
        return adminBypassPlayers.contains(playerUUID);
    }

    public static Set<String> getAdminBypassPlayers() {
        return new HashSet<String>(adminBypassPlayers);
    }

    private static boolean hasPermissionCached(ServerPlayer player, String permission, long currentTime) {
        Map<String, Boolean> playerPermissions;
        String playerUUID = player.getUUID().toString();
        Long cacheTime = permissionCacheTimestamps.get(playerUUID);
        if (cacheTime != null && currentTime - cacheTime < 40L && (playerPermissions = permissionCache.get(playerUUID)) != null && playerPermissions.containsKey(permission)) {
            return playerPermissions.get(permission);
        }
        boolean hasPermission = PermissionUtil.hasPermission(player, permission);
        permissionCache.computeIfAbsent(playerUUID, k -> new ConcurrentHashMap()).put(permission, hasPermission);
        permissionCacheTimestamps.put(playerUUID, currentTime);
        if (permissionCache.size() > 1000) {
            LOGGER.warn("Permission cache exceeded max size ({}), forcing cleanup", (Object)1000);
            RegionManager.cleanupPermissionCache(currentTime);
        }
        return hasPermission;
    }

    private static void cleanupPermissionCache(long currentTime) {
        ArrayList<String> expiredPlayers = new ArrayList<String>();
        for (Map.Entry<String, Long> entry : permissionCacheTimestamps.entrySet()) {
            if (currentTime - entry.getValue() <= 80L) continue;
            expiredPlayers.add(entry.getKey());
        }
        for (String playerUUID : expiredPlayers) {
            permissionCache.remove(playerUUID);
            permissionCacheTimestamps.remove(playerUUID);
        }
        if (!expiredPlayers.isEmpty()) {
            LOGGER.debug("Cleaned up {} expired permission cache entries", (Object)expiredPlayers.size());
        }
    }

    @SubscribeEvent
    public static void onServerTick(ServerTickEvent.Post event) {
        MinecraftServer server = event.getServer();
        if (server == null) {
            return;
        }
        long currentTime = server.getTickCount();
        if (currentTime % 5L != 0L) {
            return;
        }
        if (currentTime - lastCacheCleanup > 1200L) {
            RegionManager.cleanupPermissionCache(currentTime);
            lastCacheCleanup = currentTime;
        }
        if (regions.isEmpty()) {
            if (!playerPriorityRegion.isEmpty()) {
                for (String playerUUID : new HashSet<String>(playerPriorityRegion.keySet())) {
                    Region lastKnownRegion;
                    ServerPlayer playerToClear = server.getPlayerList().getPlayer(UUID.fromString(playerUUID));
                    Region region = lastKnownRegion = playerPriorityRegion.get(playerUUID) != null ? regions.get(playerPriorityRegion.get(playerUUID)) : null;
                    if (playerToClear == null || lastKnownRegion != null) {
                        // empty if block
                    }
                    RegionManager.cleanupPlayerData(playerUUID);
                }
                LOGGER.debug("Cleared all player region tracking as no regions are defined.");
            }
            return;
        }
        List players = server.getPlayerList().getPlayers();
        if (players.isEmpty()) {
            return;
        }
        for (ServerPlayer player : players) {
            String newPriorityRegionName;
            String playerUUID = player.getUUID().toString();
            HashSet<Region> actualCurrentRegions = new HashSet<Region>();
            for (Region region : regions.values()) {
                if (!region.contains((Player)player)) continue;
                actualCurrentRegions.add(region);
            }
            boolean inRestrictedRegionThisTick = false;
            Region currentRestrictedRegionForDamage = null;
            if (!RegionManager.hasAdminBypass(playerUUID) && !actualCurrentRegions.isEmpty()) {
                ArrayList<Region> restrictedRegionsPlayerIsIn = new ArrayList<Region>();
                for (Region region : actualCurrentRegions) {
                    if (!RegionManager.hasPermissionTagsButPlayerLacksPermission(region, player, currentTime)) continue;
                    restrictedRegionsPlayerIsIn.add(region);
                }
                currentRestrictedRegionForDamage = Region.getSmallestRegion(restrictedRegionsPlayerIsIn);
                if (currentRestrictedRegionForDamage != null) {
                    inRestrictedRegionThisTick = true;
                }
            }
            if (inRestrictedRegionThisTick) {
                Long lastDamageTime = playerDamageTimers.get(playerUUID);
                if (lastDamageTime == null || currentTime >= lastDamageTime + 20L) {
                    DamageSource damageSource = player.damageSources().generic();
                    player.invulnerableTime = 10;
                    player.hurt(damageSource, 4.0f);
                    playerDamageTimers.put(playerUUID, currentTime);
                    if (lastDamageTime == null || currentTime >= lastDamageTime + 100L) {
                        if (currentRestrictedRegionForDamage.hasAccessDeniedMessage()) {
                            player.sendSystemMessage((Component)Component.literal((String)currentRestrictedRegionForDamage.getAccessDeniedMessage().replace("&", "\u00a7")));
                        } else {
                            player.sendSystemMessage((Component)Component.literal((String)("\u00a7cYou are in a restricted area: " + currentRestrictedRegionForDamage.getName())));
                        }
                    }
                    LOGGER.debug("Damaged {} in region {}. Permission tags missing: {}", new Object[]{player.getName().getString(), currentRestrictedRegionForDamage.getName(), RegionManager.getPermissionTagsFromRegion(currentRestrictedRegionForDamage)});
                }
            } else {
                playerDamageTimers.remove(playerUUID);
            }
            HashSet<Region> allowedRegionsForEffects = new HashSet<Region>();
            for (Region region : actualCurrentRegions) {
                if (!RegionManager.regionAllowsPlayerEntry(region, player, currentTime)) continue;
                allowedRegionsForEffects.add(region);
            }
            String oldPriorityRegionName = playerPriorityRegion.get(playerUUID);
            Region oldPriorityRegion = oldPriorityRegionName != null ? regions.get(oldPriorityRegionName) : null;
            Region newPriorityRegion = Region.getSmallestRegion(allowedRegionsForEffects);
            String string = newPriorityRegionName = newPriorityRegion != null ? newPriorityRegion.getName() : null;
            if (!Objects.equals(oldPriorityRegionName, newPriorityRegionName)) {
                if (oldPriorityRegion != null) {
                    LOGGER.debug("Player {} priority changing FROM {}. Applying exit effects.", (Object)player.getName().getString(), (Object)oldPriorityRegionName);
                    RegionManager.applyExitEffects(player, oldPriorityRegion, currentTime);
                }
                if (newPriorityRegion != null) {
                    LOGGER.debug("Player {} priority changing TO {}. Applying enter effects.", (Object)player.getName().getString(), (Object)newPriorityRegionName);
                    RegionManager.applyEnterEffects(player, newPriorityRegion, currentTime);
                }
            }
            if (newPriorityRegionName == null) {
                playerPriorityRegion.remove(playerUUID);
                playerRegions.remove(playerUUID);
            } else {
                playerPriorityRegion.put(playerUUID, newPriorityRegionName);
                playerRegions.put(playerUUID, allowedRegionsForEffects.stream().map(Region::getName).collect(Collectors.toSet()));
            }
            if (newPriorityRegion != null) {
                RegionManager.applyLoopingEffects(player, newPriorityRegion, currentTime);
            }
            RegionManager.handlePlayerFlight(player);
            if (!RegionManager.hasAdminBypass(playerUUID) || currentTime % 20L != 0L) continue;
            player.displayClientMessage((Component)Component.literal((String)"\u00a76[ADMIN BYPASS MODE ACTIVE]"), true);
        }
    }

    private static void applyEnterEffects(ServerPlayer player, Region region, long currentTime) {
        if (region.hasEntryMessage()) {
            try {
                player.sendSystemMessage((Component)Component.literal((String)region.getEntryMessage().replace("&", "\u00a7")));
            }
            catch (Exception e) {
                LOGGER.error("Failed to send entry message for region {}: {}", new Object[]{region.getName(), region.getEntryMessage(), e});
            }
        }
        if (region.hasEntryActionbarMessage()) {
            try {
                player.displayClientMessage((Component)Component.literal((String)region.getEntryActionbarMessage().replace("&", "\u00a7")), true);
            }
            catch (Exception e) {
                LOGGER.error("Failed to send entry actionbar message for region {}: {}", new Object[]{region.getName(), region.getEntryActionbarMessage(), e});
            }
        } else {
            String rentalMessage = RegionRental.getRentalStatusMessage(region.getName());
            if (rentalMessage != null) {
                try {
                    player.displayClientMessage((Component)Component.literal((String)rentalMessage), true);
                }
                catch (Exception e) {
                    LOGGER.error("Failed to send rental status message for region {}: {}", new Object[]{region.getName(), rentalMessage, e});
                }
            }
        }
        if (region.hasSound()) {
            RegionManager.playSoundForPlayer(player, region);
        }
        if (region.hasGamemodeOnEntry()) {
            try {
                GameType newGameType = GameType.byName((String)region.getGamemodeOnEntry().toLowerCase());
                if (player.gameMode.getGameModeForPlayer() != newGameType) {
                    player.setGameMode(newGameType);
                    LOGGER.debug("Set gamemode to {} for player {} in region {}", new Object[]{newGameType.getName(), player.getName().getString(), region.getName()});
                }
            }
            catch (IllegalArgumentException e) {
                LOGGER.warn("Invalid gamemode string ''{}'' for region {}. Player gamemode not changed.", (Object)region.getGamemodeOnEntry(), (Object)region.getName());
            }
            catch (Exception e) {
                LOGGER.error("Failed to set gamemode for player {} in region {}", new Object[]{player.getName().getString(), region.getName(), e});
            }
        }
    }

    private static void applyExitEffects(ServerPlayer player, Region region, long currentTime) {
        if (region.hasExitMessage()) {
            try {
                player.sendSystemMessage((Component)Component.literal((String)region.getExitMessage().replace("&", "\u00a7")));
            }
            catch (Exception e) {
                LOGGER.error("Failed to send exit message for region {}: {}", new Object[]{region.getName(), region.getExitMessage(), e});
            }
        }
        if (region.hasExitActionbarMessage()) {
            try {
                player.displayClientMessage((Component)Component.literal((String)region.getExitActionbarMessage().replace("&", "\u00a7")), true);
            }
            catch (Exception e) {
                LOGGER.error("Failed to send exit actionbar message for region {}: {}", new Object[]{region.getName(), region.getExitActionbarMessage(), e});
            }
        }
        if (region.hasSound()) {
            RegionManager.stopSoundForPlayer(player, region);
        }
        if (region.hasGamemodeOnExit()) {
            try {
                GameType newGameType = GameType.byName((String)region.getGamemodeOnExit().toLowerCase());
                if (player.gameMode.getGameModeForPlayer() != newGameType) {
                    player.setGameMode(newGameType);
                    LOGGER.debug("Set gamemode to {} for player {} on exiting region {}", new Object[]{newGameType.getName(), player.getName().getString(), region.getName()});
                }
            }
            catch (IllegalArgumentException e) {
                LOGGER.warn("Invalid gamemodeOnExit string ''{}'' for region {}. Player gamemode not changed.", (Object)region.getGamemodeOnExit(), (Object)region.getName());
            }
            catch (Exception e) {
                LOGGER.error("Failed to set gamemode on exit for player {} in region {}", new Object[]{player.getName().getString(), region.getName(), e});
            }
        }
    }

    private static void applyLoopingEffects(ServerPlayer player, Region region, long currentTime) {
        if (region.hasSound() && region.getSoundLoopDelayTicks() > 0 && currentTime >= region.getLastSoundPlayTime() + (long)region.getSoundLoopDelayTicks()) {
            RegionManager.playSoundForPlayer(player, region);
            region.setLastSoundPlayTime(currentTime);
        }
    }

    private static void playSoundForPlayer(ServerPlayer player, Region region) {
        if (region.getSoundEventId() == null || region.getSoundEventId().isBlank()) {
            return;
        }
        try {
            ResourceLocation soundLocation = ResourceLocation.tryParse((String)region.getSoundEventId());
            if (soundLocation == null) {
                LOGGER.warn("Invalid sound event ID: {}", (Object)region.getSoundEventId());
                return;
            }
            SoundEvent soundEvent = SoundEvent.createVariableRangeEvent((ResourceLocation)soundLocation);
            SoundSource source = SoundSource.valueOf((String)region.getSoundCategory().toUpperCase());
            player.playNotifySound(soundEvent, source, region.getSoundVolume(), region.getSoundPitch());
            LOGGER.debug("Playing sound {} for player {} in region {}", new Object[]{region.getSoundEventId(), player.getName().getString(), region.getName()});
        }
        catch (IllegalArgumentException e) {
            LOGGER.warn("Invalid sound category: {}. Defaulting to AMBIENT. Error: {}", (Object)region.getSoundCategory(), (Object)e.getMessage());
            try {
                ResourceLocation soundLocation = ResourceLocation.tryParse((String)region.getSoundEventId());
                SoundEvent soundEvent = SoundEvent.createVariableRangeEvent((ResourceLocation)soundLocation);
                player.playNotifySound(soundEvent, SoundSource.AMBIENT, region.getSoundVolume(), region.getSoundPitch());
            }
            catch (Exception ex) {
                LOGGER.error("Failed to play sound (even with default category) for player {} in region {}: {}", new Object[]{player.getName().getString(), region.getName(), region.getSoundEventId(), ex});
            }
        }
        catch (Exception e) {
            LOGGER.error("Failed to play sound for player {} in region {}: {}", new Object[]{player.getName().getString(), region.getName(), region.getSoundEventId(), e});
        }
    }

    private static void stopSoundForPlayer(ServerPlayer player, Region region) {
        if (region.getSoundEventId() == null || region.getSoundEventId().isBlank()) {
            return;
        }
        try {
            ResourceLocation soundLocation = ResourceLocation.tryParse((String)region.getSoundEventId());
            if (soundLocation == null) {
                LOGGER.warn("Invalid sound event ID for stopping: {}", (Object)region.getSoundEventId());
                return;
            }
            SoundEvent soundEvent = SoundEvent.createVariableRangeEvent((ResourceLocation)soundLocation);
            SoundSource source = SoundSource.valueOf((String)region.getSoundCategory().toUpperCase());
            player.connection.send((Packet)new ClientboundStopSoundPacket(soundLocation, source));
            LOGGER.debug("Stopping sound {} for player {} on exiting region {}", new Object[]{region.getSoundEventId(), player.getName().getString(), region.getName()});
        }
        catch (IllegalArgumentException e) {
            LOGGER.warn("Invalid sound category for stopping: {}. Defaulting to AMBIENT. Error: {}", (Object)region.getSoundCategory(), (Object)e.getMessage());
            try {
                ResourceLocation soundLocation = ResourceLocation.tryParse((String)region.getSoundEventId());
                player.connection.send((Packet)new ClientboundStopSoundPacket(soundLocation, SoundSource.AMBIENT));
            }
            catch (Exception ex) {
                LOGGER.error("Failed to stop sound (even with default category) for player {} in region {}: {}", new Object[]{player.getName().getString(), region.getName(), region.getSoundEventId(), ex});
            }
        }
        catch (Exception e) {
            LOGGER.error("Failed to stop sound for player {} in region {}: {}", new Object[]{player.getName().getString(), region.getName(), region.getSoundEventId(), e});
        }
    }

    public static void initialize() {
        RegionManager.loadRegions();
        cleanupExecutor.scheduleAtFixedRate(() -> {
            try {
                RegionManager.cleanupPermissionCache(System.currentTimeMillis());
                LOGGER.debug("Performed scheduled permission cache cleanup");
            }
            catch (Exception e) {
                LOGGER.error("Error during scheduled cache cleanup", (Throwable)e);
            }
        }, 5L, 5L, TimeUnit.MINUTES);
        LOGGER.info("RegionManager initialized with scheduled cleanup tasks");
    }

    public static void shutdown() {
        try {
            cleanupExecutor.shutdown();
            if (!cleanupExecutor.awaitTermination(5L, TimeUnit.SECONDS)) {
                cleanupExecutor.shutdownNow();
            }
            LOGGER.info("RegionManager shutdown complete");
        }
        catch (InterruptedException e) {
            cleanupExecutor.shutdownNow();
            Thread.currentThread().interrupt();
        }
    }

    public static boolean isPlayerInTaggedRegion(Player player, String tag) {
        if (player == null || tag == null || tag.trim().isEmpty()) {
            return false;
        }
        String playerUUID = player.getUUID().toString();
        Set currentRegions = playerRegions.getOrDefault(playerUUID, ConcurrentHashMap.newKeySet());
        for (String regionName : currentRegions) {
            Region region = regions.get(regionName);
            if (region == null || !region.hasTag(tag.trim())) continue;
            return true;
        }
        return false;
    }

    public static Set<String> getPlayerRegionTags(Player player) {
        if (player == null) {
            return ConcurrentHashMap.newKeySet();
        }
        String playerUUID = player.getUUID().toString();
        Set currentRegions = playerRegions.getOrDefault(playerUUID, ConcurrentHashMap.newKeySet());
        ConcurrentHashMap.KeySetView allTags = ConcurrentHashMap.newKeySet();
        for (String regionName : currentRegions) {
            Region region = regions.get(regionName);
            if (region == null) continue;
            allTags.addAll(region.getTags());
        }
        return allTags;
    }

    public static List<String> getRegionsWithTag(String tag) {
        if (tag == null || tag.trim().isEmpty()) {
            return new ArrayList<String>();
        }
        String trimmedTag = tag.trim();
        return regions.values().stream().filter(region -> region.hasTag(trimmedTag)).map(Region::getName).collect(Collectors.toList());
    }

    public static Set<String> getAllTags() {
        ConcurrentHashMap.KeySetView allTags = ConcurrentHashMap.newKeySet();
        for (Region region : regions.values()) {
            allTags.addAll(region.getTags());
        }
        return allTags;
    }

    public static Set<String> getRegionTags(String regionName) {
        if (regionName == null || regionName.trim().isEmpty()) {
            return ConcurrentHashMap.newKeySet();
        }
        Region region = regions.get(regionName.trim());
        if (region != null) {
            return region.getTags();
        }
        return ConcurrentHashMap.newKeySet();
    }

    public static boolean regionHasTag(String regionName, String tag) {
        if (regionName == null || regionName.trim().isEmpty() || tag == null || tag.trim().isEmpty()) {
            return false;
        }
        Region region = regions.get(regionName.trim());
        return region != null && region.hasTag(tag.trim());
    }

    public static boolean addTagToRegion(String regionName, String tag) {
        if (regionName == null || regionName.trim().isEmpty() || tag == null || tag.trim().isEmpty()) {
            return false;
        }
        Region region = regions.get(regionName.trim());
        if (region != null) {
            boolean added = region.addTag(tag.trim());
            if (added) {
                RegionManager.saveRegions();
                LOGGER.info("Added tag '{}' to region '{}'", (Object)tag.trim(), (Object)regionName.trim());
            }
            return added;
        }
        return false;
    }

    public static boolean removeTagFromRegion(String regionName, String tag) {
        if (regionName == null || regionName.trim().isEmpty() || tag == null || tag.trim().isEmpty()) {
            return false;
        }
        Region region = regions.get(regionName.trim());
        if (region != null) {
            boolean removed = region.removeTag(tag.trim());
            if (removed) {
                RegionManager.saveRegions();
                LOGGER.info("Removed tag '{}' from region '{}'", (Object)tag.trim(), (Object)regionName.trim());
            }
            return removed;
        }
        return false;
    }

    public static boolean clearRegionTags(String regionName) {
        if (regionName == null || regionName.trim().isEmpty()) {
            return false;
        }
        Region region = regions.get(regionName.trim());
        if (region != null && region.hasTags()) {
            region.clearTags();
            RegionManager.saveRegions();
            LOGGER.info("Cleared all tags from region '{}'", (Object)regionName.trim());
            return true;
        }
        return false;
    }

    private static String getFlightGrantingRegion(ServerPlayer player) {
        String playerUUID = player.getUUID().toString();
        Set currentRegions = playerRegions.getOrDefault(playerUUID, ConcurrentHashMap.newKeySet());
        for (String regionName : currentRegions) {
            Region region = regions.get(regionName);
            if (region == null) continue;
            for (String tag : region.getTags()) {
                String permissionNode;
                if (!tag.startsWith("flight:") || tag.length() <= 7 || !RegionManager.hasPermissionCached(player, permissionNode = tag.substring(7), player.getServer().getTickCount())) continue;
                return regionName;
            }
        }
        return null;
    }

    private static void grantFlight(ServerPlayer player, String grantingRegion) {
        String playerUUID = player.getUUID().toString();
        if (!playerOriginalFlightState.containsKey(playerUUID)) {
            playerOriginalFlightState.put(playerUUID, player.getAbilities().mayfly);
        }
        player.getAbilities().mayfly = true;
        player.getAbilities().flying = true;
        player.onUpdateAbilities();
        playerFlightGrantedBy.put(playerUUID, grantingRegion);
        LOGGER.debug("Granted flight to player {} by region {}", (Object)player.getName().getString(), (Object)grantingRegion);
    }

    private static void removeFlight(ServerPlayer player) {
        String playerUUID = player.getUUID().toString();
        Boolean originalState = playerOriginalFlightState.get(playerUUID);
        if (originalState != null) {
            player.getAbilities().mayfly = originalState;
            if (!originalState.booleanValue()) {
                player.getAbilities().flying = false;
            }
            player.onUpdateAbilities();
        }
        String grantingRegion = playerFlightGrantedBy.remove(playerUUID);
        playerOriginalFlightState.remove(playerUUID);
        LOGGER.debug("Removed flight from player {} (was granted by region {})", (Object)player.getName().getString(), (Object)grantingRegion);
    }

    private static void handlePlayerFlight(ServerPlayer player) {
        String playerUUID = player.getUUID().toString();
        String currentFlightRegion = playerFlightGrantedBy.get(playerUUID);
        String shouldGrantFlightRegion = RegionManager.getFlightGrantingRegion(player);
        if (shouldGrantFlightRegion != null && !shouldGrantFlightRegion.equals(currentFlightRegion)) {
            if (currentFlightRegion != null) {
                RegionManager.removeFlight(player);
            }
            RegionManager.grantFlight(player, shouldGrantFlightRegion);
        } else if (shouldGrantFlightRegion == null && currentFlightRegion != null) {
            RegionManager.removeFlight(player);
        }
    }

    private static void cleanupPlayerFlightData(String playerUUID) {
        playerFlightGrantedBy.remove(playerUUID);
        playerOriginalFlightState.remove(playerUUID);
    }

    public static Map<String, RegionNode> buildRegionHierarchy() {
        ConcurrentHashMap<String, RegionNode> nodeMap = new ConcurrentHashMap<String, RegionNode>();
        for (Map.Entry<String, Region> entry : regions.entrySet()) {
            nodeMap.put(entry.getKey(), new RegionNode(entry.getKey(), entry.getValue()));
        }
        for (Map.Entry<String, Region> entry : nodeMap.entrySet()) {
            Region childRegion = ((RegionNode)((Object)entry.getValue())).getRegion();
            RegionNode bestParent = null;
            long smallestParentVolume = Long.MAX_VALUE;
            for (Map.Entry parentEntry : nodeMap.entrySet()) {
                long parentVolume;
                Region parentRegion;
                if (entry.getKey().equals(parentEntry.getKey()) || !childRegion.isFullyContainedWithin(parentRegion = ((RegionNode)parentEntry.getValue()).getRegion()) || (parentVolume = parentRegion.getVolume()) >= smallestParentVolume) continue;
                bestParent = (RegionNode)parentEntry.getValue();
                smallestParentVolume = parentVolume;
            }
            if (bestParent == null) continue;
            bestParent.addChild((RegionNode)((Object)entry.getValue()));
        }
        return nodeMap;
    }

    public static List<String> getTopLevelRegions() {
        Map<String, RegionNode> hierarchy = RegionManager.buildRegionHierarchy();
        ArrayList<String> topLevel = new ArrayList<String>();
        for (RegionNode node : hierarchy.values()) {
            if (!node.isTopLevel()) continue;
            topLevel.add(node.getRegionName());
        }
        topLevel.sort(String::compareTo);
        return topLevel;
    }

    public static List<String> getChildRegions(String regionName) {
        Map<String, RegionNode> hierarchy = RegionManager.buildRegionHierarchy();
        RegionNode node = hierarchy.get(regionName);
        if (node == null) {
            return new ArrayList<String>();
        }
        ArrayList<String> children = new ArrayList<String>();
        for (RegionNode child : node.getChildren()) {
            children.add(child.getRegionName());
        }
        children.sort(String::compareTo);
        return children;
    }

    public static String getParentRegion(String regionName) {
        Map<String, RegionNode> hierarchy = RegionManager.buildRegionHierarchy();
        RegionNode node = hierarchy.get(regionName);
        if (node == null || node.getParent() == null) {
            return null;
        }
        return node.getParent().getRegionName();
    }

    public static String buildRegionTree(String regionName, String indent, String prefix, boolean isLast) {
        StringBuilder tree = new StringBuilder();
        Map<String, RegionNode> hierarchy = RegionManager.buildRegionHierarchy();
        if (regionName == null) {
            List<String> topLevel = RegionManager.getTopLevelRegions();
            for (int i = 0; i < topLevel.size(); ++i) {
                String topRegion = topLevel.get(i);
                boolean last = i == topLevel.size() - 1;
                tree.append(RegionManager.buildRegionTree(topRegion, "", "", last));
            }
        } else {
            RegionNode node = hierarchy.get(regionName);
            if (node == null) {
                return "";
            }
            Region region = node.getRegion();
            tree.append(indent).append(prefix);
            tree.append("\u00a7e").append(regionName);
            tree.append(" \u00a77(").append(region.getVolume()).append(" blocks");
            if (region.hasTags()) {
                tree.append(", tags: ").append(String.join((CharSequence)",", region.getTags()));
            }
            tree.append(")\n");
            List<RegionNode> children = node.getChildren();
            for (int i = 0; i < children.size(); ++i) {
                RegionNode child = children.get(i);
                boolean childIsLast = i == children.size() - 1;
                String childIndent = indent + (isLast ? "  " : "\u2502 ");
                String childPrefix = childIsLast ? "\u2514\u2500" : "\u251c\u2500";
                tree.append(RegionManager.buildRegionTree(child.getRegionName(), childIndent, childPrefix, childIsLast));
            }
        }
        return tree.toString();
    }

    private static boolean hasPermissionTagsButPlayerLacksPermission(Region region, ServerPlayer player, long currentTime) {
        for (String tag : region.getTags()) {
            String permissionNode;
            if (!tag.startsWith("permission:") || RegionManager.hasPermissionCached(player, permissionNode = tag.substring(11), currentTime)) continue;
            return true;
        }
        return false;
    }

    private static boolean regionAllowsPlayerEntry(Region region, ServerPlayer player, long currentTime) {
        String playerUUID = player.getUUID().toString();
        if (RegionManager.hasAdminBypass(playerUUID)) {
            return true;
        }
        for (String tag : region.getTags()) {
            String permissionNode;
            if (!tag.startsWith("permission:") || RegionManager.hasPermissionCached(player, permissionNode = tag.substring(11), currentTime)) continue;
            return false;
        }
        return true;
    }

    private static String getPermissionTagsFromRegion(Region region) {
        return region.getTags().stream().filter(tag -> tag.startsWith("permission:")).map(tag -> tag.substring(11)).reduce((a, b) -> a + ", " + b).orElse("none");
    }

    public static class RegionNode {
        private final String regionName;
        private final Region region;
        private final List<RegionNode> children;
        private RegionNode parent;

        public RegionNode(String regionName, Region region) {
            this.regionName = regionName;
            this.region = region;
            this.children = new ArrayList<RegionNode>();
            this.parent = null;
        }

        public void addChild(RegionNode child) {
            this.children.add(child);
            child.parent = this;
        }

        public String getRegionName() {
            return this.regionName;
        }

        public Region getRegion() {
            return this.region;
        }

        public List<RegionNode> getChildren() {
            return new ArrayList<RegionNode>(this.children);
        }

        public RegionNode getParent() {
            return this.parent;
        }

        public boolean hasChildren() {
            return !this.children.isEmpty();
        }

        public boolean isTopLevel() {
            return this.parent == null;
        }
    }
}

