/*
 * Decompiled with CFR 0.152.
 */
package world.landfall.landfallessentials.regions.market;

import java.sql.SQLException;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import world.landfall.landfallessentials.database.AsyncDatabaseWriter;
import world.landfall.landfallessentials.database.MarketDAO;
import world.landfall.landfallessentials.regions.RegionManager;
import world.landfall.landfallessentials.regions.RegionRental;
import world.landfall.landfallessentials.regions.market.MarketConfig;

public class MarketManager {
    private static final Logger LOGGER = LoggerFactory.getLogger(MarketManager.class);
    private static final Map<String, MarketConfig> markets = new ConcurrentHashMap<String, MarketConfig>();
    private static final Map<String, String> regionToMarket = new ConcurrentHashMap<String, String>();
    private static final Map<String, MarketStats> marketStats = new ConcurrentHashMap<String, MarketStats>();

    public static void initialize() {
        try {
            Map<String, MarketConfig> loaded = MarketDAO.loadAllMarkets();
            markets.clear();
            markets.putAll(loaded);
            Map<String, String> mappings = MarketDAO.loadRegionMarketMappings();
            regionToMarket.clear();
            regionToMarket.putAll(mappings);
            MarketManager.rebuildAllStats();
            LOGGER.info("Market system initialized: {} markets, {} region mappings", (Object)markets.size(), (Object)regionToMarket.size());
        }
        catch (SQLException e) {
            LOGGER.error("Failed to initialize market system", (Throwable)e);
        }
    }

    public static boolean createMarket(String name, int floorPrice, int ceilingPrice, double priceExponent, String description) {
        if (markets.containsKey(name)) {
            return false;
        }
        MarketConfig market = new MarketConfig(name, floorPrice, ceilingPrice, priceExponent, description);
        markets.put(name, market);
        marketStats.put(name, new MarketStats());
        try {
            MarketDAO.saveMarket(market);
        }
        catch (SQLException e) {
            LOGGER.error("Failed to save new market: {}", (Object)name, (Object)e);
            markets.remove(name);
            marketStats.remove(name);
            return false;
        }
        LOGGER.info("Created market: {}", (Object)market);
        return true;
    }

    public static boolean deleteMarket(String name) {
        MarketConfig removed = markets.remove(name);
        if (removed == null) {
            return false;
        }
        regionToMarket.entrySet().removeIf(entry -> ((String)entry.getValue()).equals(name));
        marketStats.remove(name);
        try {
            MarketDAO.deleteMarket(name);
        }
        catch (SQLException e) {
            LOGGER.error("Failed to delete market from database: {}", (Object)name, (Object)e);
        }
        LOGGER.info("Deleted market: {}", (Object)name);
        return true;
    }

    public static MarketConfig getMarket(String name) {
        return markets.get(name);
    }

    public static MarketConfig getMarketForRegion(String regionName) {
        String marketName = regionToMarket.get(regionName);
        return marketName != null ? markets.get(marketName) : null;
    }

    public static Collection<MarketConfig> getAllMarkets() {
        return Collections.unmodifiableCollection(markets.values());
    }

    public static Set<String> getMarketNames() {
        return new HashSet<String>(markets.keySet());
    }

    public static Set<String> getRegionsInMarket(String marketName) {
        HashSet<String> regions = new HashSet<String>();
        for (Map.Entry<String, String> entry : regionToMarket.entrySet()) {
            if (!entry.getValue().equals(marketName)) continue;
            regions.add(entry.getKey());
        }
        return regions;
    }

    public static int getCurrentPrice(String marketName) {
        MarketConfig market = markets.get(marketName);
        if (market == null) {
            return -1;
        }
        MarketStats stats = marketStats.get(marketName);
        if (stats == null) {
            return market.getFloorPrice();
        }
        return market.calculateCurrentPrice(stats.totalRegions, stats.rentedRegions);
    }

    public static double getOccupancyRate(String marketName) {
        MarketStats stats = marketStats.get(marketName);
        return stats != null ? stats.getOccupancyRate() : 0.0;
    }

    public static int[] getMarketStats(String marketName) {
        MarketStats stats = marketStats.get(marketName);
        if (stats == null) {
            return null;
        }
        return new int[]{stats.totalRegions, stats.rentedRegions};
    }

    public static void onRegionRented(String regionName) {
        MarketStats stats;
        String marketName = regionToMarket.get(regionName);
        if (marketName != null && (stats = marketStats.get(marketName)) != null) {
            ++stats.rentedRegions;
            LOGGER.debug("Market '{}' occupancy: {}/{} ({:.1f}%)", new Object[]{marketName, stats.rentedRegions, stats.totalRegions, stats.getOccupancyRate() * 100.0});
        }
    }

    public static void onRegionUnrented(String regionName) {
        MarketStats stats;
        String marketName = regionToMarket.get(regionName);
        if (marketName != null && (stats = marketStats.get(marketName)) != null) {
            stats.rentedRegions = Math.max(0, stats.rentedRegions - 1);
            LOGGER.debug("Market '{}' occupancy: {}/{} ({:.1f}%)", new Object[]{marketName, stats.rentedRegions, stats.totalRegions, stats.getOccupancyRate() * 100.0});
        }
    }

    public static boolean addRegionToMarket(String marketName, String regionName) {
        if (!markets.containsKey(marketName)) {
            return false;
        }
        if (regionToMarket.containsKey(regionName)) {
            return false;
        }
        regionToMarket.put(regionName, marketName);
        MarketStats stats = marketStats.computeIfAbsent(marketName, k -> new MarketStats());
        ++stats.totalRegions;
        if (RegionRental.isRented(regionName)) {
            ++stats.rentedRegions;
        }
        AsyncDatabaseWriter.queueMarketRegionAdd(regionName, marketName);
        LOGGER.debug("Added region '{}' to market '{}'", (Object)regionName, (Object)marketName);
        return true;
    }

    public static boolean removeRegionFromMarket(String regionName) {
        String marketName = regionToMarket.remove(regionName);
        if (marketName == null) {
            return false;
        }
        MarketStats stats = marketStats.get(marketName);
        if (stats != null) {
            stats.totalRegions = Math.max(0, stats.totalRegions - 1);
            if (RegionRental.isRented(regionName)) {
                stats.rentedRegions = Math.max(0, stats.rentedRegions - 1);
            }
        }
        AsyncDatabaseWriter.queueMarketRegionRemove(regionName);
        LOGGER.debug("Removed region '{}' from market '{}'", (Object)regionName, (Object)marketName);
        return true;
    }

    public static void rebuildAllStats() {
        marketStats.clear();
        for (String marketName : markets.keySet()) {
            MarketStats stats = new MarketStats();
            Set<String> regions = MarketManager.getRegionsInMarket(marketName);
            stats.totalRegions = regions.size();
            for (String regionName : regions) {
                if (!RegionRental.isRented(regionName)) continue;
                ++stats.rentedRegions;
            }
            marketStats.put(marketName, stats);
        }
        LOGGER.debug("Rebuilt market statistics for {} markets", (Object)marketStats.size());
    }

    public static boolean updateMarketConfig(String marketName, int floorPrice, int ceilingPrice, double priceExponent) {
        MarketConfig market = markets.get(marketName);
        if (market == null) {
            return false;
        }
        if (floorPrice >= 0) {
            market.setFloorPrice(floorPrice);
        }
        if (ceilingPrice >= 0) {
            market.setCeilingPrice(ceilingPrice);
        }
        if (priceExponent >= 0.0) {
            market.setPriceExponent(priceExponent);
        }
        try {
            MarketDAO.saveMarket(market);
        }
        catch (SQLException e) {
            LOGGER.error("Failed to update market in database: {}", (Object)marketName, (Object)e);
            return false;
        }
        LOGGER.info("Updated market config: {}", (Object)market);
        return true;
    }

    public static boolean isMarketName(String name) {
        return markets.containsKey(name);
    }

    public static String getNextRegionName(String marketName) {
        if (!markets.containsKey(marketName)) {
            return null;
        }
        int highestNumber = 0;
        String prefix = marketName.toLowerCase() + " #";
        for (String regionName : RegionManager.getRegionNames()) {
            String lowerName = regionName.toLowerCase();
            if (!lowerName.startsWith(prefix)) continue;
            try {
                String numberPart = regionName.substring(prefix.length());
                int number = Integer.parseInt(numberPart);
                highestNumber = Math.max(highestNumber, number);
            }
            catch (NumberFormatException numberFormatException) {}
        }
        return marketName + " #" + (highestNumber + 1);
    }

    private static class MarketStats {
        int totalRegions = 0;
        int rentedRegions = 0;

        private MarketStats() {
        }

        double getOccupancyRate() {
            return this.totalRegions == 0 ? 0.0 : (double)this.rentedRegions / (double)this.totalRegions;
        }
    }
}

