/*
 * Decompiled with CFR 0.152.
 */
package world.landfall.landfallessentials.monitoring;

import java.util.ArrayDeque;
import java.util.Deque;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import net.minecraft.server.MinecraftServer;
import net.neoforged.bus.api.SubscribeEvent;
import net.neoforged.fml.common.EventBusSubscriber;
import net.neoforged.neoforge.event.tick.ServerTickEvent;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import world.landfall.landfallessentials.Config;
import world.landfall.landfallessentials.util.WebhookUtil;

@EventBusSubscriber(modid="landfallessentials")
public class TpsMonitor {
    private static final Logger LOGGER = LoggerFactory.getLogger(TpsMonitor.class);
    private static final Deque<Double> tpsHistory = new ArrayDeque<Double>();
    private static final ReentrantReadWriteLock historyLock = new ReentrantReadWriteLock();
    private static final int MAX_HISTORY_SIZE = 300;
    private static volatile long lastTpsCheck = 0L;
    private static volatile long lastAlertTime = 0L;
    private static volatile long lastTickTime = System.currentTimeMillis();
    private static volatile int tickCounter = 0;
    private static volatile double currentTps = 20.0;
    private static final double MIN_VALID_TPS = 0.0;
    private static final double MAX_VALID_TPS = 25.0;
    private static final long MAX_TIME_BETWEEN_TICKS = 60000L;

    @SubscribeEvent
    public static void onServerTick(ServerTickEvent.Pre event) {
        try {
            MinecraftServer server = event.getServer();
            if (server == null) {
                LOGGER.warn("Server tick event received with null server");
                return;
            }
            long currentTime = System.currentTimeMillis();
            if (currentTime < lastTickTime) {
                LOGGER.warn("System time went backwards, resetting TPS monitoring");
                TpsMonitor.reset();
                return;
            }
            if (currentTime - lastTickTime > 60000L) {
                LOGGER.warn("Large time gap detected ({}ms), resetting TPS calculation", (Object)(currentTime - lastTickTime));
                lastTickTime = currentTime;
                tickCounter = 0;
                return;
            }
            ++tickCounter;
            if (currentTime - lastTickTime >= 1000L) {
                TpsMonitor.calculateAndStoreTps(currentTime);
                if (TpsMonitor.shouldCheckAlerts(currentTime)) {
                    TpsMonitor.checkTpsAlerts(server);
                    lastTpsCheck = currentTime;
                }
            }
        }
        catch (Exception e) {
            LOGGER.error("Error during TPS monitoring tick", (Throwable)e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void calculateAndStoreTps(long currentTime) {
        try {
            double timeDelta = (double)(currentTime - lastTickTime) / 1000.0;
            if (timeDelta <= 0.0) {
                LOGGER.warn("Invalid time delta for TPS calculation: {}", (Object)timeDelta);
                return;
            }
            double secondTps = (double)tickCounter / timeDelta;
            if (!Double.isFinite(secondTps) || secondTps < 0.0 || secondTps > 25.0) {
                LOGGER.debug("Invalid TPS calculated: {}, using previous value", (Object)secondTps);
                secondTps = Math.max(0.0, Math.min(20.0, currentTps));
            }
            currentTps = secondTps;
            historyLock.writeLock().lock();
            try {
                tpsHistory.addLast(currentTps);
                if (tpsHistory.size() > 300) {
                    tpsHistory.removeFirst();
                }
            }
            finally {
                historyLock.writeLock().unlock();
            }
            tickCounter = 0;
            lastTickTime = currentTime;
        }
        catch (Exception e) {
            LOGGER.error("Error calculating TPS", (Throwable)e);
        }
    }

    private static boolean shouldCheckAlerts(long currentTime) {
        try {
            return Config.tpsCheckInterval > 0 && currentTime - lastTpsCheck >= (long)Config.tpsCheckInterval * 1000L;
        }
        catch (Exception e) {
            LOGGER.error("Error checking alert timing", (Throwable)e);
            return false;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void checkTpsAlerts(MinecraftServer server) {
        try {
            int historySize;
            if (Config.alertTpsThreshold <= 0.0 || !Config.webhookEnabled) {
                return;
            }
            if (Config.tpsAlertCooldown <= 0) {
                LOGGER.warn("Invalid TPS alert cooldown configuration: {}", (Object)Config.tpsAlertCooldown);
                return;
            }
            double averageTps = TpsMonitor.getAverageTps();
            if (!Double.isFinite(averageTps) || averageTps < 0.0) {
                LOGGER.warn("Invalid average TPS calculated: {}", (Object)averageTps);
                return;
            }
            historyLock.readLock().lock();
            try {
                historySize = tpsHistory.size();
            }
            finally {
                historyLock.readLock().unlock();
            }
            if (historySize < 60) {
                LOGGER.debug("Insufficient TPS history for alerts: {} samples", (Object)historySize);
                return;
            }
            if (averageTps < Config.alertTpsThreshold) {
                long currentTime = System.currentTimeMillis();
                long cooldownMs = (long)Config.tpsAlertCooldown * 60L * 1000L;
                if (cooldownMs < 0L) {
                    LOGGER.error("TPS alert cooldown overflow, using default");
                    cooldownMs = 600000L;
                }
                if (currentTime - lastAlertTime >= cooldownMs) {
                    TpsMonitor.sendTpsAlert(averageTps, currentTime);
                }
            }
        }
        catch (Exception e) {
            LOGGER.error("Error checking TPS alerts", (Throwable)e);
        }
    }

    private static void sendTpsAlert(double averageTps, long currentTime) {
        try {
            boolean isCritical = averageTps < Config.alertTpsThreshold * 0.7;
            LOGGER.warn("TPS Alert: Current={:.2f}, Average={:.2f}, Threshold={:.1f}, Critical={}", new Object[]{currentTps, averageTps, Config.alertTpsThreshold, isCritical});
            WebhookUtil.sendTpsAlert(currentTps, averageTps, isCritical);
            lastAlertTime = currentTime;
        }
        catch (Exception e) {
            LOGGER.error("Error sending TPS alert", (Throwable)e);
        }
    }

    public static double getCurrentTps() {
        return currentTps;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static double getAverageTps() {
        historyLock.readLock().lock();
        try {
            if (tpsHistory.isEmpty()) {
                double d = 20.0;
                return d;
            }
            double sum = 0.0;
            int count = 0;
            for (double tps : tpsHistory) {
                if (!Double.isFinite(tps) || !(tps >= 0.0)) continue;
                sum += tps;
                ++count;
            }
            double d = count > 0 ? sum / (double)count : 20.0;
            return d;
        }
        catch (Exception e) {
            LOGGER.error("Error calculating average TPS", (Throwable)e);
            double d = 20.0;
            return d;
        }
        finally {
            historyLock.readLock().unlock();
        }
    }

    public static double getMinTps() {
        historyLock.readLock().lock();
        try {
            if (tpsHistory.isEmpty()) {
                double d = 20.0;
                return d;
            }
            double d = tpsHistory.stream().mapToDouble(Double::doubleValue).filter(tps -> Double.isFinite(tps) && tps >= 0.0).min().orElse(20.0);
            return d;
        }
        catch (Exception e) {
            LOGGER.error("Error calculating minimum TPS", (Throwable)e);
            double d = 20.0;
            return d;
        }
        finally {
            historyLock.readLock().unlock();
        }
    }

    public static double getMaxTps() {
        historyLock.readLock().lock();
        try {
            if (tpsHistory.isEmpty()) {
                double d = 20.0;
                return d;
            }
            double d = tpsHistory.stream().mapToDouble(Double::doubleValue).filter(tps -> Double.isFinite(tps) && tps >= 0.0).max().orElse(20.0);
            return d;
        }
        catch (Exception e) {
            LOGGER.error("Error calculating maximum TPS", (Throwable)e);
            double d = 20.0;
            return d;
        }
        finally {
            historyLock.readLock().unlock();
        }
    }

    public static String getTpsStats() {
        try {
            int sampleCount;
            historyLock.readLock().lock();
            try {
                sampleCount = tpsHistory.size();
            }
            finally {
                historyLock.readLock().unlock();
            }
            return String.format("Current: %.2f | 5min Avg: %.2f | Min: %.2f | Max: %.2f | Samples: %d", TpsMonitor.getCurrentTps(), TpsMonitor.getAverageTps(), TpsMonitor.getMinTps(), TpsMonitor.getMaxTps(), sampleCount);
        }
        catch (Exception e) {
            LOGGER.error("Error generating TPS stats", (Throwable)e);
            return "TPS stats unavailable due to error";
        }
    }

    public static void reset() {
        try {
            historyLock.writeLock().lock();
            try {
                tpsHistory.clear();
            }
            finally {
                historyLock.writeLock().unlock();
            }
            lastTpsCheck = 0L;
            lastAlertTime = 0L;
            currentTps = 20.0;
            lastTickTime = System.currentTimeMillis();
            tickCounter = 0;
            LOGGER.info("TPS monitoring reset");
        }
        catch (Exception e) {
            LOGGER.error("Error resetting TPS monitoring", (Throwable)e);
        }
    }
}

