/*
 * Decompiled with CFR 0.152.
 */
package world.landfall.landfallessentials.regions;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import net.minecraft.network.chat.Component;
import net.minecraft.server.MinecraftServer;
import net.minecraft.server.level.ServerPlayer;
import net.minecraft.world.entity.player.Player;
import net.neoforged.bus.api.SubscribeEvent;
import net.neoforged.fml.common.EventBusSubscriber;
import net.neoforged.neoforge.event.tick.ServerTickEvent;
import net.neoforged.neoforge.server.ServerLifecycleHooks;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import world.landfall.landfallessentials.database.AsyncDatabaseWriter;
import world.landfall.landfallessentials.database.RentalDAO;
import world.landfall.landfallessentials.karma.KarmaSystem;
import world.landfall.landfallessentials.regions.Region;
import world.landfall.landfallessentials.regions.RegionManager;
import world.landfall.landfallessentials.regions.market.MarketConfig;
import world.landfall.landfallessentials.regions.market.MarketManager;
import world.landfall.landfallessentials.util.LuckPermsUtil;

@EventBusSubscriber(modid="landfallessentials")
public class RegionRental {
    private static final Logger LOGGER = LoggerFactory.getLogger(RegionRental.class);
    private static final Gson GSON = new GsonBuilder().setPrettyPrinting().create();
    private static final String RENTAL_FILE = "config/landfall_rentals.json";
    private static final long RENTAL_DURATION_DAYS = 7L;
    private static final int MAX_RENTAL_MULTIPLIER = 5;
    private static final int CLEANUP_INTERVAL_TICKS = 12000;
    private static long lastCleanupTime = 0L;
    private static final Map<String, RentalInfo> rentals = new ConcurrentHashMap<String, RentalInfo>();

    public static void initialize() {
        MarketManager.initialize();
        RegionRental.loadRentals();
        MarketManager.rebuildAllStats();
        LOGGER.info("Region rental system initialized with {} active rentals", (Object)rentals.size());
    }

    public static boolean isRentable(String regionName) {
        Region region = RegionManager.getRegion(regionName);
        if (region == null) {
            return false;
        }
        for (String tag : region.getTags()) {
            String[] parts;
            if (!tag.startsWith("rentable:") || (parts = tag.split(":", 3)).length < 2) continue;
            String priceOrMarket = parts[1];
            try {
                Integer.parseInt(priceOrMarket);
                return true;
            }
            catch (NumberFormatException e) {
                if (MarketManager.getMarket(priceOrMarket) != null) {
                    return true;
                }
                LOGGER.warn("Invalid rentable tag format in region {}: {} (not a price or valid market)", (Object)regionName, (Object)tag);
            }
        }
        return false;
    }

    public static int getRentalPrice(String regionName) {
        Region region = RegionManager.getRegion(regionName);
        if (region == null) {
            return -1;
        }
        for (String tag : region.getTags()) {
            String[] parts;
            if (!tag.startsWith("rentable:") || (parts = tag.split(":", 3)).length < 2) continue;
            String priceOrMarket = parts[1];
            try {
                return Integer.parseInt(priceOrMarket);
            }
            catch (NumberFormatException e) {
                MarketConfig market = MarketManager.getMarket(priceOrMarket);
                if (market != null) {
                    return MarketManager.getCurrentPrice(priceOrMarket);
                }
                LOGGER.warn("Invalid rentable tag in region {}: {} (not a price or valid market)", (Object)regionName, (Object)tag);
            }
        }
        return -1;
    }

    public static String getRentalPermission(String regionName) {
        Region region = RegionManager.getRegion(regionName);
        if (region == null) {
            return null;
        }
        for (String tag : region.getTags()) {
            String[] parts;
            if (!tag.startsWith("rentable:") || (parts = tag.split(":", 3)).length < 3 || parts[2].trim().isEmpty()) continue;
            return parts[2];
        }
        return null;
    }

    public static boolean isRented(String regionName) {
        RentalInfo rental = rentals.get(regionName);
        if (rental == null) {
            return false;
        }
        if (rental.isExpired()) {
            RegionRental.expireRental(regionName);
            return false;
        }
        return true;
    }

    public static RentalInfo getRental(String regionName) {
        RentalInfo rental = rentals.get(regionName);
        if (rental != null && rental.isExpired()) {
            RegionRental.expireRental(regionName);
            return null;
        }
        return rental;
    }

    public static String getOwnerFromTag(String regionName) {
        Region region = RegionManager.getRegion(regionName);
        if (region == null) {
            return null;
        }
        for (String tag : region.getTags()) {
            String[] parts;
            if (!tag.startsWith("owner:") || (parts = tag.split(":", 2)).length < 2 || parts[1].trim().isEmpty()) continue;
            return parts[1].trim();
        }
        return null;
    }

    public static boolean isOwner(String regionName, UUID playerUUID) {
        ServerPlayer player;
        RentalInfo rental = RegionRental.getRental(regionName);
        if (rental != null && rental.getRenterUUID().equals(playerUUID)) {
            return true;
        }
        String ownerUsername = RegionRental.getOwnerFromTag(regionName);
        return ownerUsername != null && (player = RegionRental.getPlayerByUsername(ownerUsername)) != null && player.getUUID().equals(playerUUID);
    }

    public static boolean isOwnerByUsername(String regionName, String username) {
        RentalInfo rental = RegionRental.getRental(regionName);
        if (rental != null && rental.getRenterName().equalsIgnoreCase(username)) {
            return true;
        }
        String ownerUsername = RegionRental.getOwnerFromTag(regionName);
        return ownerUsername != null && ownerUsername.equalsIgnoreCase(username);
    }

    private static ServerPlayer getPlayerByUsername(String username) {
        MinecraftServer server = ServerLifecycleHooks.getCurrentServer();
        if (server == null) {
            return null;
        }
        return server.getPlayerList().getPlayerByName(username);
    }

    public static RentResult rentRegion(String regionName, ServerPlayer player, int weeks) {
        RentalInfo existing;
        LOGGER.info("Player {} attempting to rent region {} for {} weeks", new Object[]{player.getName().getString(), regionName, weeks});
        if (weeks < 1 || weeks > 5) {
            LOGGER.warn("Invalid rental period: {} weeks for player {}", (Object)weeks, (Object)player.getName().getString());
            return new RentResult(false, "Rental period must be between 1 and 5 weeks.");
        }
        if (!RegionRental.isRentable(regionName)) {
            return new RentResult(false, "Region '" + regionName + "' is not available for rent.");
        }
        String requiredPermission = RegionRental.getRentalPermission(regionName);
        if (requiredPermission != null && !LuckPermsUtil.hasPermission(player.getUUID(), requiredPermission)) {
            return new RentResult(false, "You need the '" + requiredPermission + "' permission to rent this region.");
        }
        if (RegionRental.isRented(regionName) && (existing = RegionRental.getRental(regionName)) != null) {
            if (existing.getRenterUUID().equals(player.getUUID())) {
                return RegionRental.extendRental(regionName, player, weeks);
            }
            if (existing.isTrusted(player.getName().getString())) {
                return RegionRental.extendRental(regionName, player, weeks);
            }
            return new RentResult(false, "Region '" + regionName + "' is already rented by " + existing.getRenterName() + ".");
        }
        int basePrice = RegionRental.getRentalPrice(regionName);
        int totalCost = basePrice * weeks;
        if (!KarmaSystem.hasEnoughKarma((Player)player, totalCost)) {
            int currentKarma = KarmaSystem.getKarma((Player)player);
            return new RentResult(false, "Insufficient karma! You need " + totalCost + " karma but only have " + currentKarma + ".");
        }
        if (!KarmaSystem.spendKarma((Player)player, totalCost)) {
            return new RentResult(false, "Failed to deduct karma. Please try again.");
        }
        long expirationTime = System.currentTimeMillis() + (long)weeks * 7L * 24L * 60L * 60L * 1000L;
        RentalInfo rental = new RentalInfo(regionName, player.getUUID(), player.getName().getString(), expirationTime, totalCost);
        rentals.put(regionName, rental);
        RegionRental.grantRentalPermissions(regionName, player.getUUID());
        MarketManager.onRegionRented(regionName);
        AsyncDatabaseWriter.queueRentalSave(rental);
        Object duration = weeks == 1 ? "1 week" : weeks + " weeks";
        LOGGER.info("Player {} rented region {} for {} weeks (cost: {} karma)", new Object[]{player.getName().getString(), regionName, weeks, totalCost});
        return new RentResult(true, "Successfully rented region '" + regionName + "' for " + (String)duration + " (cost: " + totalCost + " karma).");
    }

    private static void grantRentalPermissions(String regionName, UUID playerUUID) {
        LOGGER.debug("Granted rental permissions for region {} to player {}", (Object)regionName, (Object)playerUUID);
    }

    private static void expireRental(String regionName) {
        RentalInfo rental = rentals.remove(regionName);
        if (rental != null) {
            LOGGER.info("Rental expired for region {} (was rented by {})", (Object)regionName, (Object)rental.getRenterName());
            MarketManager.onRegionUnrented(regionName);
            AsyncDatabaseWriter.queueRentalDelete(regionName);
        }
    }

    public static String getRentalStatusMessage(String regionName) {
        if (!RegionRental.isRentable(regionName)) {
            return null;
        }
        RentalInfo rental = RegionRental.getRental(regionName);
        if (rental != null) {
            long daysRemaining = rental.getDaysRemaining();
            long hoursRemaining = rental.getHoursRemaining();
            if (daysRemaining > 0L) {
                return "\u00a76Owned by \u00a7e" + rental.getRenterName() + "\u00a76 (" + daysRemaining + " days remaining)";
            }
            if (hoursRemaining > 0L) {
                return "\u00a76Owned by \u00a7e" + rental.getRenterName() + "\u00a76 (" + hoursRemaining + " hours remaining)";
            }
            return "\u00a76Owned by \u00a7e" + rental.getRenterName() + "\u00a76 (expires soon)";
        }
        int price = RegionRental.getRentalPrice(regionName);
        MarketConfig market = MarketManager.getMarketForRegion(regionName);
        if (market != null) {
            double occupancy = MarketManager.getOccupancyRate(market.getName()) * 100.0;
            return String.format("\u00a7aAvailable: \u00a7e%d karma/week \u00a77(%s: %.0f%% occupied)", price, market.getName(), occupancy);
        }
        return "\u00a7aAvailable for rent: \u00a7e" + price + " karma/week \u00a77(/rent)";
    }

    public static void cleanupExpiredRentals() {
        int cleaned = 0;
        for (String regionName : rentals.keySet()) {
            RentalInfo rental = rentals.get(regionName);
            if (rental == null || !rental.isExpired()) continue;
            RegionRental.expireRental(regionName);
            ++cleaned;
        }
        if (cleaned > 0) {
            LOGGER.info("Cleaned up {} expired rentals", (Object)cleaned);
        }
    }

    public static Map<String, RentalInfo> getPlayerRentals(UUID playerUUID) {
        ConcurrentHashMap<String, RentalInfo> playerRentals = new ConcurrentHashMap<String, RentalInfo>();
        for (Map.Entry<String, RentalInfo> entry : rentals.entrySet()) {
            RentalInfo rental = entry.getValue();
            if (!rental.getRenterUUID().equals(playerUUID) || rental.isExpired()) continue;
            playerRentals.put(entry.getKey(), rental);
        }
        return playerRentals;
    }

    private static void loadRentals() {
        try {
            Map<String, RentalInfo> loaded = RentalDAO.loadAllRentals();
            if (loaded != null && !loaded.isEmpty()) {
                rentals.clear();
                rentals.putAll(loaded);
                LOGGER.info("Loaded {} rentals from database", (Object)rentals.size());
            } else {
                LOGGER.info("No rentals found in database, starting with empty rentals");
            }
        }
        catch (SQLException e) {
            LOGGER.error("Failed to load rentals from database", (Throwable)e);
        }
    }

    private static void saveRentals() {
        for (RentalInfo rental : rentals.values()) {
            AsyncDatabaseWriter.queueRentalSave(rental);
        }
    }

    private static RentResult extendRental(String regionName, ServerPlayer player, int weeks) {
        Object duration;
        long maxAllowedTime;
        long extensionTime;
        RentalInfo rental = RegionRental.getRental(regionName);
        if (rental == null) {
            return new RentResult(false, "Region '" + regionName + "' is not rented.");
        }
        boolean isOwner = rental.getRenterUUID().equals(player.getUUID());
        boolean isTrusted = rental.isTrusted(player.getName().getString());
        if (!isOwner && !isTrusted) {
            return new RentResult(false, "You don't have permission to extend this rental.");
        }
        int basePrice = RegionRental.getRentalPrice(regionName);
        int totalCost = basePrice * weeks;
        long currentExpiration = rental.getExpirationTime();
        if (currentExpiration + (extensionTime = (long)weeks * 7L * 24L * 60L * 60L * 1000L) > (maxAllowedTime = System.currentTimeMillis() + 3024000000L)) {
            return new RentResult(false, "Cannot extend rental beyond maximum duration of 5 weeks total.");
        }
        if (!KarmaSystem.hasEnoughKarma((Player)player, totalCost)) {
            int currentKarma = KarmaSystem.getKarma((Player)player);
            return new RentResult(false, "Insufficient karma! You need " + totalCost + " karma but only have " + currentKarma + ".");
        }
        if (!KarmaSystem.spendKarma((Player)player, totalCost)) {
            return new RentResult(false, "Failed to deduct karma. Please try again.");
        }
        rental.expirationTime = currentExpiration + extensionTime;
        AsyncDatabaseWriter.queueRentalSave(rental);
        Object object = duration = weeks == 1 ? "1 week" : weeks + " weeks";
        if (isTrusted && !isOwner) {
            LOGGER.info("Trusted player {} extended rental for region {} (owned by {}) by {} weeks (cost: {} karma)", new Object[]{player.getName().getString(), regionName, rental.getRenterName(), weeks, totalCost});
            RegionRental.notifyOwner(rental, player.getName().getString() + " extended your rental for region '" + regionName + "' by " + (String)duration + ".");
            return new RentResult(true, "Extended rental for region '" + regionName + "' (owned by " + rental.getRenterName() + ") by " + (String)duration + " (cost: " + totalCost + " karma).");
        }
        LOGGER.info("Player {} extended rental for region {} by {} weeks (cost: {} karma)", new Object[]{player.getName().getString(), regionName, weeks, totalCost});
        return new RentResult(true, "Extended rental for region '" + regionName + "' by " + (String)duration + " (cost: " + totalCost + " karma).");
    }

    private static void notifyOwner(RentalInfo rental, String message) {
        try {
            ServerPlayer owner;
            MinecraftServer server = ServerLifecycleHooks.getCurrentServer();
            if (server != null && (owner = server.getPlayerList().getPlayer(rental.getRenterUUID())) != null) {
                owner.sendSystemMessage((Component)Component.literal((String)("\u00a7a[Rental] \u00a7e" + message)));
            }
        }
        catch (Exception e) {
            LOGGER.debug("Could not notify owner {}: {}", (Object)rental.getRenterName(), (Object)e.getMessage());
        }
    }

    public static TrustResult trustPlayer(String regionName, UUID ownerUUID, String playerName) {
        Region region = RegionManager.getRegion(regionName);
        if (region == null) {
            return new TrustResult(false, "Region '" + regionName + "' does not exist.");
        }
        RentalInfo rental = RegionRental.getRental(regionName);
        if (rental != null) {
            if (!rental.getRenterUUID().equals(ownerUUID)) {
                return new TrustResult(false, "You don't own the rental for region '" + regionName + "'.");
            }
            if (rental.isTrusted(playerName)) {
                return new TrustResult(false, "Player '" + playerName + "' is already trusted in region '" + regionName + "'.");
            }
            rental.addTrustedPlayer(playerName);
            AsyncDatabaseWriter.queueRentalSave(rental);
            LOGGER.info("Player {} trusted {} in rented region {}", new Object[]{rental.getRenterName(), playerName, regionName});
            return new TrustResult(true, "Successfully trusted player '" + playerName + "' in region '" + regionName + "'.");
        }
        if (!RegionRental.isOwner(regionName, ownerUUID)) {
            return new TrustResult(false, "You don't own region '" + regionName + "'.");
        }
        String trustTag = "trusted:" + playerName.toLowerCase();
        if (region.hasTag(trustTag)) {
            return new TrustResult(false, "Player '" + playerName + "' is already trusted in region '" + regionName + "'.");
        }
        region.addTag(trustTag);
        AsyncDatabaseWriter.queueRegionSave(region);
        LOGGER.info("Player {} trusted in owned region {}", (Object)playerName, (Object)regionName);
        return new TrustResult(true, "Successfully trusted player '" + playerName + "' in region '" + regionName + "'.");
    }

    public static TrustResult untrustPlayer(String regionName, UUID ownerUUID, String playerName) {
        Region region = RegionManager.getRegion(regionName);
        if (region == null) {
            return new TrustResult(false, "Region '" + regionName + "' does not exist.");
        }
        RentalInfo rental = RegionRental.getRental(regionName);
        if (rental != null) {
            if (!rental.getRenterUUID().equals(ownerUUID)) {
                return new TrustResult(false, "You don't own the rental for region '" + regionName + "'.");
            }
            if (!rental.isTrusted(playerName)) {
                return new TrustResult(false, "Player '" + playerName + "' is not trusted in region '" + regionName + "'.");
            }
            rental.removeTrustedPlayer(playerName);
            AsyncDatabaseWriter.queueRentalSave(rental);
            LOGGER.info("Player {} untrusted {} in rented region {}", new Object[]{rental.getRenterName(), playerName, regionName});
            return new TrustResult(true, "Successfully untrusted player '" + playerName + "' from region '" + regionName + "'.");
        }
        if (!RegionRental.isOwner(regionName, ownerUUID)) {
            return new TrustResult(false, "You don't own region '" + regionName + "'.");
        }
        String trustTag = "trusted:" + playerName.toLowerCase();
        if (!region.hasTag(trustTag)) {
            return new TrustResult(false, "Player '" + playerName + "' is not trusted in region '" + regionName + "'.");
        }
        region.removeTag(trustTag);
        AsyncDatabaseWriter.queueRegionSave(region);
        LOGGER.info("Player {} untrusted from owned region {}", (Object)playerName, (Object)regionName);
        return new TrustResult(true, "Successfully untrusted player '" + playerName + "' from region '" + regionName + "'.");
    }

    public static AdminResult adminUnrentRegion(String regionName) {
        RentalInfo rental = rentals.remove(regionName);
        if (rental != null) {
            MarketManager.onRegionUnrented(regionName);
            AsyncDatabaseWriter.queueRentalDelete(regionName);
            LOGGER.info("Admin force-unrented region {} (was rented by {})", (Object)regionName, (Object)rental.getRenterName());
            return new AdminResult(true, "Successfully force-unrented region '" + regionName + "' (was rented by " + rental.getRenterName() + ").");
        }
        return new AdminResult(false, "Region '" + regionName + "' is not currently rented.");
    }

    public static List<String> getAllRentedRegions() {
        return new ArrayList<String>(rentals.keySet());
    }

    public static boolean isTrustedInRegion(String regionName, String playerName) {
        String trustTag;
        RentalInfo rental = RegionRental.getRental(regionName);
        if (rental != null && rental.isTrusted(playerName)) {
            return true;
        }
        Region region = RegionManager.getRegion(regionName);
        return region != null && region.hasTag(trustTag = "trusted:" + playerName.toLowerCase());
    }

    @SubscribeEvent
    public static void onServerTick(ServerTickEvent.Post event) {
        long currentTime = event.getServer().getTickCount();
        if (currentTime - lastCleanupTime >= 12000L) {
            RegionRental.cleanupExpiredRentals();
            lastCleanupTime = currentTime;
        }
    }

    public static class RentalInfo {
        private String regionName;
        private UUID renterUUID;
        private String renterName;
        private long expirationTime;
        private int karmaPrice;
        private long rentedAt;
        private Set<String> trustedPlayers = new HashSet<String>();

        public RentalInfo() {
        }

        public RentalInfo(String regionName, UUID renterUUID, String renterName, long expirationTime, int karmaPrice) {
            this.regionName = regionName;
            this.renterUUID = renterUUID;
            this.renterName = renterName;
            this.expirationTime = expirationTime;
            this.karmaPrice = karmaPrice;
            this.rentedAt = System.currentTimeMillis();
        }

        public String getRegionName() {
            return this.regionName;
        }

        public UUID getRenterUUID() {
            return this.renterUUID;
        }

        public String getRenterName() {
            return this.renterName;
        }

        public long getExpirationTime() {
            return this.expirationTime;
        }

        public int getKarmaPrice() {
            return this.karmaPrice;
        }

        public long getRentedAt() {
            return this.rentedAt;
        }

        public boolean isExpired() {
            return System.currentTimeMillis() > this.expirationTime;
        }

        public long getDaysRemaining() {
            long remaining = this.expirationTime - System.currentTimeMillis();
            return Math.max(0L, remaining / 86400000L);
        }

        public long getHoursRemaining() {
            long remaining = this.expirationTime - System.currentTimeMillis();
            return Math.max(0L, remaining / 3600000L);
        }

        public Set<String> getTrustedPlayers() {
            return new HashSet<String>(this.trustedPlayers);
        }

        public boolean isTrusted(String playerName) {
            return this.trustedPlayers.contains(playerName);
        }

        public boolean addTrustedPlayer(String playerName) {
            return this.trustedPlayers.add(playerName);
        }

        public boolean removeTrustedPlayer(String playerName) {
            return this.trustedPlayers.remove(playerName);
        }
    }

    public static class RentResult {
        private final boolean success;
        private final String message;

        public RentResult(boolean success, String message) {
            this.success = success;
            this.message = message;
        }

        public boolean isSuccess() {
            return this.success;
        }

        public String getMessage() {
            return this.message;
        }
    }

    public static class TrustResult {
        private final boolean success;
        private final String message;

        public TrustResult(boolean success, String message) {
            this.success = success;
            this.message = message;
        }

        public boolean isSuccess() {
            return this.success;
        }

        public String getMessage() {
            return this.message;
        }
    }

    public static class AdminResult {
        private final boolean success;
        private final String message;

        public AdminResult(boolean success, String message) {
            this.success = success;
            this.message = message;
        }

        public boolean isSuccess() {
            return this.success;
        }

        public String getMessage() {
            return this.message;
        }
    }
}

