/*
 * Decompiled with CFR 0.152.
 */
package world.landfall.landfallessentials.util;

import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.time.Duration;
import java.time.Instant;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicLong;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import world.landfall.landfallessentials.Config;

public class WebhookUtil {
    private static final Logger LOGGER = LoggerFactory.getLogger(WebhookUtil.class);
    private static final HttpClient HTTP_CLIENT = HttpClient.newBuilder().connectTimeout(Duration.ofSeconds(10L)).build();
    private static final ConcurrentHashMap<String, AtomicLong> rateLimitMap = new ConcurrentHashMap();
    private static final long RATE_LIMIT_WINDOW_MS = 60000L;
    private static final int MAX_REQUESTS_PER_WINDOW = 10;
    private static final int MAX_STRING_LENGTH = 2000;
    private static final int MAX_FIELD_COUNT = 25;

    public static void sendConduitDestructionAlert(String conduitName, String conduitType, double x, double y, double z, String dimension, String reason) {
        try {
            double finalX;
            double finalY;
            double finalZ;
            if (!WebhookUtil.isValidConfiguration()) {
                return;
            }
            if (!Config.alertConduitDestruction) {
                return;
            }
            String safeName = WebhookUtil.sanitizeString(conduitName, "Unknown Conduit");
            String safeType = WebhookUtil.sanitizeString(conduitType, "UNKNOWN");
            String safeDimension = WebhookUtil.sanitizeString(dimension, "unknown");
            String safeReason = WebhookUtil.sanitizeString(reason, "unknown");
            if (!WebhookUtil.areValidCoordinates(x, y, z)) {
                LOGGER.warn("Invalid coordinates for conduit destruction: {}, {}, {}", new Object[]{x, y, z});
                finalZ = 0.0;
                finalY = 0.0;
                finalX = 0.0;
            } else {
                finalX = x;
                finalY = y;
                finalZ = z;
            }
            if (!WebhookUtil.checkRateLimit("conduit_destruction")) {
                LOGGER.warn("Rate limit exceeded for conduit destruction alerts");
                return;
            }
            CompletableFuture.runAsync(() -> {
                try {
                    WebhookUtil.sendConduitAlert(safeName, safeType, finalX, finalY, finalZ, safeDimension, safeReason);
                }
                catch (Exception e) {
                    LOGGER.error("Failed to send conduit destruction webhook notification", (Throwable)e);
                }
            }).exceptionally(throwable -> {
                LOGGER.error("Async webhook execution failed", throwable);
                return null;
            });
        }
        catch (Exception e) {
            LOGGER.error("Error preparing conduit destruction alert", (Throwable)e);
        }
    }

    public static void sendTpsAlert(double currentTps, double averageTps, boolean isCritical) {
        try {
            String rateLimitKey;
            if (!WebhookUtil.isValidConfiguration() || Config.alertTpsThreshold <= 0.0) {
                return;
            }
            if (!WebhookUtil.isValidTps(currentTps) || !WebhookUtil.isValidTps(averageTps)) {
                LOGGER.warn("Invalid TPS values: current={}, average={}", (Object)currentTps, (Object)averageTps);
                return;
            }
            String string = rateLimitKey = isCritical ? "tps_critical" : "tps_warning";
            if (!WebhookUtil.checkRateLimit(rateLimitKey)) {
                LOGGER.warn("Rate limit exceeded for TPS alerts");
                return;
            }
            CompletableFuture.runAsync(() -> {
                try {
                    WebhookUtil.sendTpsAlertMessage(currentTps, averageTps, isCritical);
                }
                catch (Exception e) {
                    LOGGER.error("Failed to send TPS webhook notification", (Throwable)e);
                }
            }).exceptionally(throwable -> {
                LOGGER.error("Async TPS webhook execution failed", throwable);
                return null;
            });
        }
        catch (Exception e) {
            LOGGER.error("Error preparing TPS alert", (Throwable)e);
        }
    }

    public static void sendSystemAlert(AlertType alertType, String title, String description, String ... additionalFields) {
        try {
            if (!WebhookUtil.isValidConfiguration() || alertType == null) {
                return;
            }
            if (additionalFields.length > 50) {
                LOGGER.warn("Too many additional fields provided: {}", (Object)additionalFields.length);
                return;
            }
            String safeTitle = WebhookUtil.sanitizeString(title, alertType.getTitle());
            String safeDescription = WebhookUtil.sanitizeString(description, "System alert");
            String[] safeFields = new String[additionalFields.length];
            for (int i = 0; i < additionalFields.length; ++i) {
                safeFields[i] = WebhookUtil.sanitizeString(additionalFields[i], "");
            }
            if (!WebhookUtil.checkRateLimit("system_alert")) {
                LOGGER.warn("Rate limit exceeded for system alerts");
                return;
            }
            CompletableFuture.runAsync(() -> {
                try {
                    WebhookUtil.sendGenericAlert(alertType, safeTitle, safeDescription, safeFields);
                }
                catch (Exception e) {
                    LOGGER.error("Failed to send system webhook notification", (Throwable)e);
                }
            }).exceptionally(throwable -> {
                LOGGER.error("Async system webhook execution failed", throwable);
                return null;
            });
        }
        catch (Exception e) {
            LOGGER.error("Error preparing system alert", (Throwable)e);
        }
    }

    private static boolean isValidConfiguration() {
        return Config.webhookEnabled && Config.webhookUrl != null && !Config.webhookUrl.trim().isEmpty() && WebhookUtil.isValidUrl(Config.webhookUrl);
    }

    private static boolean isValidUrl(String url) {
        try {
            URI uri = new URI(url);
            return uri.getScheme() != null && (uri.getScheme().equals("http") || uri.getScheme().equals("https")) && uri.getHost() != null;
        }
        catch (URISyntaxException e) {
            LOGGER.error("Invalid webhook URL: {}", (Object)url, (Object)e);
            return false;
        }
    }

    private static boolean areValidCoordinates(double x, double y, double z) {
        return Double.isFinite(x) && Double.isFinite(y) && Double.isFinite(z) && Math.abs(x) < 3.0E7 && Math.abs(y) < 3.0E7 && Math.abs(z) < 3.0E7;
    }

    private static boolean isValidTps(double tps) {
        return Double.isFinite(tps) && tps >= 0.0 && tps <= 25.0;
    }

    private static String sanitizeString(String input, String defaultValue) {
        if (input == null || input.trim().isEmpty()) {
            return defaultValue;
        }
        Object sanitized = input.trim();
        if (((String)sanitized).length() > 2000) {
            sanitized = ((String)sanitized).substring(0, 1997) + "...";
        }
        sanitized = ((String)sanitized).replaceAll("[\\r\\n\\t]", " ");
        sanitized = ((String)sanitized).replaceAll("\\p{Cntrl}", "");
        sanitized = ((String)sanitized).replace("*", "\\*").replace("_", "\\_").replace("`", "\\`").replace("~", "\\~");
        return sanitized;
    }

    private static boolean checkRateLimit(String key) {
        long lastRequest;
        long currentTime = System.currentTimeMillis();
        AtomicLong counter = rateLimitMap.computeIfAbsent(key, k -> new AtomicLong(0L));
        if (currentTime % 60000L < 1000L) {
            rateLimitMap.entrySet().removeIf(entry -> currentTime - ((AtomicLong)entry.getValue()).get() > 60000L);
        }
        if (currentTime - (lastRequest = counter.get()) < 6000L) {
            return false;
        }
        counter.set(currentTime);
        return true;
    }

    private static void sendConduitAlert(String conduitName, String conduitType, double x, double y, double z, String dimension, String reason) throws IOException, InterruptedException {
        JsonObject embed = new JsonObject();
        embed.addProperty("title", AlertType.CONDUIT_DESTRUCTION.emoji + " " + AlertType.CONDUIT_DESTRUCTION.title);
        embed.addProperty("description", String.format("**%s** (%s) has been destroyed!", conduitName, conduitType));
        embed.addProperty("color", (Number)AlertType.CONDUIT_DESTRUCTION.color);
        JsonObject locationField = new JsonObject();
        locationField.addProperty("name", "Location");
        locationField.addProperty("value", String.format("%.1f, %.1f, %.1f", x, y, z));
        locationField.addProperty("inline", Boolean.valueOf(true));
        JsonObject dimensionField = new JsonObject();
        dimensionField.addProperty("name", "Dimension");
        dimensionField.addProperty("value", dimension);
        dimensionField.addProperty("inline", Boolean.valueOf(true));
        JsonObject reasonField = new JsonObject();
        reasonField.addProperty("name", "Reason");
        reasonField.addProperty("value", reason);
        reasonField.addProperty("inline", Boolean.valueOf(true));
        embed.addProperty("timestamp", Instant.now().toString());
        JsonArray fields = new JsonArray();
        fields.add((JsonElement)locationField);
        fields.add((JsonElement)dimensionField);
        fields.add((JsonElement)reasonField);
        embed.add("fields", (JsonElement)fields);
        WebhookUtil.sendWebhookMessage("\ud83d\udea8 **CONDUIT ALERT** \ud83d\udea8", embed);
    }

    private static void sendTpsAlertMessage(double currentTps, double averageTps, boolean isCritical) throws IOException, InterruptedException {
        AlertType alertType = isCritical ? AlertType.TPS_CRITICAL : AlertType.TPS_WARNING;
        JsonObject embed = new JsonObject();
        embed.addProperty("title", alertType.emoji + " " + alertType.title);
        embed.addProperty("description", String.format("Server performance is %s the configured threshold!", isCritical ? "critically below" : "below"));
        embed.addProperty("color", (Number)alertType.color);
        JsonObject currentTpsField = new JsonObject();
        currentTpsField.addProperty("name", "Current TPS");
        currentTpsField.addProperty("value", String.format("%.2f", currentTps));
        currentTpsField.addProperty("inline", Boolean.valueOf(true));
        JsonObject averageTpsField = new JsonObject();
        averageTpsField.addProperty("name", "5-Minute Average");
        averageTpsField.addProperty("value", String.format("%.2f", averageTps));
        averageTpsField.addProperty("inline", Boolean.valueOf(true));
        JsonObject thresholdField = new JsonObject();
        thresholdField.addProperty("name", "Threshold");
        thresholdField.addProperty("value", String.format("%.1f", Config.alertTpsThreshold));
        thresholdField.addProperty("inline", Boolean.valueOf(true));
        embed.addProperty("timestamp", Instant.now().toString());
        JsonArray fields = new JsonArray();
        fields.add((JsonElement)currentTpsField);
        fields.add((JsonElement)averageTpsField);
        fields.add((JsonElement)thresholdField);
        embed.add("fields", (JsonElement)fields);
        String content = isCritical ? "\ud83d\udea8 **CRITICAL TPS ALERT** \ud83d\udea8" : "\u26a0\ufe0f **TPS WARNING** \u26a0\ufe0f";
        WebhookUtil.sendWebhookMessage(content, embed);
    }

    private static void sendGenericAlert(AlertType alertType, String title, String description, String ... additionalFields) throws IOException, InterruptedException {
        JsonObject embed = new JsonObject();
        embed.addProperty("title", alertType.emoji + " " + (title != null ? title : alertType.title));
        embed.addProperty("description", description);
        embed.addProperty("color", (Number)alertType.color);
        embed.addProperty("timestamp", Instant.now().toString());
        if (additionalFields.length > 0 && additionalFields.length % 2 == 0) {
            JsonArray fields = new JsonArray();
            for (int i = 0; i < additionalFields.length; i += 2) {
                JsonObject field = new JsonObject();
                field.addProperty("name", additionalFields[i]);
                field.addProperty("value", additionalFields[i + 1]);
                field.addProperty("inline", Boolean.valueOf(true));
                fields.add((JsonElement)field);
            }
            embed.add("fields", (JsonElement)fields);
        }
        WebhookUtil.sendWebhookMessage("\ud83d\udea8 **SYSTEM ALERT** \ud83d\udea8", embed);
    }

    private static void sendWebhookMessage(String content, JsonObject embed) throws IOException, InterruptedException {
        JsonArray embeds = new JsonArray();
        embeds.add((JsonElement)embed);
        JsonObject payload = new JsonObject();
        payload.addProperty("username", Config.webhookUsername);
        payload.addProperty("content", content);
        payload.add("embeds", (JsonElement)embeds);
        HttpRequest request = HttpRequest.newBuilder().uri(URI.create(Config.webhookUrl)).header("Content-Type", "application/json").header("User-Agent", "Landfall-Essentials-Webhook/1.0").timeout(Duration.ofSeconds(30L)).POST(HttpRequest.BodyPublishers.ofString(payload.toString())).build();
        HttpResponse<String> response = HTTP_CLIENT.send(request, HttpResponse.BodyHandlers.ofString());
        if (response.statusCode() >= 200 && response.statusCode() < 300) {
            LOGGER.debug("Successfully sent webhook notification");
        } else {
            LOGGER.warn("Webhook request failed with status code: {} - Response: {}", (Object)response.statusCode(), (Object)response.body());
        }
    }

    public static void sendTestMessage() {
        if (!Config.webhookEnabled || Config.webhookUrl == null || Config.webhookUrl.trim().isEmpty()) {
            LOGGER.warn("Webhook is not configured or disabled");
            return;
        }
        CompletableFuture.runAsync(() -> {
            try {
                JsonObject payload = new JsonObject();
                payload.addProperty("username", Config.webhookUsername);
                payload.addProperty("content", "\u2705 Webhook test successful! Critical system alerts are configured correctly.");
                HttpRequest request = HttpRequest.newBuilder().uri(URI.create(Config.webhookUrl)).header("Content-Type", "application/json").header("User-Agent", "Landfall-Essentials-Webhook/1.0").timeout(Duration.ofSeconds(30L)).POST(HttpRequest.BodyPublishers.ofString(payload.toString())).build();
                HttpResponse<String> response = HTTP_CLIENT.send(request, HttpResponse.BodyHandlers.ofString());
                if (response.statusCode() >= 200 && response.statusCode() < 300) {
                    LOGGER.info("Webhook test message sent successfully");
                } else {
                    LOGGER.error("Webhook test failed with status code: {} - Response: {}", (Object)response.statusCode(), (Object)response.body());
                }
            }
            catch (Exception e) {
                LOGGER.error("Failed to send webhook test message", (Throwable)e);
            }
        });
    }

    public static enum AlertType {
        CONDUIT_DESTRUCTION("\u26a0\ufe0f", "Conduit Destroyed", 0xFF0000),
        TPS_WARNING("\ud83d\udc0c", "Low TPS Warning", 0xFFFF00),
        TPS_CRITICAL("\ud83d\udea8", "Critical TPS Alert", 0xFF0000),
        SYSTEM_ERROR("\ud83d\udca5", "System Error", 0xFF0000);

        private final String emoji;
        private final String title;
        private final int color;

        private AlertType(String emoji, String title, int color) {
            this.emoji = emoji;
            this.title = title;
            this.color = color;
        }

        public String getEmoji() {
            return this.emoji;
        }

        public String getTitle() {
            return this.title;
        }

        public int getColor() {
            return this.color;
        }
    }
}

