package net.cookedseafood.pentamana.mana;

import java.util.Objects;
import java.util.UUID;
import net.cookedseafood.inferiordata.component.CustomStatusEffectManagerComponentInstance;
import net.cookedseafood.inferiordata.effect.CustomStatusEffectManager;
import net.cookedseafood.pentamana.Pentamana;
import net.cookedseafood.pentamana.api.event.ConsumManaCallback;
import net.cookedseafood.pentamana.api.event.RegenManaCallback;
import net.cookedseafood.pentamana.api.event.TickManaCallback;
import net.cookedseafood.pentamana.attribute.PentamanaAttributeIdentifiers;
import net.cookedseafood.pentamana.effect.PentamanaStatusEffectIdentifiers;
import net.cookedseafood.pentamana.enchantment.PentamanaEnchantmentIdentifiers;
import net.minecraft.class_1259;
import net.minecraft.class_2487;
import net.minecraft.class_2561;
import net.minecraft.class_266;
import net.minecraft.class_2736;
import net.minecraft.class_274;
import net.minecraft.class_2751;
import net.minecraft.class_2960;
import net.minecraft.class_3002;
import net.minecraft.class_3004;
import net.minecraft.class_3222;
import net.minecraft.class_7225;
import net.minecraft.class_8646;
import net.minecraft.server.MinecraftServer;

/**
 * A tickable manabar, with owner player.
 * 
 * <p>Contains an extra manabar which represents the manabar in client
 * to help reduce network usage.</p>
 */
public final class ServerManaBar extends ManaBar {
    private MinecraftServer server;
    private class_3222 player;
    private UUID uuid;
    private String name;
    private class_2960 id;
    private ManaBar clientManaBar;
    private byte life;

    public ServerManaBar(MinecraftServer server, class_3222 player, UUID uuid, String name, class_2960 id, float capacity, float supply, ManaBar.Position position, ManaTextual textual, boolean isVisible, class_1259.class_1260 color, class_1259.class_1261 style) {
        super(capacity, supply, position, textual, isVisible, color, style);
        this.server = server;
        this.player = player;
        this.uuid = uuid;
        this.name = name;
        this.id = id;
        this.clientManaBar = new ManaBar(capacity, supply, position, textual, isVisible, color, style);
    }

    /**
     * Shadow copy {@code manaBar}'s value.
     * 
     * @param manaBar
     * @return a new ServerManaBar
     */
    public static ServerManaBar of(ManaBar manaBar) {
        return new ServerManaBar(null, null, null, null, null, manaBar.getCapacity(), manaBar.getSupply(), manaBar.getPosition(), manaBar.getTextual(), manaBar.isVisible(), manaBar.getColor(), manaBar.getStyle());
    }

    public void tick(class_3222 player) {
        if (this.player == null) {
            this.player = player;
            this.uuid = this.player.method_5667();
            this.name = Pentamana.MANA_BAR_NAME_PREFIX + this.uuid.toString();
            this.id = class_2960.method_60655(Pentamana.MOD_ID, this.name);
        }

        CustomStatusEffectManager statusEffectManager = CustomStatusEffectManagerComponentInstance.CUSTOM_STATUS_EFFECT_MANAGER.get(this.player).getStatusEffectManager();

        float capacity = (float)this.player.getCustomModifiedValue(PentamanaAttributeIdentifiers.MANA_CAPACITY, Pentamana.manaCapacityBase);
        capacity += Pentamana.enchantmentCapacityBase * this.player.method_59958().method_58657().getLevel(PentamanaEnchantmentIdentifiers.CAPACITY);
        capacity += statusEffectManager.containsKey(PentamanaStatusEffectIdentifiers.MANA_BOOST) ? Pentamana.statusEffectManaBoostBase * (statusEffectManager.getActiveAmplifier(PentamanaStatusEffectIdentifiers.MANA_BOOST) + 1) : 0;
        capacity -= statusEffectManager.containsKey(PentamanaStatusEffectIdentifiers.MANA_REDUCTION) ? Pentamana.statusEffectManaReductionBase * (statusEffectManager.getActiveAmplifier(PentamanaStatusEffectIdentifiers.MANA_REDUCTION) + 1) : 0;
        capacity += Pentamana.isConversionExperienceLevel ? Pentamana.conversionExperienceLevelBase * this.player.field_7520 : 0;
        capacity = Math.max(capacity, 0.0f);

        TickManaCallback.EVENT.invoker().interact(this.player);

        this.capacity = capacity;

        if (this.supply == this.capacity) {
        } else if (this.supply < this.capacity && this.supply >= 0.0f) {
            this.regen();
        } else if (this.supply > this.capacity) {
            this.supply = this.capacity;
        } else if (this.supply < 0) {
            this.supply = 0.0f;
        };

        this.tickClient();
    }

    /**
     * Consume the {@link Pentamana#manaRegenerationBase} amount of mana after the custom modifiers
     * and enchantments are applied.
     * 
     * <p>Target supply is capped at capacity and 0.0f.
     * 
     * @return {@link #isFull()}.
     * 
     * @see #regen(float)
     */
    public boolean regen() {
        CustomStatusEffectManager statusEffectManager = CustomStatusEffectManagerComponentInstance.CUSTOM_STATUS_EFFECT_MANAGER.get(this.player).getStatusEffectManager();

        float regen = (float)this.player.getCustomModifiedValue(PentamanaAttributeIdentifiers.MANA_REGENERATION, Pentamana.manaRegenerationBase);
        regen += Pentamana.enchantmentStreamBase * this.player.method_59958().method_58657().getLevel(PentamanaEnchantmentIdentifiers.STREAM);
        regen += statusEffectManager.containsKey(PentamanaStatusEffectIdentifiers.INSTANT_MANA) ? Pentamana.statusEffectInstantManaBase * Math.pow(2, statusEffectManager.getActiveAmplifier(PentamanaStatusEffectIdentifiers.INSTANT_MANA)) : 0;
        regen -= statusEffectManager.containsKey(PentamanaStatusEffectIdentifiers.INSTANT_DEPLETE) ? Pentamana.statusEffectInstantDepleteBase * Math.pow(2, statusEffectManager.getActiveAmplifier(PentamanaStatusEffectIdentifiers.INSTANT_DEPLETE)) : 0;
        regen += statusEffectManager.containsKey(PentamanaStatusEffectIdentifiers.MANA_REGENERATION) ? Pentamana.manaPerPoint / (float)Math.max(1, Pentamana.statusEffectManaRegenerationBase >> statusEffectManager.getActiveAmplifier(PentamanaStatusEffectIdentifiers.MANA_REGENERATION)) : 0;
        regen -= statusEffectManager.containsKey(PentamanaStatusEffectIdentifiers.MANA_INHIBITION) ? Pentamana.manaPerPoint / (float)Math.max(1, Pentamana.statusEffectManaInhibitionBase >> statusEffectManager.getActiveAmplifier(PentamanaStatusEffectIdentifiers.MANA_INHIBITION)) : 0;

        RegenManaCallback.EVENT.invoker().interact(this.player);

        return this.regen(regen);
    }

    /**
     * Regen the {@code regen} amount of mana.
     * 
     * <p>Target supply is capped at capacity and 0.0f.
     * 
     * @param regen the amount to regen.
     * @return {@link #isFull()}.
     * 
     * @see #regen()
     */
    public boolean regen(float regen) {
        float targetSupply = this.supply + regen;
        targetSupply = Math.min(targetSupply, capacity);
        targetSupply = Math.max(targetSupply, 0.0f);
        return (this.supply = targetSupply) == this.capacity;
    }

    /**
     * Consume the {@code consum} amount of mana after the custom modifiers and enchantments
     * are applied if the player has at least {@code consum} amount of mana.
     * 
     * @param consum the amount to consume.
     * @return true if successful, otherwise false.
     */
    public boolean consum(float consum) {
        float targetConsum = (float)player.getCustomModifiedValue(PentamanaAttributeIdentifiers.MANA_CONSUMPTION, consum);
        targetConsum *= 1 - Pentamana.enchantmentUtilizationBase * player.method_59958().method_58657().getLevel(PentamanaEnchantmentIdentifiers.UTILIZATION);

        ConsumManaCallback.EVENT.invoker().interact(this.player);

        float targetSupply = this.supply - targetConsum;
        if (targetSupply >= 0.0f) {
            this.supply = targetSupply;
            return true;
        }

        return false;
    }

    public void tickClient() {
        if (this.isVisible) {
            if ((!this.equalsClient() || (this.isEndOfLife() && this.position == ManaBar.Position.ACTIONBAR)) && !this.isSuppressed()) {
                if (this.position != this.clientManaBar.position) {
                    this.finishOnScreen();
                }

                this.sendOnScreen();
                this.updateClient();
                this.life = (byte)-Pentamana.displayIdleInterval;
            }

            this.incrementLife();
        } else if (this.clientManaBar.isVisible) {
            this.finishOnScreen();
            this.updateClient();
            this.life = (byte)0;
        }
    }

    public void sendOnScreen() {
        if (this.position == ManaBar.Position.ACTIONBAR) {
            this.player.method_7353(this.textual.toText(capacity, supply), true);
        } else if (this.position == ManaBar.Position.BOSSBAR) {
            class_3002 bossbar = this.server.method_3837().getOrAdd(this.id, class_2561.method_43473());
            if (this.clientManaBar.capacity != this.capacity) {
                bossbar.method_12956((int)(this.capacity / Pentamana.manaPerPoint));
            }

            if (this.clientManaBar.supply != this.supply) {
                bossbar.method_12954((int)(this.supply / Pentamana.manaPerPoint));
            }

            if (this.clientManaBar.supply != this.supply
            || this.clientManaBar.capacity != this.capacity
            || Objects.equals(this.clientManaBar.textual, this.textual)) {
                bossbar.method_5413(this.textual.toText(capacity, supply));
            }

            if (this.clientManaBar.color != this.color) {
                bossbar.method_5416(this.color);
            }

            if (this.clientManaBar.style != this.style) {
                bossbar.method_5409(this.style);
            }

            if (this.clientManaBar.position != this.position) {
                bossbar.method_12956((int)(this.capacity / Pentamana.manaPerPoint));
                bossbar.method_12954((int)(this.supply / Pentamana.manaPerPoint));
                bossbar.method_5413(this.textual.toText(capacity, supply));
                bossbar.method_5416(this.color);
                bossbar.method_5409(this.style);
                bossbar.method_14088(this.player);
            }
        } else if (this.position == ManaBar.Position.SIDERBAR) {
            class_266 objective = this.server.method_3845().getOrAddObjective(this.name, class_274.field_1468, class_2561.method_43473(), class_274.class_275.field_1472, true, null);
            if (this.clientManaBar.position != this.position) {
                this.player.field_13987.method_14364(new class_2751(objective, class_2751.field_33343));
                this.player.field_13987.method_14364(new class_2736(class_8646.field_45157, objective));
            }

            if (this.clientManaBar.supply != this.supply
            || this.clientManaBar.capacity != this.capacity
            || Objects.equals(this.clientManaBar.textual, this.textual)) {
                objective.method_1121(this.textual.toText(capacity, supply));
                this.player.field_13987.method_14364(new class_2751(objective, class_2751.field_33345));
            }
        }
    }

    public void finishOnScreen() {
        if (this.clientManaBar.position == ServerManaBar.Position.ACTIONBAR) {
            this.player.method_7353(class_2561.method_43470(""), true);
        } else if (this.clientManaBar.position == ServerManaBar.Position.BOSSBAR) {
            class_3004 bossbarManager = this.server.method_3837();
            if (bossbarManager.contains(this.id)) {
                bossbarManager.method_12971(this.id).method_14094();
                bossbarManager.remove(this.id);
            }
        } else if (this.clientManaBar.position == ManaBar.Position.SIDERBAR) {
            this.player.field_13987.method_14364(new class_2751(this.server.method_3845().method_1170(this.name), class_2751.field_33344));
        }
    }

    public boolean isSuppressed() {
        return this.life > (byte)0 && this.life < Pentamana.displaySuppressionInterval;
    }

    public boolean isEndOfLife() {
        return this.life >= 0;
    }

    public boolean equalsClient() {
        return this.capacity == this.clientManaBar.capacity
        && this.supply == this.clientManaBar.supply
        && this.position == this.clientManaBar.position
        && Objects.equals(this.textual, this.clientManaBar.textual)
        && this.isVisible == this.clientManaBar.isVisible
        && this.color == this.clientManaBar.color
        && this.style == this.clientManaBar.style;
    }

    public void updateClient() {
        this.clientManaBar.capacity =
            this.clientManaBar.capacity == this.capacity ?
            this.clientManaBar.capacity :
            this.capacity;
        this.clientManaBar.supply =
            this.clientManaBar.supply == this.supply ?
            this.clientManaBar.supply :
            this.supply;
        this.clientManaBar.position =
            this.clientManaBar.position == this.position ?
            this.clientManaBar.position :
            this.position;
        this.clientManaBar.textual =
            Objects.equals(this.clientManaBar.textual, this.textual) ?
            this.clientManaBar.textual :
            this.textual.deepCopy();
        this.clientManaBar.isVisible =
            this.clientManaBar.isVisible == this.isVisible ?
            this.clientManaBar.isVisible :
            this.isVisible;
        this.clientManaBar.color =
            this.clientManaBar.color == this.color ?
            this.clientManaBar.color :
            this.color;
        this.clientManaBar.style =
            this.clientManaBar.style == this.style ?
            this.clientManaBar.style :
            this.style;
    }

    public MinecraftServer getServer() {
        return this.server;
    }

    public void setServer(MinecraftServer server) {
        this.server = server;
    }

    public ServerManaBar withServer(MinecraftServer server) {
        this.server = server;
        return this;
    }

    public class_3222 getPlayer() {
        return this.player;
    }

    public void setPlayer(class_3222 player) {
        this.player = player;
    }

    public ServerManaBar withPlayer(class_3222 player) {
        this.player = player;
        return this;
    }

    public UUID getUuid() {
        return this.uuid;
    }

    public void setUuid(UUID uuid) {
        this.uuid = uuid;
    }

    public ServerManaBar withUuid(UUID uuid) {
        this.uuid = uuid;
        return this;
    }

    public String getName() {
        return this.name;
    }

    public void setName(String name) {
        this.name = name;
    }

    public ServerManaBar withName(String name) {
        this.name = name;
        return this;
    }

    public class_2960 getId() {
        return this.id;
    }

    public void setId(class_2960 id) {
        this.id = id;
    }

    public ServerManaBar withId(class_2960 id) {
        this.id = id;
        return this;
    }

    public ManaBar getClientManaBar() {
        return this.clientManaBar;
    }

    public void setClientManaBar(ManaBar clientManaBar) {
        this.clientManaBar = clientManaBar;
    }

    public ServerManaBar withClientManaBar(ManaBar clientManaBar) {
        this.clientManaBar = clientManaBar;
        return this;
    }

    public byte getLife() {
        return this.life;
    }

    public void setLife(byte life) {
        this.life = life;
    }

    public byte incrementLife() {
        return ++this.life;
    }

    public byte incrementLife(byte value) {
        this.setLife((byte)(this.life + value));
        return this.life;
    }

    /**
     * @param nbtCompound
     * @param wrapperLookup
     * @return ServerManaBar with no server, player, uuid, name and id.
     */
    public static ServerManaBar fromNbt(class_2487 nbtCompound, class_7225.class_7874 wrapperLookup) {
        return ServerManaBar.of(ManaBar.fromNbt(nbtCompound, wrapperLookup))
            .withClientManaBar(ManaBar.fromNbt(nbtCompound.method_10562("clientManaBar"), wrapperLookup));
    }

    @Override
    public class_2487 toNbt(class_7225.class_7874 wrapperLookup) {
        class_2487 nbtCompound = super.toNbt(wrapperLookup);
        nbtCompound.method_10566("clientManaBar", this.clientManaBar.toNbt(wrapperLookup));
        return nbtCompound;
    }
}
