package mods.flammpfeil.slashblade.recipe;

import com.google.common.collect.ImmutableSet;
import com.google.gson.JsonArray;
import com.google.gson.JsonObject;
import com.google.gson.JsonSyntaxException;
import mods.flammpfeil.slashblade.SlashBlade;
import mods.flammpfeil.slashblade.init.SBItems;
import net.fabricmc.fabric.api.recipe.v1.ingredient.CustomIngredient;
import net.fabricmc.fabric.api.recipe.v1.ingredient.CustomIngredientSerializer;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_1935;
import net.minecraft.class_2540;
import net.minecraft.class_2960;
import net.minecraft.class_3518;
import net.minecraft.class_7923;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public class SlashBladeIngredient implements CustomIngredient {
    private final Set<class_1792> items;
    private final RequestDefinition request;

    protected SlashBladeIngredient(Set<class_1792> items, RequestDefinition request) {
        if (items.isEmpty()) {
            throw new IllegalArgumentException("Cannot create a SlashBladeIngredient with no items");
        }
        this.items = Collections.unmodifiableSet(items);
        this.request = request;
    }

    public static SlashBladeIngredient of(class_1935 item, RequestDefinition request) {
        return new SlashBladeIngredient(Set.of(item.method_8389()), request);
    }

    public static SlashBladeIngredient of(RequestDefinition request) {
        return new SlashBladeIngredient(Set.of(SBItems.slashblade), request);
    }

    @Override
    public boolean test(class_1799 input) {
        if (input == null)
            return false;
        return items.contains(input.method_7909()) && this.request.test(input);
    }

    @Override
    public List<class_1799> getMatchingStacks() {
        return items.stream().map(item -> {
            class_1799 stack = new class_1799(item);
            // copy NBT to prevent the stack from modifying the original, as capabilities or
            // vanilla item durability will modify the tag
            request.initItemStack(stack);
            // return new Ingredient.ItemValue(stack);
            return stack;
        }).toList();
    }

    @Override
    public boolean requiresTesting() {
        return true;
    }

/*    @Override
    public boolean isSimple() {
        return false;
    }*/

    @Override
    public CustomIngredientSerializer<? extends CustomIngredient> getSerializer() {
        return Serializer.INSTANCE;
    }

    public static class Serializer implements CustomIngredientSerializer<SlashBladeIngredient> {
        public static final Serializer INSTANCE = new Serializer();
        public static final class_2960 RES = SlashBlade.prefix("blade");

        @Override
        public class_2960 getIdentifier() {
            return RES;
        }

        @Override
        public SlashBladeIngredient read(JsonObject jsonObject) {
            // parse items
            Set<class_1792> items;
            if (jsonObject.has("item")) {
                class_2960 loc = class_2960.method_12829(class_3518.method_15265(jsonObject, "item"));
                if (loc == null) {
                    throw new JsonSyntaxException("Item " + loc + " is not a valid item");
                }
                items = Set.of(class_7923.field_41178.method_10223(loc));
            } else if (jsonObject.has("items")) {
                ImmutableSet.Builder<class_1792> builder = ImmutableSet.builder();
                JsonArray itemArray = class_3518.method_15261(jsonObject, "items");
                for (int i = 0; i < itemArray.size(); i++) {
                    class_2960 loc = class_2960.method_12829(class_3518.method_15287(itemArray.get(i), "items[" + i + ']'));
                    if (loc == null) {
                        throw new JsonSyntaxException("Item " + loc + " is not a valid item");
                    }
                    builder.add(class_7923.field_41178.method_10223(loc));
                }
                items = builder.build();
            } else
                throw new JsonSyntaxException("Must set either 'item' or 'items'");
            var request = RequestDefinition.fromJSON(jsonObject.getAsJsonObject("request"));
            return new SlashBladeIngredient(items, request);
        }

        @Override
        public void write(JsonObject jsonObject, SlashBladeIngredient slashBladeIngredient) {
            jsonObject.addProperty("type", RES.toString());
            if (slashBladeIngredient.items.size() == 1) {
                jsonObject.addProperty("item", class_7923.field_41178.method_10221(slashBladeIngredient.items.iterator().next()).toString());
            } else {
                JsonArray items = new JsonArray();
                // ensure the order of items in the set is deterministic when saved to JSON
                slashBladeIngredient.items.stream().map(class_7923.field_41178::method_10221).sorted().forEach(name -> items.add(name.toString()));
                jsonObject.add("items", items);
            }
            jsonObject.add("request", slashBladeIngredient.request.toJson());
        }

        @Override
        public SlashBladeIngredient read(class_2540 friendlyByteBuf) {
            Set<class_1792> items = Stream.generate(() -> friendlyByteBuf.method_42064(class_7923.field_41178))
                    .limit(friendlyByteBuf.method_10816()).collect(Collectors.toSet());
            RequestDefinition request = RequestDefinition.fromNetwork(friendlyByteBuf);
            return new SlashBladeIngredient(items, request);
        }

        @Override
        public void write(class_2540 buffer, SlashBladeIngredient ingredient) {
            buffer.method_10804(ingredient.items.size());
            for (class_1792 item : ingredient.items)
                buffer.method_42065(class_7923.field_41178, item);
            ingredient.request.toNetwork(buffer);
        }
    }
}
