package com.github.kd_gaming1.packcore.integration.resourcepack;

import com.github.kd_gaming1.packcore.PackCore;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import net.minecraft.class_310;
import net.minecraft.class_3288;

public class ResourcePackManager {
    private static final Map<String, String[]> MULTI_PACK_KEYWORDS = Map.of(
            "HypixelPlus", new String[]{"hypixel"},
            "FurfSkyOverlay", new String[]{"overlay", "furfsky"},
            "FurfSkyFull", new String[]{"full", "furfsky"},
            "SkyBlockDarkUI", new String[]{"skyblock", "dark", "ui"},
            "SkyBlockDarkMode", new String[]{"dark", "skyblock", "mode"},
            "SophieHypixelEnchants", new String[]{"sophie's", "enchants"},
            "Defrosted", new String[]{"defrosted"},
            "Looshy", new String[]{"looshy"}
    );

    private static final Map<String, String> PACK_KEYWORDS = Map.of(
            "HypixelPlus", "hypixel",
            "FurfSkyOverlay", "overlay",
            "FurfSkyFull", "full",
            "SkyBlockDarkUI", "skyblock dark ui",
            "SkyBlockDarkMode", "dark skyblock",
            "SophieHypixelEnchants", "sophie's hypixel enchants",
            "Defrosted", "defrosted",
            "Looshy", "looshy"
    );

    public static CompletableFuture<Boolean> applyResourcePacksOrdered(List<String> packKeysOrdered) {
        CompletableFuture<Boolean> result = new CompletableFuture<>();

        try {
            class_310 client = class_310.method_1551();
            if (client == null) {
                return CompletableFuture.completedFuture(false);
            }

            // The mixin handles sanitization automatically when scanPacks() is called
            client.execute(() -> applyPacksOnMainThread(packKeysOrdered, result));

        } catch (Exception e) {
            PackCore.LOGGER.error("Failed to schedule resource packs application", e);
            return CompletableFuture.completedFuture(false);
        }

        return result.completeOnTimeout(false, 10, TimeUnit.SECONDS);
    }

    private static void applyPacksOnMainThread(List<String> packKeysOrdered, CompletableFuture<Boolean> result) {
        try {
            class_310 client = class_310.method_1551();
            net.minecraft.class_3283 packManager = client.method_1520();

            // Scan packs - mixin will sanitize filenames before this processes them
            packManager.method_14445();

            List<String> foundPackIds = findAvailablePackIdsOrderedList(packKeysOrdered);

            if (foundPackIds.isEmpty()) {
                PackCore.LOGGER.warn("No matching resource packs found for keys: {}", packKeysOrdered);
                result.complete(true);
                return;
            }

            PackCore.LOGGER.info("Found {} packs in order: {}", foundPackIds.size(), foundPackIds);

            List<String> currentPacks = new ArrayList<>(client.field_1690.field_1887);
            List<String> newPacks = new ArrayList<>();
            Set<String> allKnownPackIds = getAllKnownPackIds();

            PackCore.LOGGER.info("Current packs from options.txt: {}", currentPacks);
            PackCore.LOGGER.info("All known managed pack IDs: {}", allKnownPackIds);
            PackCore.LOGGER.info("Packs to add (in order): {}", foundPackIds);

            for (String pack : currentPacks) {
                if (!allKnownPackIds.contains(pack)) {
                    newPacks.add(pack);
                    PackCore.LOGGER.info("Keeping non-managed pack: {}", pack);
                } else {
                    PackCore.LOGGER.info("Removing previously enabled managed pack: {}", pack);
                }
            }

            PackCore.LOGGER.info("Base packs (after removing managed): {}", newPacks);

            Collections.reverse(foundPackIds);

            for (int i = 0; i < foundPackIds.size(); i++) {
                String packId = foundPackIds.get(i);
                if (!newPacks.contains(packId)) {
                    newPacks.add(packId);
                    PackCore.LOGGER.info("Adding resource pack: {} (reversed index: {}, final position: {})",
                            packId, i + 1, newPacks.size() - 1);
                } else {
                    PackCore.LOGGER.warn("Pack {} was already in the list, skipping duplicate", packId);
                }
            }

            PackCore.LOGGER.info("Final pack order for options.txt: {}", newPacks);

            packManager.method_14447(newPacks);
            client.field_1690.field_1887.clear();
            client.field_1690.field_1887.addAll(newPacks);
            client.field_1690.method_1640();

            client.method_1521().thenRun(() -> {
                PackCore.LOGGER.info("Resource reload completed successfully");
                result.complete(true);
            }).exceptionally(e -> {
                PackCore.LOGGER.error("Resource reload failed", e);
                result.complete(false);
                return null;
            });

        } catch (Exception e) {
            PackCore.LOGGER.error("Failed to apply packs", e);
            result.complete(false);
        }
    }

    private static List<String> findAvailablePackIdsOrderedList(List<String> packKeysOrdered) {
        List<String> found = new ArrayList<>();
        class_310 client = class_310.method_1551();
        if (client == null) return found;

        net.minecraft.class_3283 packManager = client.method_1520();
        Collection<class_3288> allProfiles = packManager.method_14441();

        for (int index = 0; index < packKeysOrdered.size(); index++) {
            String key = packKeysOrdered.get(index).trim();
            if (key.isEmpty()) continue;

            String[] keywords = getKeywordsForPack(key);
            if (keywords == null) {
                PackCore.LOGGER.warn("No keywords found for pack key: {}", key);
                continue;
            }

            PackCore.LOGGER.info("Looking for pack '{}' (selection #{}) with keywords: {}",
                    key, index + 1, Arrays.toString(keywords));

            class_3288 bestMatch = findBestMatch(allProfiles, keywords);

            if (bestMatch != null) {
                found.add(bestMatch.method_14463());
                PackCore.LOGGER.info("MATCHED '{}' -> '{}' (display: '{}') [Selection #{} -> Pack position #{}]",
                        key, bestMatch.method_14463(), bestMatch.method_14457().getString(),
                        index + 1, found.size());
            } else {
                PackCore.LOGGER.warn("No match found for pack '{}' (selection #{}) with keywords: {}",
                        key, index + 1, Arrays.toString(keywords));
            }
        }

        return found;
    }

    private static String[] getKeywordsForPack(String key) {
        String[] keywords = MULTI_PACK_KEYWORDS.get(key);
        if (keywords == null) {
            String keyword = PACK_KEYWORDS.get(key);
            if (keyword != null) {
                keywords = new String[]{keyword};
            }
        }
        return keywords;
    }

    private static class_3288 findBestMatch(Collection<class_3288> allProfiles, String[] keywords) {
        class_3288 bestMatch = null;
        int bestScore = -1;

        for (class_3288 profile : allProfiles) {
            String name = stripMinecraftColors(profile.method_14457().getString().toLowerCase());
            String id = stripMinecraftColors(profile.method_14463().toLowerCase());
            String desc = stripMinecraftColors(profile.method_14459().getString().toLowerCase());

            int totalScore = 0;
            int matched = 0;
            for (String kw : keywords) {
                int score = getMatchScore(id, name, desc, kw.toLowerCase());
                if (score > 0) {
                    totalScore += score;
                    matched++;
                }
            }
            totalScore += matched > 1 ? matched * 2 : 0;

            if (totalScore > bestScore) {
                bestScore = totalScore;
                bestMatch = profile;
            }
        }
        return bestMatch;
    }

    private static int getMatchScore(String id, String name, String desc, String keyword) {
        if (id.equals(keyword)) return 5;
        if (id.startsWith(keyword)) return 4;
        if (id.contains(keyword)) return 3;
        if (name.contains(keyword)) return 2;
        if (desc.contains(keyword)) return 1;
        return 0;
    }

    private static Set<String> getAllKnownPackIds() {
        Set<String> allKnown = new HashSet<>();
        class_310 client = class_310.method_1551();
        if (client == null) return allKnown;

        net.minecraft.class_3283 packManager = client.method_1520();

        Set<String> allKeywords = new HashSet<>(PACK_KEYWORDS.values());
        for (String[] keywords : MULTI_PACK_KEYWORDS.values()) {
            allKeywords.addAll(Arrays.asList(keywords));
        }

        for (String keyword : allKeywords) {
            for (class_3288 profile : packManager.method_14441()) {
                String name = stripMinecraftColors(profile.method_14457().getString().toLowerCase());
                String id = stripMinecraftColors(profile.method_14463().toLowerCase());
                String desc = stripMinecraftColors(profile.method_14459().getString().toLowerCase());

                if (name.contains(keyword) || id.contains(keyword) || desc.contains(keyword)) {
                    allKnown.add(profile.method_14463());
                }
            }
        }

        PackCore.LOGGER.debug("All known managed pack IDs: {}", allKnown);
        return allKnown;
    }

    private static String stripMinecraftColors(String text) {
        return text.replaceAll("§[0-9a-fk-or]", "");
    }
}