package com.github.kd_gaming1.packcore.scamshield;

import com.github.kd_gaming1.packcore.scamshield.detector.ConfidenceLevel;
import com.github.kd_gaming1.packcore.scamshield.detector.DetectionResult;
import com.github.kd_gaming1.packcore.scamshield.detector.ScamCategory;
import net.minecraft.class_124;
import net.minecraft.class_2558;
import net.minecraft.class_2561;
import net.minecraft.class_2568;
import net.minecraft.class_5250;

/**
 * Builds tiered warning messages for scam detections.
 * Uses category-specific variants with proper Minecraft formatting.
 */
public class ScamWarningMessageBuilder {
    /**
     * Build warning message based on detection confidence and category.
     */
    public static class_2561 buildWarningMessage(DetectionResult result) {
        ConfidenceLevel level = result.getConfidenceLevel();
        ScamCategory category = result.getPrimaryCategory();

        return switch (level) {
            case LOW -> buildLowConfidenceMessage(result, category);
            case MEDIUM -> buildMediumConfidenceMessage(result, category);
            case HIGH -> buildHighConfidenceMessage(result, category);
        };
    }

    // LOW CONFIDENCE
    private static class_2561 buildLowConfidenceMessage(DetectionResult result, ScamCategory category) {
        class_5250 message = class_2561.method_43470("");

        // Header with spacing
        message.method_10852(class_2561.method_43470("\n⚠ LOW CONFIDENCE ⚠\n")
                .method_27694(style -> style.method_36139(0xFFAA00).method_10982(true)));
        message.method_10852(class_2561.method_43470("ScamShield Notice\n\n")
                .method_27694(style -> style.method_36139(0xFFAA00).method_10982(true)));

        // Category-specific message
        switch (category) {
            case DISCORD_VERIFY:
            case PHISHING:
                appendLowPhishing(message);
                break;
            case FAKE_REWARD:
                appendLowGiveaway(message);
                break;
            case ACCOUNT_THEFT:
                appendLowCoopIsland(message);
                break;
            default:
                appendLowBase(message);
                break;
        }

        // Action buttons with spacing
        message.method_10852(class_2561.method_43470("\n"));
        appendLowActions(message, result);
        message.method_10852(class_2561.method_43470("\n"));

        return message;
    }

    private static void appendLowBase(class_5250 message) {
        message.method_10852(class_2561.method_43470("This user sent unusual messages.\n")
                .method_27694(style -> style.method_36139(0xFFFFFF)));

        message.method_10852(class_2561.method_43470("\nIt might be harmless, but stay cautious:\n")
                .method_27694(style -> style.method_36139(0xAAAAAA)));

        message.method_10852(class_2561.method_43470("• Don't click suspicious links\n")
                .method_27694(style -> style.method_36139(0xFFFF55)));
        message.method_10852(class_2561.method_43470("• Don't enter login info\n")
                .method_27694(style -> style.method_36139(0xFFFF55)));
    }

    private static void appendLowPhishing(class_5250 message) {
        message.method_10852(class_2561.method_43470("Possible suspicious link or verification request detected.\n")
                .method_27694(style -> style.method_36139(0xFFFFFF)));

        message.method_10852(class_2561.method_43470("\n⚠ Stay Safe:\n")
                .method_27694(style -> style.method_36139(0xFFAA00).method_10982(true)));

        message.method_10852(class_2561.method_43470("• Scams often ask you to verify through fake sites\n")
                .method_27694(style -> style.method_36139(0xAAAAAA)));
        message.method_10852(class_2561.method_43470("• Don't click links from strangers\n")
                .method_27694(style -> style.method_36139(0xFFFF55)));
        message.method_10852(class_2561.method_43470("• Never enter your email or password\n")
                .method_27694(style -> style.method_36139(0xFFFF55)));
    }

    private static void appendLowGiveaway(class_5250 message) {
        message.method_10852(class_2561.method_43470("\"Too good to be true\" offer detected.\n")
                .method_27694(style -> style.method_36139(0xFFFFFF)));
        message.method_10852(class_2561.method_43470("(Free ranks, giveaways, high-profit trades)\n\n")
                .method_27694(style -> style.method_36139(0xAAAAAA)));

        message.method_10852(class_2561.method_43470("Could be real, but scammers often:\n")
                .method_27694(style -> style.method_36139(0xAAAAAA)));
        message.method_10852(class_2561.method_43470("• Promise big rewards to make you act fast\n")
                .method_27694(style -> style.method_36139(0xFFFF55)));
        message.method_10852(class_2561.method_43470("• Ask you to trade or visit their island first\n")
                .method_27694(style -> style.method_36139(0xFFFF55)));
    }

    private static void appendLowCoopIsland(class_5250 message) {
        message.method_10852(class_2561.method_43470("Co-op or island access mentioned.\n\n")
                .method_27694(style -> style.method_36139(0xFFFFFF)));

        message.method_10852(class_2561.method_43470("⚠ Warning:\n")
                .method_27694(style -> style.method_36139(0xFFAA00).method_10982(true)));
        message.method_10852(class_2561.method_43470("Some scammers use co-op to steal items or delete islands.\n\n")
                .method_27694(style -> style.method_36139(0xAAAAAA)));

        message.method_10852(class_2561.method_43470("Double-check before using ")
                .method_27694(style -> style.method_36139(0xFFFF55)));
        message.method_10852(class_2561.method_43470("/coopadd")
                .method_27694(style -> style.method_36139(0xFF5555).method_10982(true)));
    }

    private static void appendLowActions(class_5250 message, DetectionResult result) {
        message.method_10852(class_2561.method_43470("📚 Learn more: ")
                .method_27694(style -> style.method_36139(0xAAAAAA)));
        message.method_10852(buildEducationButton());
        message.method_10852(class_2561.method_43470("\n"));

        message.method_10852(class_2561.method_43470("✓ Trust this player? ")
                .method_27694(style -> style.method_36139(0xAAAAAA)));
        message.method_10852(buildWhitelistButton(result.getSender()));
    }

    // MEDIUM CONFIDENCE
    private static class_2561 buildMediumConfidenceMessage(DetectionResult result, ScamCategory category) {
        class_5250 message = class_2561.method_43470("");

        // Header with spacing
        message.method_10852(class_2561.method_43470("\n🟠 MEDIUM CONFIDENCE 🟠\n")
                .method_27694(style -> style.method_36139(0xFF5555).method_10982(true)));
        message.method_10852(class_2561.method_43470("ScamShield Warning\n\n")
                .method_27694(style -> style.method_36139(0xFF5555).method_10982(true)));

        // Category-specific message
        switch (category) {
            case DISCORD_VERIFY:
            case PHISHING:
                appendMediumPhishing(message);
                break;
            case FAKE_REWARD:
                appendMediumGiveaway(message);
                break;
            case ACCOUNT_THEFT:
                appendMediumCoopIsland(message);
                break;
            case CUSTOM:
                if (result.getTriggeredScamTypes().stream()
                        .anyMatch(type -> type.contains("trade") || type.contains("manipulation"))) {
                    appendMediumTrade(message);
                } else {
                    appendMediumBase(message);
                }
                break;
            default:
                appendMediumBase(message);
                break;
        }

        // Action buttons with spacing
        message.method_10852(class_2561.method_43470("\n"));
        appendMediumActions(message, result);
        message.method_10852(class_2561.method_43470("\n"));

        return message;
    }

    private static void appendMediumBase(class_5250 message) {
        message.method_10852(class_2561.method_43470("Multiple messages match known scam patterns.\n\n")
                .method_27694(style -> style.method_36139(0xFFFFFF)));

        message.method_10852(class_2561.method_43470("🛑 DO NOT:\n")
                .method_27694(style -> style.method_36139(0xFF5555).method_10982(true)));
        message.method_10852(class_2561.method_43470("• Click suspicious links\n")
                .method_27694(style -> style.method_36139(0xFFFF55)));
        message.method_10852(class_2561.method_43470("• Log in on unfamiliar websites\n")
                .method_27694(style -> style.method_36139(0xFFFF55)));
        message.method_10852(class_2561.method_43470("• Enter email, password, or codes\n")
                .method_27694(style -> style.method_36139(0xFFFF55)));
        message.method_10852(class_2561.method_43470("• \"Verify\" through Discord\n")
                .method_27694(style -> style.method_36139(0xFFFF55)));
    }

    private static void appendMediumPhishing(class_5250 message) {
        message.method_10852(class_2561.method_43470("Phishing/Verification scam detected.\n\n")
                .method_27694(style -> style.method_36139(0xFFFFFF)));

        message.method_10852(class_2561.method_43470("⚠ This scammer likely wants to:\n")
                .method_27694(style -> style.method_36139(0xFF5555).method_10982(true)));
        message.method_10852(class_2561.method_43470("• Make you \"verify\" through fake sites\n")
                .method_27694(style -> style.method_36139(0xAAAAAA)));
        message.method_10852(class_2561.method_43470("• Steal your account login details\n\n")
                .method_27694(style -> style.method_36139(0xAAAAAA)));

        message.method_10852(class_2561.method_43470("✓ Only log in on official Microsoft/Minecraft pages\n")
                .method_27694(style -> style.method_36139(0x55FF55)));
        message.method_10852(class_2561.method_43470("✗ Never enter credentials elsewhere\n")
                .method_27694(style -> style.method_36139(0xFF5555)));
    }

    private static void appendMediumGiveaway(class_5250 message) {
        message.method_10852(class_2561.method_43470("\"Free rank\" or giveaway scam detected.\n\n")
                .method_27694(style -> style.method_36139(0xFFFFFF)));

        message.method_10852(class_2561.method_43470("Common tactics:\n")
                .method_27694(style -> style.method_36139(0xAAAAAA)));
        message.method_10852(class_2561.method_43470("• Ask you to join Discord first\n")
                .method_27694(style -> style.method_36139(0xFFFF55)));
        message.method_10852(class_2561.method_43470("• Request coins before \"rewarding\" you\n\n")
                .method_27694(style -> style.method_36139(0xFFFF55)));

        message.method_10852(class_2561.method_43470("⚠ If it sounds too good to be true, it usually is.\n")
                .method_27694(style -> style.method_36139(0xFF5555).method_10982(true)));
    }

    private static void appendMediumTrade(class_5250 message) {
        message.method_10852(class_2561.method_43470("Trade or \"item flip\" scam detected.\n\n")
                .method_27694(style -> style.method_36139(0xFFFFFF)));

        message.method_10852(class_2561.method_43470("Red flags:\n")
                .method_27694(style -> style.method_36139(0xAAAAAA)));
        message.method_10852(class_2561.method_43470("• Deals far better than market value\n")
                .method_27694(style -> style.method_36139(0xFFFF55)));
        message.method_10852(class_2561.method_43470("• Pressure to trade quickly\n")
                .method_27694(style -> style.method_36139(0xFFFF55)));
        message.method_10852(class_2561.method_43470("• Asking for payment before delivery\n\n")
                .method_27694(style -> style.method_36139(0xFFFF55)));

        message.method_10852(class_2561.method_43470("⚠ Be cautious when paying before receiving items.\n")
                .method_27694(style -> style.method_36139(0xFF5555)));
    }

    private static void appendMediumCoopIsland(class_5250 message) {
        message.method_10852(class_2561.method_43470("Co-op access scam detected.\n\n")
                .method_27694(style -> style.method_36139(0xFFFFFF)));

        message.method_10852(class_2561.method_43470("⚠ Danger:\n")
                .method_27694(style -> style.method_36139(0xFF5555).method_10982(true)));
        message.method_10852(class_2561.method_43470("Scammers may ask you to run ")
                .method_27694(style -> style.method_36139(0xAAAAAA)));
        message.method_10852(class_2561.method_43470("/coopadd")
                .method_27694(style -> style.method_36139(0xFF5555).method_10982(true)));
        message.method_10852(class_2561.method_43470("\nfor \"gifts\" or \"help.\"\n\n")
                .method_27694(style -> style.method_36139(0xAAAAAA)));

        message.method_10852(class_2561.method_43470("Once added, they can:\n")
                .method_27694(style -> style.method_36139(0xFF5555)));
        message.method_10852(class_2561.method_43470("• Steal your items\n")
                .method_27694(style -> style.method_36139(0xFFFF55)));
        message.method_10852(class_2561.method_43470("• Delete your island\n\n")
                .method_27694(style -> style.method_36139(0xFFFF55)));

        message.method_10852(class_2561.method_43470("⛔ Don't add players you don't personally know.\n")
                .method_27694(style -> style.method_36139(0xFF5555).method_10982(true)));
    }

    private static void appendMediumActions(class_5250 message, DetectionResult result) {
        message.method_10852(class_2561.method_43470("📚 Learn more: ")
                .method_27694(style -> style.method_36139(0xAAAAAA)));
        message.method_10852(buildEducationButton());
        message.method_10852(class_2561.method_43470("\n\n"));

        message.method_10852(class_2561.method_43470("⚠ Whitelist only if you fully trust this player:\n")
                .method_27694(style -> style.method_36139(0xFF5555)));
        message.method_10852(class_2561.method_43470("/scamshield whitelist add " + result.getSender())
                .method_27694(style -> style.method_36139(0xAAAAAA).method_10978(true)));
    }

    // HIGH CONFIDENCE
    private static class_2561 buildHighConfidenceMessage(DetectionResult result, ScamCategory category) {
        class_5250 message = class_2561.method_43470("");

        // Critical header
        message.method_10852(class_2561.method_43470("\n⛔⛔⛔⛔⛔⛔⛔⛔⛔⛔⛔⛔⛔⛔⛔⛔⛔⛔⛔⛔\n")
                .method_27694(style -> style.method_36139(0xFF0000).method_10982(true)));
        message.method_10852(class_2561.method_43470("   CRITICAL SCAM ALERT   \n")
                .method_27694(style -> style.method_36139(0xFF0000).method_10982(true)));
        message.method_10852(class_2561.method_43470("⛔⛔⛔⛔⛔⛔⛔⛔⛔⛔⛔⛔⛔⛔⛔⛔⛔⛔⛔⛔\n\n")
                .method_27694(style -> style.method_36139(0xFF0000).method_10982(true)));

        message.method_10852(class_2561.method_43470("🚨 NEARLY CERTAIN SCAM ATTEMPT 🚨\n\n")
                .method_27694(style -> style.method_36139(0xFF0000).method_10982(true)));

        // Category-specific critical message
        switch (category) {
            case DISCORD_VERIFY:
            case PHISHING:
                appendHighPhishing(message);
                break;
            case FAKE_REWARD:
                appendHighGiveaway(message);
                break;
            case ACCOUNT_THEFT:
                appendHighCoopIsland(message);
                break;
            case CUSTOM:
                if (result.getTriggeredScamTypes().stream()
                        .anyMatch(type -> type.contains("trade") || type.contains("manipulation"))) {
                    appendHighTrade(message);
                } else {
                    appendHighBase(message);
                }
                break;
            default:
                appendHighBase(message);
                break;
        }

        // Critical actions
        message.method_10852(class_2561.method_43470("\n"));
        appendHighActions(message, result);
        message.method_10852(class_2561.method_43470("\n"));

        return message;
    }

    private static void appendHighBase(class_5250 message) {
        message.method_10852(class_2561.method_43470("🚨 Scam Type: ")
                .method_27694(style -> style.method_36139(0xFF5555).method_10982(true)));
        message.method_10852(class_2561.method_43470("Account Theft\n\n")
                .method_27694(style -> style.method_36139(0xFFFFFF).method_10982(true)));

        message.method_10852(class_2561.method_43470("⛔ STOP ALL COMMUNICATION NOW!\n\n")
                .method_27694(style -> style.method_36139(0xFF0000).method_10982(true)));

        message.method_10852(class_2561.method_43470("What they want:\n")
                .method_27694(style -> style.method_36139(0xFFFF55).method_10982(true)));
        message.method_10852(class_2561.method_43470("• Your login credentials\n")
                .method_27694(style -> style.method_36139(0xFFFFFF)));
        message.method_10852(class_2561.method_43470("• Your 2FA codes\n")
                .method_27694(style -> style.method_36139(0xFFFFFF)));
        message.method_10852(class_2561.method_43470("• Access to steal everything\n\n")
                .method_27694(style -> style.method_36139(0xFFFFFF)));

        appendRecommendedActions(message);
    }

    private static void appendHighPhishing(class_5250 message) {
        message.method_10852(class_2561.method_43470("🚨 Scam Type: ")
                .method_27694(style -> style.method_36139(0xFF5555).method_10982(true)));
        message.method_10852(class_2561.method_43470("Discord/Verification Phishing\n\n")
                .method_27694(style -> style.method_36139(0xFFFFFF).method_10982(true)));

        message.method_10852(class_2561.method_43470("⛔ STOP! DO NOT:\n")
                .method_27694(style -> style.method_36139(0xFF0000).method_10982(true)));
        message.method_10852(class_2561.method_43470("• Click any links\n")
                .method_27694(style -> style.method_36139(0xFF5555)));
        message.method_10852(class_2561.method_43470("• \"Verify\" your account\n")
                .method_27694(style -> style.method_36139(0xFF5555)));
        message.method_10852(class_2561.method_43470("• Enter Microsoft login details\n\n")
                .method_27694(style -> style.method_36139(0xFF5555)));

        appendRecommendedActions(message);
    }

    private static void appendHighGiveaway(class_5250 message) {
        message.method_10852(class_2561.method_43470("🚨 Scam Type: ")
                .method_27694(style -> style.method_36139(0xFF5555).method_10982(true)));
        message.method_10852(class_2561.method_43470("Fake Giveaway/Free Rank\n\n")
                .method_27694(style -> style.method_36139(0xFFFFFF).method_10982(true)));

        message.method_10852(class_2561.method_43470("⛔ STOP! DO NOT:\n")
                .method_27694(style -> style.method_36139(0xFF0000).method_10982(true)));
        message.method_10852(class_2561.method_43470("• Send any items or coins\n")
                .method_27694(style -> style.method_36139(0xFF5555)));
        message.method_10852(class_2561.method_43470("• Log in to any links\n")
                .method_27694(style -> style.method_36139(0xFF5555)));
        message.method_10852(class_2561.method_43470("• Join their Discord\n\n")
                .method_27694(style -> style.method_36139(0xFF5555)));

        appendRecommendedActions(message);
    }

    private static void appendHighTrade(class_5250 message) {
        message.method_10852(class_2561.method_43470("🚨 Scam Type: ")
                .method_27694(style -> style.method_36139(0xFF5555).method_10982(true)));
        message.method_10852(class_2561.method_43470("Trade/Item Flip Scam\n\n")
                .method_27694(style -> style.method_36139(0xFFFFFF).method_10982(true)));

        message.method_10852(class_2561.method_43470("⛔ STOP THE TRADE!\n\n")
                .method_27694(style -> style.method_36139(0xFF0000).method_10982(true)));

        message.method_10852(class_2561.method_43470("✅ Keep your items and coins safe\n")
                .method_27694(style -> style.method_36139(0x55FF55)));
        message.method_10852(class_2561.method_43470("✅ Report and block immediately\n\n")
                .method_27694(style -> style.method_36139(0x55FF55)));

        appendRecommendedActions(message);
    }

    private static void appendHighCoopIsland(class_5250 message) {
        message.method_10852(class_2561.method_43470("🚨 Scam Type: ")
                .method_27694(style -> style.method_36139(0xFF5555).method_10982(true)));
        message.method_10852(class_2561.method_43470("Co-op Access Theft\n\n")
                .method_27694(style -> style.method_36139(0xFFFFFF).method_10982(true)));

        message.method_10852(class_2561.method_43470("⛔ DO NOT RUN ")
                .method_27694(style -> style.method_36139(0xFF0000).method_10982(true)));
        message.method_10852(class_2561.method_43470("/coopadd")
                .method_27694(style -> style.method_36139(0xFF0000).method_10982(true)));
        message.method_10852(class_2561.method_43470("!\n\n")
                .method_27694(style -> style.method_36139(0xFF0000).method_10982(true)));

        message.method_10852(class_2561.method_43470("They can:\n")
                .method_27694(style -> style.method_36139(0xFF5555)));
        message.method_10852(class_2561.method_43470("• Steal ALL your items\n")
                .method_27694(style -> style.method_36139(0xFFFF55)));
        message.method_10852(class_2561.method_43470("• Delete your island entirely\n\n")
                .method_27694(style -> style.method_36139(0xFFFF55)));

        appendRecommendedActions(message);
    }

    private static void appendRecommendedActions(class_5250 message) {
        message.method_10852(class_2561.method_43470("✅ TAKE ACTION NOW:\n")
                .method_27694(style -> style.method_36139(0x55FF55).method_10982(true)));
        message.method_10852(class_2561.method_43470("1. Stop replying immediately\n")
                .method_27694(style -> style.method_36139(0xFFFFFF)));
        message.method_10852(class_2561.method_43470("2. Use ")
                .method_27694(style -> style.method_36139(0xFFFFFF)));
        message.method_10852(class_2561.method_43470("/report")
                .method_27694(style -> style.method_36139(0x55FF55).method_10982(true)));
        message.method_10852(class_2561.method_43470(" to notify staff\n")
                .method_27694(style -> style.method_36139(0xFFFFFF)));
        message.method_10852(class_2561.method_43470("3. Block with ")
                .method_27694(style -> style.method_36139(0xFFFFFF)));
        message.method_10852(class_2561.method_43470("/ignore add <PlayerName>\n")
                .method_27694(style -> style.method_36139(0x55FF55).method_10982(true)));
    }

    private static void appendHighActions(class_5250 message, DetectionResult result) {
        message.method_10852(class_2561.method_43470("📚 Learn how to protect your account:\n")
                .method_27694(style -> style.method_36139(0xAAAAAA)));
        message.method_10852(buildEducationButton());
    }

    // HELPER METHODS - Clickable Buttons

    /**
     * Builds a clickable ScamShield Education menu button.
     */
    private static class_5250 buildEducationButton() {
        return class_2561.method_43470("[Open ScamShield Education Menu]")
                .method_27694(style -> style
                        .method_10977(class_124.field_1078)
                        .method_30938(true)
                        .method_10958(new class_2558.class_10609("/scamshield education"))
                        .method_10949(new class_2568.class_10613(
                                class_2561.method_43470("§aClick to learn about common scams\n§7Run: /scamshield education")
                        )));
    }

    /**
     * Builds a clickable whitelist button for the specified player.
     *
     * @param playerName The name of the player to whitelist.
     */
    private static class_5250 buildWhitelistButton(String playerName) {
        String command = "/scamshield whitelist add " + playerName;

        return class_2561.method_43470("[Whitelist User]")
                .method_27694(style -> style
                        .method_10977(class_124.field_1060)
                        .method_30938(true)
                        .method_10958(new class_2558.class_10610(command))
                        .method_10949(new class_2568.class_10613(
                                class_2561.method_43470("§aClick to whitelist §e" + playerName
                                        + "\n§7This will skip future scam checks for this player."
                                        + "\n§7Suggest: " + command)
                        )));
    }
}