/*
 * Decompiled with CFR 0.152.
 */
package com.github.kd_gaming1.packcore.util.copysystem;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.FileTime;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicLong;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import java.util.zip.ZipInputStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AsyncUnzipFiles {
    private static final Logger LOGGER = LoggerFactory.getLogger(AsyncUnzipFiles.class);
    private static final int BUFFER_SIZE = 16384;
    private static final ExecutorService UNZIP_EXECUTOR = Executors.newCachedThreadPool(r -> {
        Thread thread = new Thread(r);
        thread.setName("AsyncUnzip-" + thread.threadId());
        thread.setDaemon(true);
        return thread;
    });

    public CompletableFuture<Void> unzipAsync(String zipFilePath, String destDir, ProgressCallback progressCallback) {
        return CompletableFuture.runAsync(() -> {
            try {
                this.unzip(zipFilePath, destDir, progressCallback);
            }
            catch (IOException e) {
                throw new RuntimeException("Failed to unzip file", e);
            }
        }, UNZIP_EXECUTOR);
    }

    public void unzip(String zipFilePath, String destDir, ProgressCallback progressCallback) throws IOException {
        long totalSize = this.calculateTotalUncompressedSize(zipFilePath);
        AtomicLong processedBytes = new AtomicLong(0L);
        int lastReportedProgress = 0;
        Path destPath = Path.of(destDir, new String[0]);
        Files.createDirectories(destPath, new FileAttribute[0]);
        byte[] buffer = new byte[16384];
        try (ZipFile zipFile = new ZipFile(zipFilePath);){
            Enumeration<? extends ZipEntry> entries = zipFile.entries();
            while (entries.hasMoreElements()) {
                ZipEntry entry = entries.nextElement();
                Path entryPath = destPath.resolve(entry.getName()).normalize();
                if (!entryPath.startsWith(destPath)) {
                    LOGGER.warn("Skipping entry with path outside destination: {}", (Object)entry.getName());
                    continue;
                }
                if (entry.isDirectory()) {
                    Files.createDirectories(entryPath, new FileAttribute[0]);
                    LOGGER.debug("Created directory: {}", (Object)entryPath);
                    continue;
                }
                Files.createDirectories(entryPath.getParent(), new FileAttribute[0]);
                try (InputStream is = zipFile.getInputStream(entry);
                     BufferedInputStream bis = new BufferedInputStream(is, 16384);
                     OutputStream os = Files.newOutputStream(entryPath, new OpenOption[0]);
                     BufferedOutputStream bos = new BufferedOutputStream(os, 16384);){
                    int bytesRead;
                    while ((bytesRead = bis.read(buffer)) > 0) {
                        int currentProgress;
                        bos.write(buffer, 0, bytesRead);
                        long processed = processedBytes.addAndGet(bytesRead);
                        if (progressCallback == null || totalSize <= 0L || (currentProgress = (int)(processed * 100L / totalSize)) == lastReportedProgress) continue;
                        lastReportedProgress = currentProgress;
                        progressCallback.onProgress(processed, totalSize, currentProgress);
                    }
                }
                if (entry.getTime() != -1L) {
                    Files.setLastModifiedTime(entryPath, FileTime.fromMillis(entry.getTime()));
                }
                LOGGER.debug("Extracted: {}", (Object)entryPath);
            }
            if (progressCallback != null) {
                progressCallback.onProgress(processedBytes.get(), totalSize, 100);
            }
            LOGGER.info("Successfully unzipped {} to {}", (Object)zipFilePath, (Object)destDir);
        }
    }

    private long calculateTotalUncompressedSize(String zipFilePath) {
        long totalSize = 0L;
        try (ZipFile zipFile = new ZipFile(zipFilePath);){
            Enumeration<? extends ZipEntry> entries = zipFile.entries();
            while (entries.hasMoreElements()) {
                long size;
                ZipEntry entry = entries.nextElement();
                if (entry.isDirectory() || (size = entry.getSize()) <= 0L) continue;
                totalSize += size;
            }
        }
        catch (IOException e) {
            LOGGER.error("Failed to calculate total size", (Throwable)e);
            return this.calculateTotalSizeAlternate(zipFilePath);
        }
        return totalSize;
    }

    private long calculateTotalSizeAlternate(String zipFilePath) {
        long totalSize = 0L;
        try (FileInputStream fis = new FileInputStream(zipFilePath);
             BufferedInputStream bis = new BufferedInputStream(fis, 16384);
             ZipInputStream zis = new ZipInputStream(bis);){
            ZipEntry entry;
            while ((entry = zis.getNextEntry()) != null) {
                if (!entry.isDirectory() && entry.getSize() > 0L) {
                    totalSize += entry.getSize();
                }
                zis.closeEntry();
            }
        }
        catch (IOException e) {
            LOGGER.error("Failed to calculate total size (alternate method)", (Throwable)e);
        }
        return totalSize;
    }

    public CompletableFuture<Void> extractFileAsync(String zipFilePath, String entryName, String destPath, ProgressCallback progressCallback) {
        return CompletableFuture.runAsync(() -> {
            try {
                this.extractFile(zipFilePath, entryName, destPath, progressCallback);
            }
            catch (IOException e) {
                throw new RuntimeException("Failed to extract file", e);
            }
        }, UNZIP_EXECUTOR);
    }

    public void extractFile(String zipFilePath, String entryName, String destPath, ProgressCallback progressCallback) throws IOException {
        try (ZipFile zipFile = new ZipFile(zipFilePath);){
            ZipEntry entry = zipFile.getEntry(entryName);
            if (entry == null) {
                throw new IOException("Entry not found in zip: " + entryName);
            }
            Path destFile = Path.of(destPath, new String[0]).getParent();
            if (entry.isDirectory()) {
                Files.createDirectories(destFile, new FileAttribute[0]);
                return;
            }
            Files.createDirectories(destFile.getParent(), new FileAttribute[0]);
            long totalSize = entry.getSize();
            AtomicLong processedBytes = new AtomicLong(0L);
            int lastReportedProgress = 0;
            byte[] buffer = new byte[16384];
            try (InputStream is = zipFile.getInputStream(entry);
                 BufferedInputStream bis = new BufferedInputStream(is, 16384);
                 OutputStream os = Files.newOutputStream(destFile, new OpenOption[0]);
                 BufferedOutputStream bos = new BufferedOutputStream(os, 16384);){
                int bytesRead;
                while ((bytesRead = bis.read(buffer)) > 0) {
                    int currentProgress;
                    bos.write(buffer, 0, bytesRead);
                    long processed = processedBytes.addAndGet(bytesRead);
                    if (progressCallback == null || totalSize <= 0L || (currentProgress = (int)(processed * 100L / totalSize)) == lastReportedProgress) continue;
                    lastReportedProgress = currentProgress;
                    progressCallback.onProgress(processed, totalSize, currentProgress);
                }
            }
            if (entry.getTime() != -1L) {
                Files.setLastModifiedTime(destFile, FileTime.fromMillis(entry.getTime()));
            }
            if (progressCallback != null) {
                progressCallback.onProgress(totalSize, totalSize, 100);
            }
            LOGGER.info("Successfully extracted {} from {}", (Object)entryName, (Object)zipFilePath);
        }
    }

    public CompletableFuture<List<String>> listEntriesAsync(String zipFilePath) {
        return CompletableFuture.supplyAsync(() -> {
            ArrayList<String> entries = new ArrayList<String>();
            try (ZipFile zipFile = new ZipFile(zipFilePath);){
                Enumeration<? extends ZipEntry> enumeration = zipFile.entries();
                while (enumeration.hasMoreElements()) {
                    entries.add(enumeration.nextElement().getName());
                }
            }
            catch (IOException e) {
                LOGGER.error("Failed to list entries", (Throwable)e);
            }
            return entries;
        }, UNZIP_EXECUTOR);
    }

    public static void shutdown() {
        UNZIP_EXECUTOR.shutdown();
    }

    public static interface ProgressCallback {
        public void onProgress(long var1, long var3, int var5);
    }
}

