package com.github.kd_gaming1.packcore.ui.screen.configmanager;

import com.github.kd_gaming1.packcore.PackCore;
import com.github.kd_gaming1.packcore.config.backup.BackupManager;
import com.github.kd_gaming1.packcore.notification.BackupNotifications;
import com.github.kd_gaming1.packcore.ui.screen.base.BasePackCoreScreen;
import com.github.kd_gaming1.packcore.ui.screen.components.ScreenUIComponents;
import io.wispforest.owo.ui.component.ButtonComponent;
import io.wispforest.owo.ui.component.Components;
import io.wispforest.owo.ui.component.LabelComponent;
import io.wispforest.owo.ui.component.TextBoxComponent;
import io.wispforest.owo.ui.container.Containers;
import io.wispforest.owo.ui.container.FlowLayout;
import io.wispforest.owo.ui.core.*;
import java.nio.file.Path;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import net.minecraft.class_156;
import net.minecraft.class_2561;
import net.minecraft.class_2583;
import net.minecraft.class_310;

import static com.github.kd_gaming1.packcore.PackCore.MOD_ID;
import static com.github.kd_gaming1.packcore.PackCore.getModpackInfo;
import static com.github.kd_gaming1.packcore.ui.theme.UITheme.*;

/**
 * Backup management screen - refactored for improved code clarity.
 * Handles backup creation, restoration, and deletion with async operations.
 */
public class BackupManagementScreen extends BasePackCoreScreen {

    private BackupManager.BackupInfo selectedBackup = null;
    private FlowLayout infoPanel;
    private FlowLayout sidebarContent;
    private final Map<BackupManager.BackupInfo, FlowLayout> entryComponents = new HashMap<>();

    // Progress tracking
    private FlowLayout progressDialog = null;
    private LabelComponent progressLabel = null;
    private volatile boolean operationInBackground = false;
    private volatile String currentOperationName = "";
    private volatile boolean isRestoreOperation = false;

    public BackupManagementScreen() {
        super(new ConfigManagerScreen());
    }

    @Override
    protected Component createTitleLabel() {
        return Components.label(
                class_2561.method_43470("Backup Manager - " + getModpackInfo().getName())
                        .method_27694(s -> s.method_27704(net.minecraft.class_2960.method_60655(MOD_ID, "gallaeciaforte")))
        ).color(color(TEXT_PRIMARY));
    }

    @Override
    protected FlowLayout createMainContent() {
        FlowLayout mainContent = Containers.horizontalFlow(Sizing.fill(100), Sizing.expand())
                .gap(8);

        mainContent.child(createSidebar());
        mainContent.child(createInfoPanel());

        return mainContent;
    }

    // ===== Sidebar =====

    private FlowLayout createSidebar() {
        FlowLayout sidebar = ScreenUIComponents.createSidebar(35);

        // Info text
        sidebar.child(createInfoText());

        // Backup sections container
        sidebarContent = Containers.verticalFlow(Sizing.fill(98), Sizing.content())
                .gap(8);

        sidebar.child(ScreenUIComponents.createScrollContainer(sidebarContent));

        // Action buttons
        sidebar.child(createSidebarButtons());

        // Load backups
        rebuildSidebarContent();

        return sidebar;
    }

    private Component createInfoText() {
        int guiScale = class_310.method_1551().field_1690.method_42474().method_41753();
        int padding = guiScale <= 2 ? 16 : 8;

        FlowLayout container = (FlowLayout) Containers.verticalFlow(Sizing.fill(100), Sizing.content())
                .padding(Insets.of(padding, 0, padding, 0));

        LabelComponent infoLabel = (LabelComponent) Components.label(
                        class_2561.method_43470("Manage your configuration backups. Auto backups are created before applying new configs.")
                ).color(color(TEXT_PRIMARY))
                .sizing(Sizing.fill(95), Sizing.content());

        container.child(infoLabel);
        return container;
    }

    private void rebuildSidebarContent() {
        sidebarContent.clearChildren();
        entryComponents.clear();

        // Show loading
        sidebarContent.child(ScreenUIComponents.createEmptyState("Loading backups..."));

        // Load backups asynchronously
        BackupManager.getBackupsAsync().thenAccept(allBackups ->
                class_310.method_1551().execute(() -> {
                    sidebarContent.clearChildren();

                    List<BackupManager.BackupInfo> manualBackups = allBackups.stream()
                            .filter(b -> b.type() == BackupManager.BackupType.MANUAL)
                            .toList();

                    List<BackupManager.BackupInfo> autoBackups = allBackups.stream()
                            .filter(b -> b.type() == BackupManager.BackupType.AUTO)
                            .toList();

                    sidebarContent.child(createBackupSection("Manual Backups", manualBackups, true));
                    sidebarContent.child(createBackupSection("Auto Backups", autoBackups, false));
                })
        );
    }

    private FlowLayout createBackupSection(String title, List<BackupManager.BackupInfo> backups,
                                           boolean isManual) {
        FlowLayout section = ScreenUIComponents.createSection(title, isManual ? 45 : 50);
        section.horizontalSizing(Sizing.fill(98));

        FlowLayout listContent = Containers.verticalFlow(Sizing.fill(100), Sizing.content())
                .gap(2);

        if (backups.isEmpty()) {
            listContent.child(Components.label(class_2561.method_43470("No backups found"))
                    .color(color(TEXT_SECONDARY)));
        } else {
            for (BackupManager.BackupInfo backup : backups) {
                listContent.child(createBackupEntry(backup));
            }
        }

        section.child(ScreenUIComponents.createScrollContainer(listContent));
        return section;
    }

    private FlowLayout createBackupEntry(BackupManager.BackupInfo backup) {
        FlowLayout entry = ScreenUIComponents.createListEntry();

        // Display title
        String displayTitle = backup.title() != null && !backup.title().isEmpty()
                ? backup.title()
                : backup.configName();

        entry.child(Components.label(class_2561.method_43470(displayTitle))
                .color(color(TEXT_PRIMARY)));

        // Badges
        FlowLayout badges = Containers.horizontalFlow(Sizing.fill(100), Sizing.content())
                .gap(4);

        badges.child(Components.label(class_2561.method_43470(backup.type().getDisplayName()))
                .color(color(backup.type() == BackupManager.BackupType.MANUAL
                        ? SUCCESS_BORDER
                        : WARNING_BORDER)));

        badges.child(Components.label(class_2561.method_43470("v" + backup.configVersion()))
                .color(color(TEXT_SECONDARY)));

        entry.child(badges);

        // Store reference
        entryComponents.put(backup, entry);

        // Apply hover and selection
        ScreenUIComponents.applyHoverEffects(entry, () -> selectBackup(backup));

        return entry;
    }

    private FlowLayout createSidebarButtons() {
        FlowLayout buttonRow = (FlowLayout) Containers.ltrTextFlow(Sizing.fill(100), Sizing.content())
                .gap(4)
                .horizontalAlignment(HorizontalAlignment.CENTER);

        buttonRow.child(ScreenUIComponents.createButton("Create Backup",
                        btn -> showCreateBackupDialog(), 90, 19)
                .margins(Insets.bottom(4)));

        buttonRow.child(ScreenUIComponents.createButton("Open Folder",
                        btn -> BackupManager.openBackupsFolder(), 90, 19)
                .margins(Insets.bottom(4)));

        buttonRow.child(ScreenUIComponents.createButton("Refresh",
                        btn -> refreshBackupsList(), 90, 19)
                .margins(Insets.bottom(4)));

        return buttonRow;
    }

    // ===== Info Panel =====

    private FlowLayout createInfoPanel() {
        infoPanel = ScreenUIComponents.createInfoPanel(65);
        showEmptyState();
        return infoPanel;
    }

    private void showEmptyState() {
        infoPanel.clearChildren();
        infoPanel.child(ScreenUIComponents.createEmptyState(
                "Select a backup to view details"));
    }

    private void selectBackup(BackupManager.BackupInfo backup) {
        // Reset previous selection
        if (selectedBackup != null && entryComponents.containsKey(selectedBackup)) {
            ScreenUIComponents.applySelectedState(entryComponents.get(selectedBackup), false);
        }

        // Set new selection
        selectedBackup = backup;
        if (entryComponents.containsKey(backup)) {
            ScreenUIComponents.applySelectedState(entryComponents.get(backup), true);
        }

        showBackupDetails();
    }

    private void showBackupDetails() {
        if (selectedBackup == null) return;

        infoPanel.clearChildren();
        infoPanel.horizontalAlignment(HorizontalAlignment.LEFT);
        infoPanel.verticalAlignment(VerticalAlignment.TOP);

        int padding = class_310.method_1551().field_1690.method_42474().method_41753() <= 2 ? 6 : 0;

        // Header
        String headerText = selectedBackup.title() != null && !selectedBackup.title().isEmpty()
                ? selectedBackup.title()
                : selectedBackup.configName();

        infoPanel.child(Components.label(class_2561.method_43470(headerText)
                        .method_10862(class_2583.field_24360.method_10982(true)))
                .color(color(ACCENT_SECONDARY))
                .margins(Insets.of(padding, 0, 0, 0)));

        // Info box
        FlowLayout infoBox = ScreenUIComponents.createInfoBox();
        infoBox.child(ScreenUIComponents.createInfoRow("Type:", selectedBackup.type().getDisplayName()));
        infoBox.child(ScreenUIComponents.createInfoRow("Config:", selectedBackup.configName()));
        infoBox.child(ScreenUIComponents.createInfoRow("Version:", selectedBackup.configVersion()));
        infoBox.child(ScreenUIComponents.createInfoRow("Created:",
                ScreenUIComponents.formatTimestamp(selectedBackup.timestamp())));
        infoBox.child(ScreenUIComponents.createInfoRow("Size:",
                ScreenUIComponents.formatSize(selectedBackup.sizeBytes())));
        infoBox.child(ScreenUIComponents.createInfoRow("Backup ID:", selectedBackup.backupId()));

        infoPanel.child(infoBox);

        // Description
        if (selectedBackup.description() != null && !selectedBackup.description().trim().isEmpty()) {
            infoPanel.child(Components.label(class_2561.method_43470("Description:")
                            .method_10862(class_2583.field_24360.method_10982(true)))
                    .color(color(ACCENT_SECONDARY)));

            infoPanel.child(Components.label(class_2561.method_43470(selectedBackup.description()))
                    .color(color(TEXT_PRIMARY))
                    .sizing(Sizing.fill(95), Sizing.content()));
        }

        // Warning box
        infoPanel.child(ScreenUIComponents.createWarningCard(
                "Restore Information",
                "Restoring will replace current files. An auto-backup will be created first."
        ));

        // Action buttons
        infoPanel.child(createActionButtons());
    }

    private FlowLayout createActionButtons() {
        FlowLayout buttonPanel = (FlowLayout) Containers.verticalFlow(Sizing.fill(100), Sizing.content())
                .gap(8)
                .horizontalAlignment(HorizontalAlignment.CENTER);

        // Full restore
        buttonPanel.child(ScreenUIComponents.createButton("Restore Full Backup",
                btn -> showRestoreConfirmation(), 120, 20));

        // Selective restore (NEW)
        buttonPanel.child(ScreenUIComponents.createButton("Restore Specific Files",
                btn -> class_310.method_1551().method_1507(
                        new SelectiveFileApplicationScreen(selectedBackup, this)), 120, 20));

        // Delete (if manual)
        if (selectedBackup.type() == BackupManager.BackupType.MANUAL) {
            buttonPanel.child(ScreenUIComponents.createButton("Delete",
                    btn -> showDeleteConfirmation(), 90, 20));
        }

        return buttonPanel;
    }

    // ===== Backup Operations =====

    private void showCreateBackupDialog() {
        FlowLayout dialog = ScreenUIComponents.createDialog(
                "Create Manual Backup",
                null,
                450
        );

        dialog.child(Components.label(class_2561.method_43470("Title:*"))
                .color(color(TEXT_PRIMARY)));

        TextBoxComponent titleField = Components.textBox(Sizing.fill(95), "");
        titleField.method_47404(class_2561.method_43470("Enter backup title"));
        dialog.child(titleField);

        dialog.child(Components.label(class_2561.method_43470("Description (optional):"))
                .color(color(TEXT_PRIMARY)));

        TextBoxComponent descriptionField = Components.textBox(Sizing.fill(95), "");
        descriptionField.method_47404(class_2561.method_43470("Additional details about this backup"));
        dialog.child(descriptionField);

        dialog.child(ScreenUIComponents.createButtonRow(
                ScreenUIComponents.createButton("Create", btn -> {
                    String title = titleField.method_1882().trim();
                    String description = descriptionField.method_1882().trim();
                    closeTopOverlay();
                    performCreateBackup(
                            title.isEmpty() ? null : title,
                            description.isEmpty() ? null : description
                    );
                }),
                ScreenUIComponents.createButton("Cancel", btn -> closeTopOverlay())
        ));

        showOverlay(dialog, false);
    }

    private void performCreateBackup(String title, String description) {
        String finalTitle = title != null ? title : "Manual backup - " +
                java.time.LocalDateTime.now().format(
                        java.time.format.DateTimeFormatter.ofPattern("MM/dd/yyyy HH:mm"));

        currentOperationName = finalTitle;
        operationInBackground = false;
        isRestoreOperation = false;

        showBackupWarningDialog(finalTitle, description);
    }

    private void showBackupWarningDialog(String title, String description) {
        FlowLayout dialog = ScreenUIComponents.createWarningDialog(
                "Backup Notice",
                null,
                400
        );

        FlowLayout warningText = Containers.verticalFlow(Sizing.fill(100), Sizing.content())
                .gap(4);

        warningText.child(Components.label(class_2561.method_43470("⚠ Important Notice:"))
                .color(color(TEXT_PRIMARY))
                .margins(Insets.bottom(4)));

        warningText.child(Components.label(class_2561.method_43470("• The backup will run in the background"))
                .color(color(TEXT_PRIMARY)));

        warningText.child(Components.label(class_2561.method_43470("• A progress indicator will show the status"))
                .color(color(TEXT_PRIMARY)));

        warningText.child(Components.label(class_2561.method_43470("• You can continue using the interface"))
                .color(color(TEXT_PRIMARY))
                .margins(Insets.bottom(8)));

        dialog.child(warningText);

        dialog.child(ScreenUIComponents.createButtonRow(
                ScreenUIComponents.createButton("Cancel", btn -> closeTopOverlay(), 80, 20),
                ScreenUIComponents.createButton("Continue", btn -> {
                    closeTopOverlay();
                    executeBackupCreation(title, description);
                }, 120, 20)
        ));

        showOverlay(dialog, false);
    }

    private void executeBackupCreation(String title, String description) {
        operationInBackground = false;
        showProgressDialog("Creating Backup", "Preparing backup...");

        BackupManager.createManualBackupAsync(title, description, this::updateProgress)
                .thenAccept(backupPath -> class_310.method_1551().execute(() -> {
                    closeProgressDialog();
                    refreshBackupsList();

                    BackupNotifications.notifyBackupComplete(
                            currentOperationName, backupPath, false);

                    // Auto-open folder if still on screen
                    if (class_310.method_1551().field_1755 == this) {
                        try {
                            class_156.method_668().method_672(backupPath.getParent().toFile());
                        } catch (Exception e) {
                            PackCore.LOGGER.warn("Failed to auto-open backup folder", e);
                        }
                    }
                }))
                .exceptionally(throwable -> {
                    class_310.method_1551().execute(() -> {
                        closeProgressDialog();
                        PackCore.LOGGER.error("Failed to create backup", throwable);
                        showErrorDialog("Backup failed: " + throwable.getMessage());
                    });
                    return null;
                });
    }

    private void showRestoreConfirmation() {
        if (selectedBackup == null) return;

        FlowLayout dialog = ScreenUIComponents.createWarningDialog(
                "Restore Backup?",
                null,
                500
        );

        dialog.child(Components.label(class_2561.method_43470("Backup: " + selectedBackup.getDisplayName())
                        .method_10862(class_2583.field_24360.method_10982(true)))
                .color(color(TEXT_PRIMARY)));

        FlowLayout warningBox = (FlowLayout) Containers.verticalFlow(Sizing.fill(100), Sizing.content())
                .gap(4)
                .surface(Surface.flat(ENTRY_BACKGROUND).and(Surface.outline(WARNING_BORDER)))
                .padding(Insets.of(12));

        warningBox.child(Components.label(class_2561.method_43470("This will:"))
                .color(color(TEXT_PRIMARY)));

        warningBox.child(Components.label(class_2561.method_43470("• Replace your current configuration files"))
                .color(color(TEXT_SECONDARY)));

        warningBox.child(Components.label(class_2561.method_43470("• Create an auto-backup of your current state"))
                .color(color(TEXT_SECONDARY)));

        warningBox.child(Components.label(class_2561.method_43470("• Overwrite mod configs and settings"))
                .color(color(TEXT_SECONDARY)));

        dialog.child(warningBox);

        dialog.child(ScreenUIComponents.createButtonRow(
                ScreenUIComponents.createButton("Restore", btn -> {
                    closeTopOverlay();
                    showRestoreWarningDialog();
                }),
                ScreenUIComponents.createButton("Cancel", btn -> closeTopOverlay())
        ));

        showOverlay(dialog, false);
    }

    private void showRestoreWarningDialog() {
        FlowLayout dialog = ScreenUIComponents.createWarningDialog(
                "Restore Notice",
                null,
                400
        );

        FlowLayout warningText = Containers.verticalFlow(Sizing.fill(100), Sizing.content())
                .gap(4);

        warningText.child(Components.label(class_2561.method_43470("⚠ Important Notice:"))
                .color(color(TEXT_PRIMARY))
                .margins(Insets.bottom(4)));

        warningText.child(Components.label(class_2561.method_43470("• The restore will run in the background"))
                .color(color(TEXT_PRIMARY)));

        warningText.child(Components.label(class_2561.method_43470("• A progress indicator will show the status"))
                .color(color(TEXT_PRIMARY)));

        warningText.child(Components.label(class_2561.method_43470("• You can continue using the interface"))
                .color(color(TEXT_PRIMARY))
                .margins(Insets.bottom(8)));

        dialog.child(warningText);

        dialog.child(ScreenUIComponents.createButtonRow(
                ScreenUIComponents.createButton("Cancel", btn -> closeTopOverlay(), 80, 20),
                ScreenUIComponents.createButton("Continue Restore", btn -> {
                    closeTopOverlay();
                    performRestore();
                }, 120, 20)
        ));

        showOverlay(dialog, false);
    }

    private void performRestore() {
        if (selectedBackup == null) return;

        currentOperationName = selectedBackup.getDisplayName();
        operationInBackground = false;
        isRestoreOperation = true;

        showProgressDialog("Restoring Backup", "Preparing restore...");

        BackupManager.restoreBackupAsync(selectedBackup, this::updateProgress)
                .thenAccept(success -> class_310.method_1551().execute(() -> {
                    closeProgressDialog();

                    if (success) {
                        refreshBackupsList();

                        Path gameDir = class_310.method_1551().field_1697.toPath();
                        Path backupsDir = gameDir.resolve("packcore/backups");
                        Path backupPath = backupsDir.resolve(selectedBackup.backupId() + ".zip");

                        BackupNotifications.notifyBackupComplete(
                                currentOperationName, backupPath, true);
                    } else {
                        showErrorDialog("Failed to restore backup!");
                    }
                }))
                .exceptionally(throwable -> {
                    class_310.method_1551().execute(() -> {
                        closeProgressDialog();
                        PackCore.LOGGER.error("Failed to restore backup", throwable);
                        showErrorDialog("Restore failed: " + throwable.getMessage());
                    });
                    return null;
                });
    }

    private void showDeleteConfirmation() {
        if (selectedBackup == null) return;

        FlowLayout dialog = ScreenUIComponents.createDialog(
                "Delete Backup?",
                selectedBackup.getDisplayName() + "\n\nThis action cannot be undone.",
                400
        );

        dialog.surface(Surface.flat(PANEL_BACKGROUND).and(Surface.outline(ERROR_BORDER)));

        dialog.child(ScreenUIComponents.createButtonRow(
                ScreenUIComponents.createButton("Delete", btn -> {
                    closeTopOverlay();
                    performDelete();
                }),
                ScreenUIComponents.createButton("Cancel", btn -> closeTopOverlay())
        ));

        showOverlay(dialog, false);
    }

    private void performDelete() {
        if (selectedBackup == null) return;

        if (BackupManager.deleteBackup(selectedBackup)) {
            PackCore.LOGGER.info("Deleted backup: {}", selectedBackup.getDisplayName());
            selectedBackup = null;
            refreshBackupsList();
        } else {
            showErrorDialog("Failed to delete backup");
        }
    }

    // ===== Progress & Dialogs =====

    private void showProgressDialog(String title, String message) {
        progressDialog = ScreenUIComponents.createDialog(title, null, 350);
        progressDialog.positioning(Positioning.absolute(
                (this.field_22789 - 350) / 2,
                (this.field_22790 - 150) / 2
        ));
        progressDialog.zIndex(20);

        progressLabel = (LabelComponent) Components.label(class_2561.method_43470(message))
                .color(color(TEXT_PRIMARY))
                .margins(Insets.bottom(12));
        progressDialog.child(progressLabel);

        FlowLayout buttonRow = (FlowLayout) Containers.horizontalFlow(Sizing.fill(100), Sizing.content())
                .gap(8)
                .horizontalAlignment(HorizontalAlignment.CENTER);

        ButtonComponent backgroundButton = (ButtonComponent) Components.button(
                class_2561.method_43470("Continue in Background"),
                btn -> {
                    operationInBackground = true;
                    closeProgressDialog();
                }
        ).horizontalSizing(Sizing.content());

        buttonRow.child(backgroundButton);
        progressDialog.child(buttonRow);

        rootComponent.child(progressDialog);
    }

    private void updateProgress(String message) {
        class_310.method_1551().execute(() -> {
            if (progressLabel != null && !operationInBackground) {
                progressLabel.text(class_2561.method_43470(message));
            }
        });
    }

    private void closeProgressDialog() {
        if (progressDialog != null) {
            rootComponent.removeChild(progressDialog);
            progressDialog = null;
            progressLabel = null;
        }
    }

    private void showErrorDialog(String message) {
        FlowLayout dialog = ScreenUIComponents.createDialog("Error", message, 350);
        dialog.surface(Surface.flat(DARK_PANEL_BACKGROUND).and(Surface.outline(ERROR_BORDER)));
        dialog.positioning(Positioning.absolute(
                (this.field_22789 - 350) / 2,
                (this.field_22790 - 120) / 2
        ));
        dialog.zIndex(20);

        dialog.child(ScreenUIComponents.createButton("OK",
                        btn -> rootComponent.removeChild(dialog), 80, 20)
                .horizontalSizing(Sizing.content()));

        rootComponent.child(dialog);
    }

    private void refreshBackupsList() {
        selectedBackup = null;
        showEmptyState();
        rebuildSidebarContent();
    }

    @Override
    public void method_25419() {
        super.method_25419();
    }
}