/*
 * Decompiled with CFR 0.152.
 */
package com.github.kd_gaming1.packcore.util;

import com.github.kd_gaming1.packcore.config.PackCoreConfig;
import com.github.kd_gaming1.packcore.util.ConfigFileOperations;
import com.github.kd_gaming1.packcore.util.ConfigFileUtils;
import com.github.kd_gaming1.packcore.util.ConfigMetadata;
import com.github.kd_gaming1.packcore.util.copysystem.AsyncUnzipFiles;
import com.github.kd_gaming1.packcore.util.copysystem.AsyncZipFiles;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import java.awt.Desktop;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import net.minecraft.class_310;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BackupManager {
    private static final Logger LOGGER = LoggerFactory.getLogger(BackupManager.class);
    private static final Gson GSON = new GsonBuilder().setPrettyPrinting().create();
    private static final DateTimeFormatter TIMESTAMP_FORMAT = DateTimeFormatter.ofPattern("yyyyMMdd_HHmmss");
    private static final String BACKUPS_DIR = "packcore/backups";
    private static final String METADATA_FILE = "backup_metadata.json";
    private static final ExecutorService BACKUP_EXECUTOR = Executors.newCachedThreadPool(r -> {
        Thread thread = new Thread(r);
        thread.setName("BackupManager-" + thread.getId());
        thread.setDaemon(true);
        return thread;
    });
    private static final Set<String> CONFIG_PATHS = Set.of("config", "options.txt", "servers.dat", "resourcepacks", "shaderpacks", "packcore/current_config.json");
    private static final Set<String> EXCLUDED_CONFIG_SUBFOLDERS = Set.of("firmament/profiles", "skyhanni/backup", "skyhanni/repo", "skyblocker/item-repo", "skyocean/data");
    private static final int BATCH_SIZE = 50;

    public static CompletableFuture<Path> createAutoBackupAsync(Consumer<String> progressCallback) {
        if (!PackCoreConfig.enableAutoBackups) {
            LOGGER.debug("Auto-backups are disabled");
            return CompletableFuture.completedFuture(null);
        }
        ConfigMetadata currentConfig = ConfigFileUtils.getCurrentConfig();
        String title = "Auto backup before applying: " + (currentConfig != null ? currentConfig.getName() : "Unknown Config");
        return BackupManager.createBackupAsync(BackupType.AUTO, title, null, progressCallback);
    }

    public static Path createAutoBackup() {
        try {
            return BackupManager.createAutoBackupAsync(msg -> {}).get();
        }
        catch (Exception e) {
            LOGGER.error("Failed to create auto backup", (Throwable)e);
            return null;
        }
    }

    public static void createManualBackup(String title, String description) {
        BackupManager.createManualBackupAsync(title, description, msg -> LOGGER.info("Backup progress: {}", msg));
    }

    public static CompletableFuture<Path> createManualBackupAsync(String title, String description, Consumer<String> progressCallback) {
        return BackupManager.createBackupAsync(BackupType.MANUAL, title, description, progressCallback);
    }

    private static CompletableFuture<Path> createBackupAsync(BackupType type, String title, String description, Consumer<String> progressCallback) {
        return CompletableFuture.supplyAsync(() -> {
            Path path;
            progressCallback.accept("Preparing backup...");
            Path gameDir = class_310.method_1551().field_1697.toPath();
            Path backupsDir = gameDir.resolve(BACKUPS_DIR);
            Files.createDirectories(backupsDir, new FileAttribute[0]);
            String timestamp = LocalDateTime.now().format(TIMESTAMP_FORMAT);
            String backupId = type.name().toLowerCase() + "_" + timestamp;
            Path backupZip = backupsDir.resolve(backupId + ".zip");
            Path tempDir = Files.createTempDirectory("packcore_backup", new FileAttribute[0]);
            try {
                progressCallback.accept("Copying configuration files...");
                BackupManager.copyConfigFilesAsync(gameDir, tempDir, progressCallback).join();
                progressCallback.accept("Calculating backup size...");
                ConfigMetadata currentConfig = ConfigFileUtils.getCurrentConfig();
                long size = BackupManager.calculateDirectorySizeAsync(tempDir).join();
                BackupInfo backupInfo = new BackupInfo(backupId, LocalDateTime.now().format(DateTimeFormatter.ISO_LOCAL_DATE_TIME), type, currentConfig != null ? currentConfig.getName() : "Unknown", currentConfig != null ? currentConfig.getVersion() : "1.0.0", size, title != null ? title : "Manual backup", description);
                Path metadataPath = tempDir.resolve(METADATA_FILE);
                Files.writeString(metadataPath, (CharSequence)GSON.toJson((Object)backupInfo), StandardCharsets.UTF_8, new OpenOption[0]);
                progressCallback.accept("Creating backup archive...");
                AsyncZipFiles zipFiles = new AsyncZipFiles();
                zipFiles.zipDirectoryAsync(tempDir.toFile(), backupZip.toString(), (bytesProcessed, totalBytes, percentage) -> progressCallback.accept(String.format("Zipping: %d%%", percentage))).join();
                LOGGER.info("Created {} backup: {}", (Object)type.getDisplayName().toLowerCase(), (Object)backupZip);
                CompletableFuture.runAsync(() -> BackupManager.cleanupOldBackups(backupsDir), BACKUP_EXECUTOR);
                progressCallback.accept("Backup complete!");
                path = backupZip;
            }
            catch (Throwable throwable) {
                try {
                    CompletableFuture.runAsync(() -> {
                        try {
                            ConfigFileOperations.deleteDirectory(tempDir);
                        }
                        catch (Exception e) {
                            LOGGER.warn("Failed to clean up temp directory", (Throwable)e);
                        }
                    }, BACKUP_EXECUTOR);
                    throw throwable;
                }
                catch (Exception e) {
                    LOGGER.error("Failed to create backup", (Throwable)e);
                    progressCallback.accept("Backup failed: " + e.getMessage());
                    throw new RuntimeException("Backup creation failed", e);
                }
            }
            CompletableFuture.runAsync(() -> {
                try {
                    ConfigFileOperations.deleteDirectory(tempDir);
                }
                catch (Exception e) {
                    LOGGER.warn("Failed to clean up temp directory", (Throwable)e);
                }
            }, BACKUP_EXECUTOR);
            return path;
        }, BACKUP_EXECUTOR);
    }

    private static CompletableFuture<Void> copyConfigFilesAsync(Path gameDir, Path backupDir, Consumer<String> progressCallback) {
        return CompletableFuture.runAsync(() -> {
            try {
                int total = CONFIG_PATHS.size();
                int processed = 0;
                for (String configPath : CONFIG_PATHS) {
                    Path sourcePath = gameDir.resolve(configPath);
                    if (Files.exists(sourcePath, new LinkOption[0])) {
                        Path targetPath = backupDir.resolve(configPath);
                        progressCallback.accept(String.format("Copying %s...", configPath));
                        if (Files.isDirectory(sourcePath, new LinkOption[0])) {
                            BackupManager.copyDirectoryWithExclusionsAsync(sourcePath, targetPath).join();
                        } else {
                            Files.createDirectories(targetPath.getParent(), new FileAttribute[0]);
                            Files.copy(sourcePath, targetPath, StandardCopyOption.REPLACE_EXISTING);
                        }
                    }
                    int percentage = ++processed * 100 / total;
                    progressCallback.accept(String.format("Copying files: %d%%", percentage));
                }
            }
            catch (IOException e) {
                throw new RuntimeException("Failed to copy config files", e);
            }
        }, BACKUP_EXECUTOR);
    }

    private static CompletableFuture<Void> copyDirectoryWithExclusionsAsync(Path source, Path target) {
        return CompletableFuture.runAsync(() -> {
            try {
                List pathsToCopy;
                Files.createDirectories(target, new FileAttribute[0]);
                try (Stream<Path> paths = Files.walk(source, new FileVisitOption[0]);){
                    pathsToCopy = paths.collect(Collectors.toList());
                }
                for (int i = 0; i < pathsToCopy.size(); i += 50) {
                    int end = Math.min(i + 50, pathsToCopy.size());
                    List batch = pathsToCopy.subList(i, end);
                    batch.parallelStream().forEach(sourcePath -> {
                        try {
                            Path relativePath = source.relativize((Path)sourcePath);
                            String relativePathStr = relativePath.toString().replace("\\", "/");
                            for (String excludedFolder : EXCLUDED_CONFIG_SUBFOLDERS) {
                                if (!relativePathStr.startsWith(excludedFolder)) continue;
                                return;
                            }
                            Path targetPath = target.resolve(relativePath);
                            if (Files.isDirectory(sourcePath, new LinkOption[0])) {
                                Files.createDirectories(targetPath, new FileAttribute[0]);
                            } else {
                                Files.createDirectories(targetPath.getParent(), new FileAttribute[0]);
                                Files.copy(sourcePath, targetPath, StandardCopyOption.REPLACE_EXISTING);
                            }
                        }
                        catch (IOException e) {
                            LOGGER.warn("Failed to copy file during backup: {}", sourcePath, (Object)e);
                        }
                    });
                }
            }
            catch (IOException e) {
                throw new RuntimeException("Failed to copy directory", e);
            }
        }, BACKUP_EXECUTOR);
    }

    private static CompletableFuture<Long> calculateDirectorySizeAsync(Path directory) {
        return CompletableFuture.supplyAsync(() -> {
            AtomicLong totalSize = new AtomicLong(0L);
            try (Stream<Path> paths = Files.walk(directory, new FileVisitOption[0]);){
                ((Stream)paths.filter(x$0 -> Files.isRegularFile(x$0, new LinkOption[0])).parallel()).forEach(path -> {
                    try {
                        totalSize.addAndGet(Files.size(path));
                    }
                    catch (IOException e) {
                        LOGGER.debug("Could not get size for: {}", path);
                    }
                });
            }
            catch (IOException e) {
                LOGGER.warn("Failed to calculate directory size", (Throwable)e);
            }
            return totalSize.get();
        }, BACKUP_EXECUTOR);
    }

    public static CompletableFuture<List<BackupInfo>> getBackupsAsync() {
        return CompletableFuture.supplyAsync(() -> {
            try {
                Path gameDir = class_310.method_1551().field_1697.toPath();
                Path backupsDir = gameDir.resolve(BACKUPS_DIR);
                if (!Files.exists(backupsDir, new LinkOption[0])) {
                    return new ArrayList();
                }
                ArrayList backups = new ArrayList();
                try (Stream<Path> backupFiles = Files.list(backupsDir);){
                    backupFiles.filter(path -> path.toString().endsWith(".zip")).forEach(backupZip -> {
                        BackupInfo info = BackupManager.readBackupMetadata(backupZip);
                        if (info != null) {
                            backups.add(info);
                        }
                    });
                }
                backups.sort((a, b) -> b.timestamp.compareTo(a.timestamp));
                return backups;
            }
            catch (IOException e) {
                LOGGER.error("Failed to list backups", (Throwable)e);
                return new ArrayList();
            }
        }, BACKUP_EXECUTOR);
    }

    public static List<BackupInfo> getBackups() {
        try {
            return BackupManager.getBackupsAsync().get();
        }
        catch (Exception e) {
            LOGGER.error("Failed to get backups", (Throwable)e);
            return new ArrayList<BackupInfo>();
        }
    }

    /*
     * Enabled aggressive exception aggregation
     */
    private static BackupInfo readBackupMetadata(Path backupZip) {
        try (ZipFile zip = new ZipFile(backupZip.toFile());){
            BackupInfo backupInfo;
            block15: {
                ZipEntry metadataEntry = zip.getEntry(METADATA_FILE);
                if (metadataEntry == null) {
                    BackupInfo backupInfo2 = BackupManager.createLegacyBackupInfo(backupZip);
                    return backupInfo2;
                }
                InputStream inputStream = zip.getInputStream(metadataEntry);
                try {
                    String json = new String(inputStream.readAllBytes(), StandardCharsets.UTF_8);
                    backupInfo = (BackupInfo)GSON.fromJson(json, BackupInfo.class);
                    if (inputStream == null) break block15;
                }
                catch (Throwable throwable) {
                    if (inputStream != null) {
                        try {
                            inputStream.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                inputStream.close();
            }
            return backupInfo;
        }
        catch (Exception e) {
            LOGGER.warn("Failed to read backup metadata: {}", (Object)backupZip, (Object)e);
            return BackupManager.createLegacyBackupInfo(backupZip);
        }
    }

    private static BackupInfo createLegacyBackupInfo(Path backupZip) {
        try {
            String fileName = backupZip.getFileName().toString();
            String backupId = fileName.replace(".zip", "");
            String timestamp = LocalDateTime.now().format(DateTimeFormatter.ISO_LOCAL_DATE_TIME);
            if (fileName.contains("backup_")) {
                try {
                    String timestampPart = fileName.substring(fileName.lastIndexOf("_") + 1, fileName.lastIndexOf("."));
                    LocalDateTime dateTime = LocalDateTime.parse(timestampPart, TIMESTAMP_FORMAT);
                    timestamp = dateTime.format(DateTimeFormatter.ISO_LOCAL_DATE_TIME);
                }
                catch (Exception timestampPart) {
                    // empty catch block
                }
            }
            long size = Files.size(backupZip);
            return new BackupInfo(backupId, timestamp, BackupType.AUTO, "Legacy Config", "Unknown", size, "Legacy backup (no metadata)", null);
        }
        catch (IOException e) {
            LOGGER.error("Failed to create legacy backup info", (Throwable)e);
            return null;
        }
    }

    public static CompletableFuture<Boolean> restoreBackupAsync(BackupInfo backupInfo, Consumer<String> progressCallback) {
        return CompletableFuture.supplyAsync(() -> {
            Boolean bl;
            Path gameDir = class_310.method_1551().field_1697.toPath();
            Path backupsDir = gameDir.resolve(BACKUPS_DIR);
            Path backupZip = backupsDir.resolve(backupInfo.backupId + ".zip");
            if (!Files.exists(backupZip, new LinkOption[0])) {
                LOGGER.error("Backup file not found: {}", (Object)backupZip);
                progressCallback.accept("Error: Backup file not found");
                return false;
            }
            LOGGER.info("Restoring backup: {}", (Object)backupInfo.getDisplayName());
            progressCallback.accept("Creating safety backup...");
            BackupManager.createAutoBackupAsync(msg -> {}).join();
            progressCallback.accept("Extracting backup...");
            Path tempDir = Files.createTempDirectory("packcore_restore", new FileAttribute[0]);
            try {
                AsyncUnzipFiles unzipper = new AsyncUnzipFiles();
                unzipper.unzipAsync(backupZip.toString(), tempDir.toString(), (bytesProcessed, totalBytes, percentage) -> progressCallback.accept(String.format("Extracting: %d%%", percentage))).join();
                progressCallback.accept("Restoring files...");
                BackupManager.copyRestoredFilesAsync(tempDir, gameDir, progressCallback).join();
                progressCallback.accept("Restore complete!");
                LOGGER.info("Backup restored successfully");
                bl = true;
            }
            catch (Throwable throwable) {
                try {
                    CompletableFuture.runAsync(() -> {
                        try {
                            ConfigFileOperations.deleteDirectory(tempDir);
                        }
                        catch (Exception e) {
                            LOGGER.warn("Failed to clean up temp directory", (Throwable)e);
                        }
                    }, BACKUP_EXECUTOR);
                    throw throwable;
                }
                catch (Exception e) {
                    LOGGER.error("Failed to restore backup", (Throwable)e);
                    progressCallback.accept("Restore failed: " + e.getMessage());
                    return false;
                }
            }
            CompletableFuture.runAsync(() -> {
                try {
                    ConfigFileOperations.deleteDirectory(tempDir);
                }
                catch (Exception e) {
                    LOGGER.warn("Failed to clean up temp directory", (Throwable)e);
                }
            }, BACKUP_EXECUTOR);
            return bl;
        }, BACKUP_EXECUTOR);
    }

    public static boolean restoreBackup(BackupInfo backupInfo) {
        try {
            return BackupManager.restoreBackupAsync(backupInfo, msg -> {}).get();
        }
        catch (Exception e) {
            LOGGER.error("Failed to restore backup", (Throwable)e);
            return false;
        }
    }

    private static CompletableFuture<Void> copyRestoredFilesAsync(Path sourceDir, Path gameDir, Consumer<String> progressCallback) {
        return CompletableFuture.runAsync(() -> {
            try {
                List pathsToRestore;
                try (Stream<Path> paths = Files.walk(sourceDir, new FileVisitOption[0]);){
                    pathsToRestore = paths.filter(path -> !path.equals(sourceDir)).filter(path -> !path.getFileName().toString().equals(METADATA_FILE)).collect(Collectors.toList());
                }
                int total = pathsToRestore.size();
                AtomicLong processed = new AtomicLong(0L);
                for (int i = 0; i < pathsToRestore.size(); i += 50) {
                    int end = Math.min(i + 50, pathsToRestore.size());
                    List batch = pathsToRestore.subList(i, end);
                    batch.parallelStream().forEach(sourcePath -> {
                        try {
                            Path relativePath = sourceDir.relativize((Path)sourcePath);
                            Path targetPath = gameDir.resolve(relativePath);
                            if (Files.isDirectory(sourcePath, new LinkOption[0])) {
                                Files.createDirectories(targetPath, new FileAttribute[0]);
                            } else {
                                Files.createDirectories(targetPath.getParent(), new FileAttribute[0]);
                                Files.copy(sourcePath, targetPath, StandardCopyOption.REPLACE_EXISTING);
                            }
                            long count = processed.incrementAndGet();
                            if (count % 100L == 0L) {
                                int percentage = (int)(count * 100L / (long)total);
                                progressCallback.accept(String.format("Restoring files: %d%%", percentage));
                            }
                        }
                        catch (IOException e) {
                            LOGGER.warn("Failed to restore file: {}", sourcePath, (Object)e);
                        }
                    });
                }
            }
            catch (IOException e) {
                throw new RuntimeException("Failed to restore files", e);
            }
        }, BACKUP_EXECUTOR);
    }

    public static boolean deleteBackup(BackupInfo backupInfo) {
        try {
            Path gameDir = class_310.method_1551().field_1697.toPath();
            Path backupsDir = gameDir.resolve(BACKUPS_DIR);
            Path backupZip = backupsDir.resolve(backupInfo.backupId + ".zip");
            if (Files.exists(backupZip, new LinkOption[0])) {
                Files.delete(backupZip);
                LOGGER.info("Deleted backup: {}", (Object)backupInfo.getDisplayName());
                return true;
            }
            return false;
        }
        catch (IOException e) {
            LOGGER.error("Failed to delete backup", (Throwable)e);
            return false;
        }
    }

    private static void cleanupOldBackups(Path backupsDir) {
        try {
            List<BackupInfo> backups = BackupManager.getBackups();
            List autoBackups = backups.stream().filter(backup -> backup.type == BackupType.AUTO).collect(Collectors.toList());
            if (autoBackups.size() > PackCoreConfig.maxBackups) {
                List toDelete = autoBackups.subList(PackCoreConfig.maxBackups, autoBackups.size());
                for (BackupInfo backup2 : toDelete) {
                    BackupManager.deleteBackup(backup2);
                }
                LOGGER.info("Cleaned up {} old auto backups", (Object)toDelete.size());
            }
        }
        catch (Exception e) {
            LOGGER.error("Failed to cleanup old backups", (Throwable)e);
        }
    }

    public static void openBackupsFolder() {
        CompletableFuture.runAsync(() -> {
            try {
                Path gameDir = class_310.method_1551().field_1697.toPath();
                Path backupsDir = gameDir.resolve(BACKUPS_DIR);
                Files.createDirectories(backupsDir, new FileAttribute[0]);
                Desktop.getDesktop().open(backupsDir.toFile());
            }
            catch (Exception e) {
                LOGGER.error("Failed to open backups folder", (Throwable)e);
            }
        }, BACKUP_EXECUTOR);
    }

    public static void shutdown() {
        BACKUP_EXECUTOR.shutdown();
    }

    public static enum BackupType {
        AUTO("Auto"),
        MANUAL("Manual");

        private final String displayName;

        private BackupType(String displayName) {
            this.displayName = displayName;
        }

        public String getDisplayName() {
            return this.displayName;
        }
    }

    public static class BackupInfo {
        public final String backupId;
        public final String timestamp;
        public final BackupType type;
        public final String configName;
        public final String configVersion;
        public final long sizeBytes;
        public final String title;
        public final String description;

        public BackupInfo(String backupId, String timestamp, BackupType type, String configName, String configVersion, long sizeBytes, String title, String description) {
            this.backupId = backupId;
            this.timestamp = timestamp;
            this.type = type;
            this.configName = configName;
            this.configVersion = configVersion;
            this.sizeBytes = sizeBytes;
            this.title = title;
            this.description = description;
        }

        public String getDisplayName() {
            return String.format("[%s] %s - %s", this.type.getDisplayName(), this.title != null ? this.title : (this.configName != null ? this.configName : "Unknown Config"), this.formatTimestamp());
        }

        private String formatTimestamp() {
            try {
                LocalDateTime dateTime = LocalDateTime.parse(this.timestamp, DateTimeFormatter.ISO_LOCAL_DATE_TIME);
                return dateTime.format(DateTimeFormatter.ofPattern("MM/dd/yyyy HH:mm"));
            }
            catch (Exception e) {
                return this.timestamp;
            }
        }
    }
}

