package com.github.kd_gaming1.packcore.command.packcore;

import com.github.kd_gaming1.packcore.integration.minecraft.PerformanceProfileService;
import com.mojang.brigadier.arguments.StringArgumentType;
import com.mojang.brigadier.builder.LiteralArgumentBuilder;
import com.mojang.brigadier.context.CommandContext;
import net.fabricmc.fabric.api.client.command.v2.ClientCommandManager;
import net.fabricmc.fabric.api.client.command.v2.FabricClientCommandSource;
import net.minecraft.class_124;
import net.minecraft.class_2561;
import net.minecraft.class_310;
import java.util.concurrent.CompletableFuture;

public class PerformanceCommand {

    public static LiteralArgumentBuilder<FabricClientCommandSource> register() {
        return ClientCommandManager.literal("performance")
                .then(ClientCommandManager.literal("list")
                        .executes(PerformanceCommand::listPerformanceProfiles))
                .then(ClientCommandManager.literal("apply")
                        .then(ClientCommandManager.argument("profile", StringArgumentType.word())
                                .suggests((context, builder) -> {
                                    builder.suggest("performance");
                                    builder.suggest("balanced");
                                    builder.suggest("quality");
                                    builder.suggest("shaders");
                                    return builder.buildFuture();
                                })
                                .executes(PerformanceCommand::applyPerformanceProfile)));
    }

    private static int applyPerformanceProfile(CommandContext<FabricClientCommandSource> context) {
        String profileName = StringArgumentType.getString(context, "profile").toLowerCase();

        PerformanceProfileService.PerformanceProfile profile;

        // Map string to enum
        switch (profileName) {
            case "performance" -> profile = PerformanceProfileService.PerformanceProfile.PERFORMANCE;
            case "balanced" -> profile = PerformanceProfileService.PerformanceProfile.BALANCED;
            case "quality" -> profile = PerformanceProfileService.PerformanceProfile.QUALITY;
            case "shaders" -> profile = PerformanceProfileService.PerformanceProfile.SHADERS;
            default -> {
                context.getSource().sendError(class_2561.method_43470("Unknown performance profile: " + profileName)
                        .method_27692(class_124.field_1061));
                context.getSource().sendFeedback(class_2561.method_43470("Available profiles: performance, balanced, quality, shaders")
                        .method_27692(class_124.field_1054));
                return 0;
            }
        }

        context.getSource().sendFeedback(class_2561.method_43470("Applying performance profile: " + profile.getDisplayName() + "...")
                .method_27692(class_124.field_1054));

        // Apply the profile asynchronously to avoid blocking the main thread
        CompletableFuture.runAsync(() -> {
            try {
                PerformanceProfileService.ProfileResult result = PerformanceProfileService.applyPerformanceProfile(profile);

                // Send feedback on main thread
                class_310.method_1551().execute(() -> {
                    if (result.isFullySuccessful()) {
                        context.getSource().sendFeedback(class_2561.method_43470("✓ Performance profile '" + profile.getDisplayName() + "' applied successfully!")
                                .method_27692(class_124.field_1060));

                        // Show what was applied
                        if (result.isVanillaApplied()) {
                            context.getSource().sendFeedback(class_2561.method_43470("  ✓ Minecraft settings applied")
                                    .method_27692(class_124.field_1080));
                        }
                        if (result.isSodiumAvailable() && result.isSodiumApplied()) {
                            context.getSource().sendFeedback(class_2561.method_43470("  ✓ Sodium settings applied")
                                    .method_27692(class_124.field_1080));
                        }
                        if (result.isIrisAvailable() && result.isIrisApplied()) {
                            context.getSource().sendFeedback(class_2561.method_43470("  ✓ Iris/Shader settings applied")
                                    .method_27692(class_124.field_1080));
                        }
                    } else {
                        context.getSource().sendError(class_2561.method_43470("⚠ Performance profile applied with some issues:")
                                .method_27692(class_124.field_1054));

                        if (!result.isVanillaApplied()) {
                            context.getSource().sendError(class_2561.method_43470("  ✗ Failed to apply Minecraft settings")
                                    .method_27692(class_124.field_1061));
                        }
                        if (result.isSodiumAvailable() && !result.isSodiumApplied()) {
                            context.getSource().sendError(class_2561.method_43470("  ✗ Failed to apply Sodium settings")
                                    .method_27692(class_124.field_1061));
                        }
                        if (result.isIrisAvailable() && !result.isIrisApplied()) {
                            context.getSource().sendError(class_2561.method_43470("  ✗ Failed to apply Iris/Shader settings")
                                    .method_27692(class_124.field_1061));
                        }
                    }
                });

            } catch (Exception e) {
                class_310.method_1551().execute(() ->
                        context.getSource().sendError(class_2561.method_43470("✗ Failed to apply performance profile: " + e.getMessage())
                                .method_27692(class_124.field_1061)));
            }
        });

        return 1;
    }

    private static int listPerformanceProfiles(CommandContext<FabricClientCommandSource> context) {
        var availability = PerformanceProfileService.getSystemAvailability();

        context.getSource().sendFeedback(class_2561.method_43470("=== PackCore Performance Profiles ===")
                .method_27692(class_124.field_1065));

        // Show available systems
        context.getSource().sendFeedback(class_2561.method_43470("Available Systems:")
                .method_27692(class_124.field_1054));
        context.getSource().sendFeedback(class_2561.method_43470("  • Minecraft: ✓")
                .method_27692(class_124.field_1060));
        context.getSource().sendFeedback(class_2561.method_43470("  • Sodium: " + (availability.sodiumAvailable() ? "✓" : "✗"))
                .method_27692(availability.sodiumAvailable() ? class_124.field_1060 : class_124.field_1061));
        context.getSource().sendFeedback(class_2561.method_43470("  • Iris/Shaders: " + (availability.irisAvailable() ? "✓" : "✗"))
                .method_27692(availability.irisAvailable() ? class_124.field_1060 : class_124.field_1061));

        context.getSource().sendFeedback(class_2561.method_43470(""));

        // Show available profiles
        context.getSource().sendFeedback(class_2561.method_43470("Available Profiles:")
                .method_27692(class_124.field_1054));

        for (PerformanceProfileService.PerformanceProfile profile : PerformanceProfileService.PerformanceProfile.values()) {
            String command = "/packcore performance apply " + profile.name().toLowerCase();
            context.getSource().sendFeedback(class_2561.method_43470("  • " + profile.getDisplayName() + " - " + profile.getDescription())
                    .method_27692(class_124.field_1068));
            context.getSource().sendFeedback(class_2561.method_43470("    Command: " + command)
                    .method_27692(class_124.field_1080));
        }

        return 1;
    }
}
