package com.github.kd_gaming1.packcore.command.packcore;

import com.github.kd_gaming1.packcore.integration.tabdesign.TabDesignManager;
import com.mojang.brigadier.arguments.StringArgumentType;
import com.mojang.brigadier.builder.LiteralArgumentBuilder;
import com.mojang.brigadier.context.CommandContext;
import net.fabricmc.fabric.api.client.command.v2.ClientCommandManager;
import net.fabricmc.fabric.api.client.command.v2.FabricClientCommandSource;
import net.minecraft.class_124;
import net.minecraft.class_2561;
import net.minecraft.class_310;
import java.util.concurrent.CompletableFuture;

public class TabDesignCommand {

    public static LiteralArgumentBuilder<FabricClientCommandSource> register() {
        return ClientCommandManager.literal("tabdesign")
                .then(ClientCommandManager.literal("list")
                        .executes(TabDesignCommand::listTabDesigns))
                .then(ClientCommandManager.literal("apply")
                        .then(ClientCommandManager.argument("design", StringArgumentType.word())
                                .suggests((context, builder) -> {
                                    builder.suggest("skyhanni");
                                    builder.suggest("skyblocker");
                                    return builder.buildFuture();
                                })
                                .executes(TabDesignCommand::applyTabDesign)));
    }

    private static int applyTabDesign(CommandContext<FabricClientCommandSource> context) {
        String designName = StringArgumentType.getString(context, "design").toLowerCase();

        // Validate the design name
        if (!designName.equals("skyhanni") && !designName.equals("skyblocker")) {
            context.getSource().sendError(class_2561.method_43470("Unknown tab design: " + designName)
                    .method_27692(class_124.field_1061));
            context.getSource().sendFeedback(class_2561.method_43470("Available designs: skyhanni, skyblocker")
                    .method_27692(class_124.field_1054));
            return 0;
        }

        // Check mod availability
        TabDesignManager.TabDesignAvailability availability = TabDesignManager.getAvailability();

        if (designName.equals("skyhanni") && !availability.isSkyHanniAvailable()) {
            context.getSource().sendError(class_2561.method_43470("✗ SkyHanni mod is not installed!")
                    .method_27692(class_124.field_1061));
            return 0;
        }

        if (designName.equals("skyblocker") && !availability.isSkyblockerAvailable()) {
            context.getSource().sendError(class_2561.method_43470("✗ Skyblocker mod is not installed!")
                    .method_27692(class_124.field_1061));
            return 0;
        }

        String displayName = designName.equals("skyhanni") ? "SkyHanni Compact" : "Skyblocker Fancy";
        context.getSource().sendFeedback(class_2561.method_43470("Applying tab design: " + displayName + "...")
                .method_27692(class_124.field_1054));

        // Apply the tab design asynchronously
        CompletableFuture.runAsync(() -> {
            try {
                boolean success = TabDesignManager.applyTabDesign(designName);

                // Send feedback on main thread
                class_310.method_1551().execute(() -> {
                    if (success) {
                        context.getSource().sendFeedback(class_2561.method_43470("✓ Tab design '" + displayName + "' applied successfully!")
                                .method_27692(class_124.field_1060));

                        if (designName.equals("skyhanni")) {
                            context.getSource().sendFeedback(class_2561.method_43470("  ℹ SkyHanni Compact tab list is now active")
                                    .method_27692(class_124.field_1080));
                        } else {
                            context.getSource().sendFeedback(class_2561.method_43470("  ℹ Skyblocker Fancy tab HUD is now active")
                                    .method_27692(class_124.field_1080));
                        }
                    } else {
                        context.getSource().sendError(class_2561.method_43470("⚠ Failed to apply tab design")
                                .method_27692(class_124.field_1054));
                        context.getSource().sendFeedback(class_2561.method_43470("  The mod may not be loaded properly")
                                .method_27692(class_124.field_1080));
                    }
                });

            } catch (Exception e) {
                class_310.method_1551().execute(() ->
                        context.getSource().sendError(class_2561.method_43470("✗ Failed to apply tab design: " + e.getMessage())
                                .method_27692(class_124.field_1061)));
            }
        });

        return 1;
    }

    private static int listTabDesigns(CommandContext<FabricClientCommandSource> context) {
        TabDesignManager.TabDesignAvailability availability = TabDesignManager.getAvailability();

        context.getSource().sendFeedback(class_2561.method_43470("=== PackCore Tab Designs ===")
                .method_27692(class_124.field_1065));

        // Show available mods
        context.getSource().sendFeedback(class_2561.method_43470("Available Mods:")
                .method_27692(class_124.field_1054));
        context.getSource().sendFeedback(class_2561.method_43470("  • SkyHanni: " + (availability.isSkyHanniAvailable() ? "✓" : "✗"))
                .method_27692(availability.isSkyHanniAvailable() ? class_124.field_1060 : class_124.field_1061));
        context.getSource().sendFeedback(class_2561.method_43470("  • Skyblocker: " + (availability.isSkyblockerAvailable() ? "✓" : "✗"))
                .method_27692(availability.isSkyblockerAvailable() ? class_124.field_1060 : class_124.field_1061));

        context.getSource().sendFeedback(class_2561.method_43470(""));

        // Show available designs
        context.getSource().sendFeedback(class_2561.method_43470("Available Designs:")
                .method_27692(class_124.field_1054));

        if (availability.isSkyHanniAvailable()) {
            context.getSource().sendFeedback(class_2561.method_43470("  • SkyHanni Compact - Minimalist compact tab list")
                    .method_27692(class_124.field_1068));
            context.getSource().sendFeedback(class_2561.method_43470("    Command: /packcore tabdesign apply skyhanni")
                    .method_27692(class_124.field_1080));
        }

        if (availability.isSkyblockerAvailable()) {
            context.getSource().sendFeedback(class_2561.method_43470("  • Skyblocker Fancy - Feature-rich tab HUD")
                    .method_27692(class_124.field_1068));
            context.getSource().sendFeedback(class_2561.method_43470("    Command: /packcore tabdesign apply skyblocker")
                    .method_27692(class_124.field_1080));
        }

        if (!availability.isSkyHanniAvailable() && !availability.isSkyblockerAvailable()) {
            context.getSource().sendFeedback(class_2561.method_43470("  ⚠ No compatible tab design mods found")
                    .method_27692(class_124.field_1061));
        }

        return 1;
    }
}
