package com.github.kd_gaming1.packcore.command.scamshield;

import com.github.kd_gaming1.packcore.PackCore;
import com.github.kd_gaming1.packcore.scamshield.debug.ScamShieldDebugger;
import com.github.kd_gaming1.packcore.scamshield.detector.DetectionResult;
import com.mojang.brigadier.arguments.StringArgumentType;
import com.mojang.brigadier.builder.LiteralArgumentBuilder;
import com.mojang.brigadier.context.CommandContext;
import net.fabricmc.fabric.api.client.command.v2.ClientCommandManager;
import net.fabricmc.fabric.api.client.command.v2.FabricClientCommandSource;
import net.minecraft.class_2561;

/**
 * Commands for testing ScamShield detection capabilities.
 */
public class ScamShieldTestCommands {

    public static LiteralArgumentBuilder<FabricClientCommandSource> registerTest() {
        return ClientCommandManager.literal("test")
                .then(ClientCommandManager.argument("message", StringArgumentType.greedyString())
                        .executes(ScamShieldTestCommands::testMessage)
                );
    }

    public static LiteralArgumentBuilder<FabricClientCommandSource> registerDebug() {
        return ClientCommandManager.literal("debug").executes(ScamShieldTestCommands::runDebugTests);
    }

    private static int testMessage(CommandContext<FabricClientCommandSource> context) {
        String message = StringArgumentType.getString(context, "message");
        FabricClientCommandSource source = context.getSource();

        source.sendFeedback(class_2561.method_43470("§e[ScamShield] §7Testing message..."));
        source.sendFeedback(class_2561.method_43470("§7━━━━━━━━━━━━━━━━━━━━━━━━━━━━"));

        try {
            DetectionResult result = PackCore.getScamDetector().analyze(message, "TestUser");

            source.sendFeedback(class_2561.method_43470("§7Message: §f" + message));
            source.sendFeedback(class_2561.method_43470(""));

            if (result.isTriggered()) {
                source.sendFeedback(class_2561.method_43470("§c§l⚠ SCAM DETECTED"));
                source.sendFeedback(
                        class_2561.method_43470("§7Category: §e" + result.getPrimaryCategory().getDisplayName())
                );
            } else {
                source.sendFeedback(class_2561.method_43470("§a✓ No scam detected"));
            }

            source.sendFeedback(class_2561.method_43470(""));
            source.sendFeedback(class_2561.method_43470("§7Score Breakdown:"));
            source.sendFeedback(class_2561.method_43470("§7  Total: §f" + result.getTotalScore()));
            source.sendFeedback(class_2561.method_43470("§7  ScamType: §f" + result.getScamTypeScore()));
            source.sendFeedback(class_2561.method_43470("§7  Progression: §f" + result.getProgressionScore()));

            if (!result.getScamTypeContributions().isEmpty()) {
                source.sendFeedback(class_2561.method_43470(""));
                source.sendFeedback(class_2561.method_43470("§7Detected Patterns:"));
                result.getScamTypeContributions().forEach((type, score) -> {
                    source.sendFeedback(class_2561.method_43470("§7  • §e" + type + "§7: §f" + score + " points"));
                });
            }

            source.sendFeedback(class_2561.method_43470("§7━━━━━━━━━━━━━━━━━━━━━━━━━━━━"));

            return 1;
        } catch (Exception e) {
            source.sendError(class_2561.method_43470("§c[ScamShield] Error: " + e.getMessage()));
            PackCore.LOGGER.error("[ScamShield] Test command error", e);
            return 0;
        }
    }

    private static int runDebugTests(CommandContext<FabricClientCommandSource> context) {
        FabricClientCommandSource source = context.getSource();

        source.sendFeedback(class_2561.method_43470("§e[ScamShield] §7Running debug test suite..."));
        source.sendFeedback(class_2561.method_43470("§7This will take about 30 seconds..."));
        source.sendFeedback(class_2561.method_43470("§7Check console/logs for detailed output!"));
        source.sendFeedback(class_2561.method_43470(""));

        // Run tests asynchronously to avoid blocking
        new Thread(() -> {
            try {
                ScamShieldDebugger debugger = new ScamShieldDebugger();
                ScamShieldDebugger.DebugReport report = debugger.runTests();

                // Send summary to chat
                source.sendFeedback(class_2561.method_43470(""));
                source.sendFeedback(class_2561.method_43470("§7━━━━━━━━━━━━━━━━━━━━━━━━━━━━"));
                source.sendFeedback(class_2561.method_43470("§e[Debug Test Summary]"));
                source.sendFeedback(class_2561.method_43470(""));
                source.sendFeedback(
                        class_2561.method_43470("§7Total Tests: §f" + report.getTotalTests())
                );
                source.sendFeedback(
                        class_2561.method_43470("§a✓ Passed: §f" + report.getPassedTests())
                );
                source.sendFeedback(
                        class_2561.method_43470("§c✗ Failed: §f" + report.getFailedTests())
                );
                source.sendFeedback(
                        class_2561.method_43470("§7Pass Rate: §f" + report.getPassRate() + "%")
                );

                if (report.getPassRate() >= 90) {
                    source.sendFeedback(class_2561.method_43470(""));
                    source.sendFeedback(class_2561.method_43470("§a§l✓ EXCELLENT PERFORMANCE!"));
                } else if (report.getPassRate() >= 75) {
                    source.sendFeedback(class_2561.method_43470(""));
                    source.sendFeedback(class_2561.method_43470("§e⚠ GOOD - Some improvements needed"));
                } else {
                    source.sendFeedback(class_2561.method_43470(""));
                    source.sendFeedback(class_2561.method_43470("§c✗ NEEDS IMPROVEMENT"));
                    source.sendFeedback(class_2561.method_43470("§7Check console for failed tests"));
                }

                source.sendFeedback(class_2561.method_43470("§7━━━━━━━━━━━━━━━━━━━━━━━━━━━━"));

            } catch (Exception e) {
                source.sendError(class_2561.method_43470("§c[ScamShield] Debug test failed: " + e.getMessage()));
                PackCore.LOGGER.error("[ScamShield] Debug test error", e);
            }
        }, "ScamShield-Debug").start();

        return 1;
    }
}