package com.github.kd_gaming1.packcore.ui.screen.title;

import com.github.kd_gaming1.packcore.PackCore;
import com.github.kd_gaming1.packcore.config.PackCoreConfig;
import com.github.kd_gaming1.packcore.ui.surface.effects.TextureSurfaces;
import com.github.kd_gaming1.packcore.ui.screen.configmanager.ConfigManagerScreen;
import com.github.kd_gaming1.packcore.ui.help.guide.GuideListScreen;
import com.github.kd_gaming1.packcore.modpack.ModpackInfo;
import com.github.kd_gaming1.packcore.util.update.modrinth.UpdateCache;
import com.github.kd_gaming1.packcore.util.update.UpdateResult;
import com.github.kd_gaming1.packcore.notification.UpdateNotifier;
import com.github.kd_gaming1.packcore.ui.toast.PackCoreToast;
import com.terraformersmc.modmenu.api.ModMenuApi;
import io.wispforest.lavendermd.MarkdownProcessor;
import io.wispforest.lavendermd.compiler.OwoUICompiler;
import io.wispforest.lavendermd.feature.*;
import io.wispforest.owo.ui.base.BaseOwoScreen;
import io.wispforest.owo.ui.component.*;
import io.wispforest.owo.ui.container.Containers;
import io.wispforest.owo.ui.container.FlowLayout;
import io.wispforest.owo.ui.container.ScrollContainer;
import io.wispforest.owo.ui.core.*;
import org.jetbrains.annotations.NotNull;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import net.minecraft.class_156;
import net.minecraft.class_2561;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_412;
import net.minecraft.class_4185;
import net.minecraft.class_429;
import net.minecraft.class_437;
import net.minecraft.class_500;
import net.minecraft.class_526;
import net.minecraft.class_639;
import net.minecraft.class_642;

import static com.github.kd_gaming1.packcore.PackCore.MOD_ID;
import static com.github.kd_gaming1.packcore.ui.theme.UITheme.*;

/**
 * Improved styled title screen with better component organization
 */
public class SBEStyledTitleScreen extends BaseOwoScreen<FlowLayout> {

    private final class_2960 backgroundTexture = class_2960.method_60655(MOD_ID, "textures/gui/title/main_menu_background.png");
    private static final ModpackInfo info = PackCore.getModpackInfo();

    private static long lastToastTime = 0;
    private static final long TOAST_COOLDOWN_MS = 5 * 60 * 1000;

    // Update state
    private final boolean updateNotificationEnabled =
            info != null &&
                    PackCoreConfig.showUpdateNotificationsOnTitleScreen &&
                    PackCoreConfig.enableUpdateNotifications;
    private boolean updateAvailable;
    private String currentVersion;
    private String newVersion;
    private String changelog;
    private String modrinthName;

    // UI state
    private boolean showChangelog = false;
    private FlowLayout mainButtonLayout;
    private FlowLayout changelogLayout;

    // Cached Markdown processor
    private static final MarkdownProcessor<ParentComponent> MARKDOWN_PROCESSOR =
            new MarkdownProcessor<>(
                    OwoUICompiler::new,
                    new BasicFormattingFeature(),
                    new ColorFeature(),
                    new LinkFeature(),
                    new ListFeature(),
                    new BlockQuoteFeature(),
                    new ImageFeature()
            );

    private static final Map<String, ParentComponent> COMPONENT_CACHE = new ConcurrentHashMap<>();

    @Override
    protected @NotNull OwoUIAdapter<FlowLayout> createAdapter() {
        return OwoUIAdapter.create(this, Containers::verticalFlow);
    }

    @Override
    protected void build(FlowLayout rootComponent) {
        rootComponent.surface(TextureSurfaces.stretched(backgroundTexture, 1920, 1082));

        // Main components
        rootComponent.child(createMainButtonAndTitle()).horizontalAlignment(HorizontalAlignment.CENTER);
        rootComponent.child(createSocialButtons().positioning(Positioning.relative(0, 100)));
        rootComponent.child(createSeeWhatIsNewButtons().positioning(Positioning.relative(100, 0)));
        rootComponent.child(createModpackButtons().positioning(Positioning.relative(100, 100)));

        // Create changelog layout but don't add it initially
        changelogLayout = createChangelogPanel();
    }

    @Override
    public void method_25426() {
        UpdateResult result = checkForUpdates();

        if (result.isSuccess() && result.isUpdateAvailable() && updateNotificationEnabled) {
            long currentTime = System.currentTimeMillis();
            if (currentTime - lastToastTime > TOAST_COOLDOWN_MS && UpdateNotifier.shouldShowMainMenuToast(result.getVersionNumber())) {
                PackCoreToast.showUpdateAvailable(currentVersion, newVersion, modrinthName);
                lastToastTime = currentTime;
            }
        }

        if (!result.isSuccess()) {
            PackCore.LOGGER.warn("Update check failed: {}", result.getErrorMessage());
        }

        super.method_25426();
    }

    /**
     * Create main button area and title
     */
    private FlowLayout createMainButtonAndTitle() {
        FlowLayout buttonAndTitle = (FlowLayout) Containers.verticalFlow(Sizing.fixed(320), Sizing.fill(100))
                .gap(4)
                .padding(Insets.of(4))
                .margins(Insets.of(4, 4, 4, 4));

        // Title texture
        TextureComponent title = (TextureComponent) Components.texture(
                        class_2960.method_60655(MOD_ID, "textures/gui/title/title.png"),
                        0, 0, 1476, 157, 1476, 157
                )
                .margins(Insets.top(8))
                .horizontalSizing(Sizing.fixed(312))
                .verticalSizing(Sizing.fixed(34));

        // Button layout
        mainButtonLayout = (FlowLayout) Containers.verticalFlow(Sizing.fill(100), Sizing.content())
                .gap(8)
                .horizontalAlignment(HorizontalAlignment.CENTER)
                .padding(Insets.of(8))
                .margins(Insets.top(12));

        // Add all buttons
        ButtonComponent joinHypixel = createButton("Join Hypixel", this::joinHypixel);
        ButtonComponent openSingleplayer = createButton("SINGLEPLAYER", this::openSingleplayer);
        ButtonComponent openMultiplayer = createButton("Multiplayer", this::openMultiplayer);
        ButtonComponent openMods = createButton("MODS", this::openMods);
        ButtonComponent openOptions = createButton("OPTIONS", this::openOptions);

        mainButtonLayout
                .child(joinHypixel)
                .child(openSingleplayer)
                .child(openMultiplayer)
                .child(openMods)
                .child(openOptions)
                .child(createButton("QUIT", button -> class_310.method_1551().method_1592()));


        buttonAndTitle.child(title);
        buttonAndTitle.child(mainButtonLayout);

        return buttonAndTitle;
    }

    /**
     * Create a standard button
     */
    private ButtonComponent createButton(String text, ButtonComponent.PressAction action) {
        return (ButtonComponent) Components.button(
                        class_2561.method_43470(text).method_27694(s -> s.method_27704(class_2960.method_60655(MOD_ID, "gallaeciaforte"))),
                        action::onPress
                )
                .renderer(ButtonComponent.Renderer.texture(
                        class_2960.method_60655(MOD_ID, "textures/gui/menu/blank_button.png"), 0, 0, 200, 66))
                .horizontalSizing(Sizing.fixed(200))
                .verticalSizing(Sizing.fixed(22));
    }


    /**
     * Create an icon button
     */
    private ButtonComponent createIconButton(String texture, String tooltip, Runnable action) {
        return (ButtonComponent) Components.button(
                        class_2561.method_43473(),
                        button -> action.run()
                )
                .renderer(ButtonComponent.Renderer.texture(
                        class_2960.method_60655(MOD_ID, texture), 0, 0, 22, 22))
                .horizontalSizing(Sizing.fixed(22))
                .verticalSizing(Sizing.fixed(22))
                .tooltip(class_2561.method_43470(tooltip));
    }

    /**
     * Create social buttons panel
     */
    private FlowLayout createSocialButtons() {
        FlowLayout buttonLayout = (FlowLayout) Containers.verticalFlow(Sizing.content(), Sizing.content())
                .gap(6)
                .horizontalAlignment(HorizontalAlignment.LEFT)
                .padding(Insets.of(4));

        buttonLayout
                .child(createIconButton("textures/gui/menu/discord_icon.png",
                        "Join our Discord server",
                        () -> class_156.method_668().method_670(info.getDiscord())))
                .child(createIconButton("textures/gui/menu/modrinth_icon.png",
                        "Visit the modrinth page",
                        () -> class_156.method_668().method_670(info.getWebsite())))
                .child(createIconButton("textures/gui/menu/github_icon.png",
                        "Report an issue",
                        () -> class_156.method_668().method_670(info.getIssueTracker())))
                .child(createVersionInfo());

        return buttonLayout;
    }

    /**
     * Create version info display
     */
    private FlowLayout createVersionInfo() {
        FlowLayout mainLayout = (FlowLayout) Containers.verticalFlow(Sizing.content(), Sizing.content())
                .gap(4)
                .horizontalAlignment(HorizontalAlignment.LEFT);

        LabelComponent versionLabel = Components.label(
                class_2561.method_43470("Pack Version: " + currentVersion)
                        .method_27694(s -> s.method_27704(class_2960.method_60655(MOD_ID, "gallaeciaforte")))
        ).color(Color.ofArgb(TEXT_DARK));

        mainLayout.child(versionLabel);

        if (updateAvailable) {
            LabelComponent updateAvailableLabel = Components.label(
                    class_2561.method_43470("Update Available: " + newVersion)
                            .method_27694(s -> s.method_27704(class_2960.method_60655(MOD_ID, "gallaeciaforte")))
            ).color(Color.ofArgb(TEXT_DARK));
            mainLayout.child(updateAvailableLabel);
        }

        return mainLayout;
    }

    /**
     * Create see what's new button
     */
    private FlowLayout createSeeWhatIsNewButtons() {
        FlowLayout buttonLayout = (FlowLayout) Containers.verticalFlow(Sizing.content(), Sizing.content())
                .gap(6)
                .horizontalAlignment(HorizontalAlignment.RIGHT)
                .padding(Insets.of(4));

        buttonLayout.child(createIconButton("textures/gui/menu/update_icon.png",
                "See what's new",
                this::toggleChangelog));

        return buttonLayout;
    }

    /**
     * Create modpack buttons
     */
    private FlowLayout createModpackButtons() {
        FlowLayout buttonLayout = (FlowLayout) Containers.verticalFlow(Sizing.content(), Sizing.content())
                .gap(6)
                .horizontalAlignment(HorizontalAlignment.RIGHT)
                .padding(Insets.of(4));

        buttonLayout
                .child(createIconButton("textures/gui/menu/settings_icon.png",
                        "Modpack Settings import/export your config",
                        () -> {
                            assert this.field_22787 != null;
                            this.field_22787.method_1507(new ConfigManagerScreen());
                        }))
                .child(createIconButton("textures/gui/menu/guide_icon.png",
                        "See Guides on how to use the modpack",
                        () -> {
                            assert this.field_22787 != null;
                            this.field_22787.method_1507(new GuideListScreen());
                        }));

        return buttonLayout;
    }

    /**
     * Create changelog panel with help button
     */
    private FlowLayout createChangelogPanel() {
        FlowLayout mainLayout = (FlowLayout) Containers.verticalFlow(Sizing.fill(65), Sizing.fill(75))
                .gap(4)
                .horizontalAlignment(HorizontalAlignment.CENTER)
                .padding(Insets.of(4))
                .surface(TextureSurfaces.stretched(
                        class_2960.method_60655(MOD_ID, "textures/gui/menu/info_box.png"), 1142, 934))
                .margins(Insets.of(4, 4, 4, 4))
                .positioning(Positioning.relative(50, 75));

        // Header section
        FlowLayout changelogInfo = (FlowLayout) Containers.verticalFlow(Sizing.fill(100), Sizing.content())
                .gap(2)
                .padding(Insets.of(6, 0, 8, 8))
                .horizontalAlignment(HorizontalAlignment.CENTER);

        // Determine status text
        String changeLogInfoText;
        if (currentVersion.equals(newVersion)) {
            changeLogInfoText = "You are up to date! See change log for current version below:";
        } else if (compareVersions(currentVersion, newVersion) < 0) {
            changeLogInfoText = "A new version is available! See what's new below:";
        } else {
            changeLogInfoText = "You are using a newer or unknown version.";
        }

        LabelComponent changelogLabel = Components.label(
                class_2561.method_43470(changeLogInfoText)
                        .method_27694(s -> s.method_27704(class_2960.method_60655(MOD_ID, "gallaeciaforte")))
        ).shadow(false);

        // Divider
        FlowLayout divider = (FlowLayout) Containers.horizontalFlow(Sizing.fill(98), Sizing.fill(8))
                .surface(TextureSurfaces.scaledContain(
                        class_2960.method_60655(MOD_ID, "textures/gui/menu/divider.png"), 2401, 96));

        changelogInfo.child(changelogLabel);
        changelogInfo.child(divider);
        mainLayout.child(changelogInfo);

        // Changelog content with markdown
        String changelogContent = changelog != null ? changelog : "No changelog available.";

        // Process markdown
        var markdownComponent = COMPONENT_CACHE.computeIfAbsent(
                changelogContent,
                MARKDOWN_PROCESSOR::process
        );
        markdownComponent.horizontalSizing(Sizing.fill(98));
        markdownComponent.padding(Insets.of(0, 4, 4, 4));

        // Scrollable content
        ScrollContainer<FlowLayout> scrollContainer = Containers.verticalScroll(
                Sizing.fill(98),
                Sizing.expand(),
                (FlowLayout) markdownComponent
        );
        scrollContainer.scrollbar(ScrollContainer.Scrollbar.vanilla());
        scrollContainer.margins(Insets.bottom(10));

        mainLayout.child(scrollContainer);

        // Help button section
        FlowLayout buttonSection = (FlowLayout) Containers.horizontalFlow(Sizing.fill(100), Sizing.content())
                .gap(8)
                .horizontalAlignment(HorizontalAlignment.CENTER)
                .padding(Insets.of(8));

        ButtonComponent helpButton = (ButtonComponent) Components.button(
                        class_2561.method_43470("📚 Open Update Guide"),
                        button -> {
                            // Open the guide screen and close the changelog
                            assert this.field_22787 != null;
                            this.field_22787.method_1507(new GuideListScreen(this));
                            toggleChangelog(); // Hide changelog when opening guides
                        }
                ).renderer(ButtonComponent.Renderer.texture(
                        class_2960.method_60655(MOD_ID, "textures/gui/wizard/button.png"), 0, 0, 180, 60))
                .horizontalSizing(Sizing.fixed(180))
                .verticalSizing(Sizing.fixed(20));

        buttonSection.child(helpButton);
        mainLayout.child(buttonSection);

        return mainLayout;
    }

    /**
     * Toggle changelog visibility
     */
    private void toggleChangelog() {
        showChangelog = !showChangelog;

        if (showChangelog) {
            // Hide main buttons and show changelog
            mainButtonLayout.remove();
            this.uiAdapter.rootComponent.child(changelogLayout);
        } else {
            // Hide changelog and show main buttons
            changelogLayout.remove();
            FlowLayout buttonAndTitle = (FlowLayout) this.uiAdapter.rootComponent.children().getFirst();
            buttonAndTitle.child(mainButtonLayout);
        }
    }

    // ===== Button Actions =====

    private void joinHypixel(class_4185 button) {
        class_310 client = class_310.method_1551();
        class_642 serverInfo = new class_642("Hypixel",
                PackCoreConfig.serverAddressForQuickJoinButton,
                class_642.class_8678.field_45611);
        class_412.method_36877(this, client,
                class_639.method_2950(PackCoreConfig.serverAddressForQuickJoinButton),
                serverInfo, false, null);
    }

    private void openSingleplayer(class_4185 button) {
        class_310.method_1551().method_1507(new class_526(this));
    }

    private void openMultiplayer(class_4185 button) {
        class_310.method_1551().method_1507(new class_500(this));
    }

    private void openMods(class_4185 button) {
        class_310 client = class_310.method_1551();
        class_437 current = client.field_1755;

        try {
            class_437 modsScreen = ModMenuApi.createModsScreen(current);
            client.method_1507(modsScreen);
        } catch (Throwable t) {
            PackCore.LOGGER.error("Failed to open Mod Menu screen", t);
            PackCoreToast.showError("Mod Menu Error", "Could not open Mod Menu");
        }
    }

    private void openOptions(class_4185 button) {
        class_310 client = class_310.method_1551();
        client.method_1507(new class_429(this, client.field_1690));
    }

    // ===== Update Check Logic =====

    public UpdateResult checkForUpdates() {
        UpdateCache updateManager = PackCore.getUpdateManager();
        ModpackInfo info = PackCore.getModpackInfo();

        if (updateManager == null || info == null) {
            PackCore.LOGGER.error("Update system not initialized properly");
            return UpdateResult.error("Update system not initialized properly");
        }

        // Check if the configuration is valid
        if (info.isConfigurationValid()) {
            this.updateAvailable = false;
            this.currentVersion = "";
            this.newVersion = "";
            this.changelog = "";
            this.modrinthName = "";
            PackCore.LOGGER.warn("Skipping update check - configuration not properly set up: {}",
                    info.getValidationError());
            return UpdateResult.error("Configuration not properly set up: " + info.getValidationError());
        }

        UpdateResult result = updateManager.checkForUpdates(info);

        if (!result.isSuccess()) {
            PackCore.LOGGER.error("Update check failed: {}", result.getErrorMessage());
            return result;
        }

        // Update instance variables
        this.updateAvailable = result.isUpdateAvailable();
        this.currentVersion = info.getVersion();
        this.newVersion = result.getVersionNumber();
        this.changelog = result.getChangelog();
        result.getModrinthUrl();
        this.modrinthName = info.getName();

        return result;
    }

    /**
     * Compare version strings
     */
    public static int compareVersions(String v1, String v2) {
        String[] parts1 = v1.replaceAll("[^0-9.]", "").split("\\.");
        String[] parts2 = v2.replaceAll("[^0-9.]", "").split("\\.");

        int maxLength = Math.max(parts1.length, parts2.length);

        for (int i = 0; i < maxLength; i++) {
            int p1 = i < parts1.length ? Integer.parseInt(parts1[i]) : 0;
            int p2 = i < parts2.length ? Integer.parseInt(parts2[i]) : 0;

            if (p1 != p2) {
                return p1 - p2;
            }
        }

        return 0;
    }
}