/*
 * Decompiled with CFR 0.152.
 */
package com.github.kd_gaming1.packcore.config.backup;

import com.github.kd_gaming1.packcore.PackCore;
import com.github.kd_gaming1.packcore.config.PackCoreConfig;
import com.github.kd_gaming1.packcore.config.backup.BackupManager;
import com.github.kd_gaming1.packcore.ui.toast.PackCoreToast;
import com.github.kd_gaming1.packcore.util.GsonUtils;
import com.google.gson.Gson;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import net.fabricmc.loader.api.FabricLoader;
import net.minecraft.class_310;

public class ScheduledBackupManager {
    private static final Gson GSON = GsonUtils.GSON;
    private static final String LAST_BACKUP_FILE = "packcore/last_auto_backup.json";
    private static final String LAST_ACTIVE_FILE = "packcore/last_active.json";
    private static final long THREE_DAYS_MILLIS = 259200000L;
    private static ScheduledExecutorService scheduledExecutor;

    public static void initialize() {
        if (scheduledExecutor != null) {
            PackCore.LOGGER.debug("[ScheduledBackupManager] Scheduled backup manager already initialized");
            return;
        }
        scheduledExecutor = Executors.newSingleThreadScheduledExecutor(r -> {
            Thread thread = new Thread(r);
            thread.setName("ScheduledBackupManager");
            thread.setDaemon(true);
            return thread;
        });
        ScheduledBackupManager.updateLastActive();
        scheduledExecutor.scheduleAtFixedRate(ScheduledBackupManager::checkAndPerformScheduledBackup, 1L, 1L, TimeUnit.HOURS);
        PackCore.LOGGER.info("[ScheduledBackupManager] Scheduled backup manager initialized (checks every hour, backups every 3 days)");
    }

    public static void updateLastActive() {
        try {
            Path gameDir = ScheduledBackupManager.getGameDirectorySafe();
            Path activeFile = gameDir.resolve(LAST_ACTIVE_FILE);
            long currentTime = System.currentTimeMillis();
            Files.createDirectories(activeFile.getParent(), new FileAttribute[0]);
            Files.writeString(activeFile, (CharSequence)String.valueOf(currentTime), StandardCharsets.UTF_8, new OpenOption[0]);
            PackCore.LOGGER.debug("[ScheduledBackupManager] Updated last active timestamp: {}", (Object)currentTime);
        }
        catch (Exception e) {
            PackCore.LOGGER.error("[ScheduledBackupManager] Failed to update last active timestamp", (Throwable)e);
        }
    }

    private static void checkAndPerformScheduledBackup() {
        if (!PackCoreConfig.enableAutoBackups) {
            PackCore.LOGGER.debug("[ScheduledBackupManager] Auto backups disabled, skipping scheduled backup check");
            return;
        }
        try {
            Path gameDir = ScheduledBackupManager.getGameDirectorySafe();
            Path trackingFile = gameDir.resolve(LAST_BACKUP_FILE);
            BackupTracking tracking = BackupTracking.loadOrCreate(trackingFile);
            long currentTime = System.currentTimeMillis();
            long timeSinceLastBackup = currentTime - tracking.lastBackupTimestamp;
            if (timeSinceLastBackup < 259200000L) {
                long hoursRemaining = (259200000L - timeSinceLastBackup) / 3600000L;
                PackCore.LOGGER.debug("[ScheduledBackupManager] Not time for scheduled backup yet.  Hours remaining: {}", (Object)hoursRemaining);
                return;
            }
            if (!ScheduledBackupManager.hasBeenActiveInPastThreeDays()) {
                PackCore.LOGGER.debug("[ScheduledBackupManager] User has not been active in past 3 days, skipping scheduled backup");
                return;
            }
            PackCore.LOGGER.info("[ScheduledBackupManager] Performing scheduled automatic backup");
            String title = "Scheduled backup - " + LocalDateTime.now().format(DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm"));
            String description = "Automatic backup created after 3 days of activity";
            ((CompletableFuture)BackupManager.createBackupAsync(gameDir, BackupManager.BackupType.AUTO, title, description, msg -> PackCore.LOGGER.debug("[ScheduledBackupManager] Scheduled backup: {}", msg)).thenAccept(backupPath -> {
                if (backupPath != null) {
                    BackupTracking newTracking = new BackupTracking(System.currentTimeMillis(), tracking.lastActiveTimestamp);
                    newTracking.save(trackingFile);
                    PackCore.LOGGER.info("[ScheduledBackupManager] Scheduled backup completed: {}", (Object)backupPath.getFileName());
                    class_310 client = class_310.method_1551();
                    if (client != null) {
                        client.execute(() -> PackCoreToast.showBackupComplete("Scheduled Config Backup", backupPath.getFileName().toString(), false));
                    }
                }
            })).exceptionally(ex -> {
                PackCore.LOGGER.error("[ScheduledBackupManager] Scheduled backup failed", ex);
                class_310 client = class_310.method_1551();
                if (client != null) {
                    client.execute(() -> PackCoreToast.showError("Scheduled Config Backup Failed", "Check logs for details"));
                }
                return null;
            });
        }
        catch (Exception e) {
            PackCore.LOGGER.error("[ScheduledBackupManager] Failed to check/perform scheduled backup", (Throwable)e);
        }
    }

    private static boolean hasBeenActiveInPastThreeDays() {
        try {
            Path gameDir = ScheduledBackupManager.getGameDirectorySafe();
            Path activeFile = gameDir.resolve(LAST_ACTIVE_FILE);
            if (!Files.exists(activeFile, new LinkOption[0])) {
                PackCore.LOGGER.debug("[ScheduledBackupManager] No activity file found, considering user active");
                return true;
            }
            String content = Files.readString(activeFile, StandardCharsets.UTF_8);
            long lastActive = Long.parseLong(content.trim());
            long currentTime = System.currentTimeMillis();
            long daysSinceActive = (currentTime - lastActive) / 86400000L;
            PackCore.LOGGER.debug("[ScheduledBackupManager] Days since last active: {}", (Object)daysSinceActive);
            return daysSinceActive <= 3L;
        }
        catch (Exception e) {
            PackCore.LOGGER.warn("[ScheduledBackupManager] Failed to check activity, assuming user is active", (Throwable)e);
            return true;
        }
    }

    public static long getTimeUntilNextBackup() {
        try {
            Path gameDir = ScheduledBackupManager.getGameDirectorySafe();
            Path trackingFile = gameDir.resolve(LAST_BACKUP_FILE);
            BackupTracking tracking = BackupTracking.loadOrCreate(trackingFile);
            long currentTime = System.currentTimeMillis();
            long timeSinceLastBackup = currentTime - tracking.lastBackupTimestamp;
            if (timeSinceLastBackup >= 259200000L) {
                return 0L;
            }
            return 259200000L - timeSinceLastBackup;
        }
        catch (Exception e) {
            PackCore.LOGGER.warn("[ScheduledBackupManager] Failed to calculate time until next backup", (Throwable)e);
            return -1L;
        }
    }

    private static Path getGameDirectorySafe() {
        class_310 client = class_310.method_1551();
        if (client != null && client.field_1697 != null) {
            return client.field_1697.toPath();
        }
        return FabricLoader.getInstance().getGameDir();
    }

    public static void shutdown() {
        if (scheduledExecutor != null) {
            PackCore.LOGGER.info("[ScheduledBackupManager] Shutting down scheduled backup manager");
            scheduledExecutor.shutdown();
            try {
                if (!scheduledExecutor.awaitTermination(5L, TimeUnit.SECONDS)) {
                    scheduledExecutor.shutdownNow();
                }
            }
            catch (InterruptedException e) {
                scheduledExecutor.shutdownNow();
                Thread.currentThread().interrupt();
            }
            scheduledExecutor = null;
        }
    }

    private record BackupTracking(long lastBackupTimestamp, long lastActiveTimestamp) {
        public static BackupTracking loadOrCreate(Path filePath) {
            try {
                if (Files.exists(filePath, new LinkOption[0])) {
                    String json = Files.readString(filePath, StandardCharsets.UTF_8);
                    return (BackupTracking)GSON.fromJson(json, BackupTracking.class);
                }
            }
            catch (Exception e) {
                PackCore.LOGGER.warn("[ScheduledBackupManager] Failed to load backup tracking", (Throwable)e);
            }
            return new BackupTracking(0L, 0L);
        }

        public void save(Path filePath) {
            try {
                Files.createDirectories(filePath.getParent(), new FileAttribute[0]);
                Files.writeString(filePath, (CharSequence)GSON.toJson((Object)this), StandardCharsets.UTF_8, new OpenOption[0]);
            }
            catch (IOException e) {
                PackCore.LOGGER.error("[ScheduledBackupManager] Failed to save backup tracking", (Throwable)e);
            }
        }
    }
}

