/*
 * Decompiled with CFR 0.152.
 */
package com.github.kd_gaming1.packcore.config.imports;

import com.github.kd_gaming1.packcore.config.apply.ConfigApplyService;
import com.github.kd_gaming1.packcore.config.model.ConfigMetadata;
import com.github.kd_gaming1.packcore.config.storage.ConfigFileRepository;
import com.github.kd_gaming1.packcore.util.task.ProgressListener;
import java.awt.Desktop;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Stream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import net.minecraft.class_310;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ConfigImportService {
    private static final Logger LOGGER = LoggerFactory.getLogger(ConfigImportService.class);
    private static final DateTimeFormatter TIMESTAMP_FORMAT = DateTimeFormatter.ofPattern("yyyyMMdd_HHmmss");
    public static final String IMPORTS_FOLDER = "packcore/imports";

    public static Path getImportsFolder() {
        Path gameDir = class_310.method_1551().field_1697.toPath();
        Path importsDir = gameDir.resolve(IMPORTS_FOLDER);
        try {
            Files.createDirectories(importsDir, new FileAttribute[0]);
        }
        catch (IOException e) {
            LOGGER.error("Failed to create imports directory", (Throwable)e);
        }
        return importsDir;
    }

    public static CompletableFuture<Boolean> openImportsFolder() {
        return CompletableFuture.supplyAsync(() -> {
            try {
                Desktop desktop;
                Path importsFolder = ConfigImportService.getImportsFolder();
                if (!Files.exists(importsFolder, new LinkOption[0])) {
                    Files.createDirectories(importsFolder, new FileAttribute[0]);
                }
                if (Desktop.isDesktopSupported() && (desktop = Desktop.getDesktop()).isSupported(Desktop.Action.OPEN)) {
                    desktop.open(importsFolder.toFile());
                    LOGGER.info("Opened imports folder: {}", (Object)importsFolder);
                    return true;
                }
                LOGGER.warn("Desktop operations not supported on this system");
                return false;
            }
            catch (IOException e) {
                LOGGER.error("Failed to open imports folder", (Throwable)e);
                return false;
            }
        });
    }

    public static CompletableFuture<List<ImportableFile>> scanImportsFolder() {
        return CompletableFuture.supplyAsync(() -> {
            ArrayList importableFiles = new ArrayList();
            Path importsFolder = ConfigImportService.getImportsFolder();
            if (!Files.exists(importsFolder, new LinkOption[0])) {
                LOGGER.warn("Imports folder does not exist: {}", (Object)importsFolder);
                return importableFiles;
            }
            try (Stream<Path> files = Files.list(importsFolder);){
                files.filter(path -> path.toString().toLowerCase().endsWith(".zip")).forEach(path -> {
                    try {
                        long size = Files.size(path);
                        LocalDateTime modified = LocalDateTime.ofInstant(Files.getLastModifiedTime(path, new LinkOption[0]).toInstant(), ZoneId.systemDefault());
                        ValidationResult validation = ConfigImportService.validateConfigZip(path);
                        ConfigMetadata metadata = null;
                        if (validation.isValid()) {
                            metadata = ConfigFileRepository.readMetadataFromZip(path);
                        }
                        ImportableFile importableFile = new ImportableFile((Path)path, path.getFileName().toString(), size, modified, metadata, validation);
                        importableFiles.add(importableFile);
                    }
                    catch (IOException e) {
                        LOGGER.error("Failed to process file: {}", path, (Object)e);
                    }
                });
            }
            catch (IOException e) {
                LOGGER.error("Failed to scan imports folder", (Throwable)e);
            }
            importableFiles.sort((a, b) -> b.lastModified().compareTo(a.lastModified()));
            return importableFiles;
        });
    }

    public static ConfigMetadata previewConfig(Path configPath) {
        if (configPath == null || !Files.exists(configPath, new LinkOption[0])) {
            LOGGER.error("Config file does not exist: {}", (Object)configPath);
            return null;
        }
        if (!configPath.toString().toLowerCase().endsWith(".zip")) {
            LOGGER.error("File is not a zip: {}", (Object)configPath);
            return null;
        }
        ValidationResult validation = ConfigImportService.validateConfigZip(configPath);
        if (!validation.isValid) {
            LOGGER.error("Config validation failed: {}", (Object)validation.errorMessage);
            return null;
        }
        return ConfigFileRepository.readMetadataFromZip(configPath);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static ValidationResult validateConfigZip(Path zipPath) {
        if (zipPath == null) return ValidationResult.invalid("File does not exist");
        if (!Files.exists(zipPath, new LinkOption[0])) {
            return ValidationResult.invalid("File does not exist");
        }
        if (!zipPath.toString().toLowerCase().endsWith(".zip")) {
            return ValidationResult.invalid("File must be a .zip file");
        }
        try {
            if (Files.size(zipPath) == 0L) {
                return ValidationResult.invalid("File is empty");
            }
        }
        catch (IOException e) {
            return ValidationResult.invalid("Cannot read file size");
        }
        try (ZipFile zipFile = new ZipFile(zipPath.toFile());){
            ValidationResult validationResult;
            boolean hasMetadata = false;
            boolean hasJarFiles = false;
            Enumeration<? extends ZipEntry> entries = zipFile.entries();
            while (entries.hasMoreElements()) {
                ZipEntry entry = entries.nextElement();
                String entryName = entry.getName();
                if (entryName.equals("packcore_metadata.json") || entryName.endsWith("/packcore_metadata.json")) {
                    hasMetadata = true;
                }
                if (!entryName.toLowerCase().endsWith(".jar")) continue;
                hasJarFiles = true;
                LOGGER.warn("Found .jar file in config: {}", (Object)entryName);
            }
            if (!hasMetadata) {
                validationResult = ValidationResult.invalid("Invalid config file: Missing packcore_metadata.json\n\nThis ZIP must contain configuration metadata to be imported.");
                return validationResult;
            }
            if (hasJarFiles) {
                validationResult = ValidationResult.invalid("Invalid config file: Contains .jar files\n\nConfiguration files should not contain mod .jar files.\nPlease use config files only.");
                return validationResult;
            }
            validationResult = ValidationResult.valid();
            return validationResult;
        }
        catch (IOException e) {
            LOGGER.error("Failed to validate config zip", (Throwable)e);
            return ValidationResult.invalid("Cannot read ZIP file: " + e.getMessage());
        }
    }

    public static void importConfig(Path sourceFile, boolean applyImmediately, ProgressListener callback) {
        if (callback == null) {
            callback = new ProgressListener(){

                @Override
                public void onProgress(String msg, int pct) {
                }

                @Override
                public void onComplete(boolean success, String msg) {
                }
            };
        }
        ProgressListener finalCallback = callback;
        CompletableFuture.runAsync(() -> {
            try {
                finalCallback.onProgress("Validating file...", 10);
                ValidationResult validation = ConfigImportService.validateConfigZip(sourceFile);
                if (!validation.isValid) {
                    finalCallback.onComplete(false, validation.errorMessage);
                    return;
                }
                finalCallback.onProgress("Reading metadata...", 30);
                ConfigMetadata metadata = ConfigFileRepository.readMetadataFromZip(sourceFile);
                if (metadata == null || !metadata.isValid()) {
                    finalCallback.onComplete(false, "Could not read config metadata or metadata is invalid");
                    return;
                }
                finalCallback.onProgress("Copying file...", 50);
                Path destination = ConfigImportService.copyToCustomConfigs(sourceFile, metadata);
                if (destination == null) {
                    finalCallback.onComplete(false, "Failed to copy config file");
                    return;
                }
                finalCallback.onProgress("Import complete", 80);
                if (applyImmediately) {
                    finalCallback.onProgress("Scheduling restart...", 90);
                    ConfigFileRepository.ConfigFile configFile = new ConfigFileRepository.ConfigFile(destination.getFileName().toString(), destination, false, metadata);
                    ConfigApplyService.scheduleConfigApplication(configFile);
                    finalCallback.onComplete(true, "Config imported and will be applied on restart.");
                } else {
                    finalCallback.onComplete(true, "Config imported successfully: " + metadata.getName());
                }
            }
            catch (Exception e) {
                LOGGER.error("Failed to import config", (Throwable)e);
                finalCallback.onComplete(false, "Import failed: " + e.getMessage());
            }
        });
    }

    public static void importConfigAndCleanup(final Path sourceFile, boolean applyImmediately, final boolean deleteAfterImport, final ProgressListener callback) {
        ProgressListener wrappedCallback = new ProgressListener(){

            @Override
            public void onProgress(String msg, int pct) {
                if (callback != null) {
                    callback.onProgress(msg, pct);
                }
            }

            @Override
            public void onComplete(boolean success, String msg) {
                if (success && deleteAfterImport) {
                    try {
                        Files.deleteIfExists(sourceFile);
                        LOGGER.info("Deleted imported file from imports folder: {}", (Object)sourceFile);
                    }
                    catch (IOException e) {
                        LOGGER.warn("Failed to delete imported file: {}", (Object)sourceFile, (Object)e);
                    }
                }
                if (callback != null) {
                    callback.onComplete(success, msg);
                }
            }
        };
        ConfigImportService.importConfig(sourceFile, applyImmediately, wrappedCallback);
    }

    private static Path copyToCustomConfigs(Path sourceFile, ConfigMetadata metadata) {
        try {
            Path gameDir = class_310.method_1551().field_1697.toPath();
            Path customConfigsDir = gameDir.resolve("packcore/modpack_config/custom_configs");
            Files.createDirectories(customConfigsDir, new FileAttribute[0]);
            String baseName = metadata.getName().replaceAll("[^a-zA-Z0-9\\-_]", "_");
            String timestamp = LocalDateTime.now().format(TIMESTAMP_FORMAT);
            String fileName = baseName + "_" + timestamp + ".zip";
            Path destination = customConfigsDir.resolve(fileName);
            int counter = 1;
            while (Files.exists(destination, new LinkOption[0])) {
                fileName = baseName + "_" + timestamp + "_" + counter + ".zip";
                destination = customConfigsDir.resolve(fileName);
                ++counter;
            }
            Files.copy(sourceFile, destination, StandardCopyOption.COPY_ATTRIBUTES);
            LOGGER.info("Config imported to: {}", (Object)destination);
            return destination;
        }
        catch (IOException e) {
            LOGGER.error("Failed to copy config file", (Throwable)e);
            return null;
        }
    }

    public static boolean configExists(String configName) {
        if (configName == null || configName.isBlank()) {
            return false;
        }
        String sanitizedName = configName.replaceAll("[^a-zA-Z0-9\\-_]", "_").toLowerCase();
        return ConfigFileRepository.getAllConfigs().stream().anyMatch(config -> config.fileName().toLowerCase().contains(sanitizedName));
    }

    public static boolean deleteImportFile(Path filePath) {
        try {
            Files.deleteIfExists(filePath);
            LOGGER.info("Deleted import file: {}", (Object)filePath);
            return true;
        }
        catch (IOException e) {
            LOGGER.error("Failed to delete import file: {}", (Object)filePath, (Object)e);
            return false;
        }
    }

    public static String getImportsFolderPath() {
        return ConfigImportService.getImportsFolder().toAbsolutePath().toString();
    }

    public record ValidationResult(boolean isValid, String errorMessage) {
        public static ValidationResult valid() {
            return new ValidationResult(true, null);
        }

        public static ValidationResult invalid(String message) {
            return new ValidationResult(false, message);
        }
    }

    public record ImportableFile(Path path, String fileName, long fileSize, LocalDateTime lastModified, ConfigMetadata metadata, ValidationResult validation) {
        public boolean isValid() {
            return this.validation.isValid();
        }

        public String getDisplayName() {
            if (this.metadata != null && this.metadata.getName() != null) {
                return this.metadata.getName();
            }
            return this.fileName;
        }

        public String getFileSizeFormatted() {
            if (this.fileSize < 1024L) {
                return this.fileSize + " B";
            }
            if (this.fileSize < 0x100000L) {
                return String.format("%.1f KB", (double)this.fileSize / 1024.0);
            }
            return String.format("%.1f MB", (double)this.fileSize / 1048576.0);
        }
    }
}

