package com.github.kd_gaming1.packcore.ui.toast;

import io.wispforest.owo.ui.base.BaseOwoToast;
import io.wispforest.owo.ui.component.Components;
import io.wispforest.owo.ui.component.LabelComponent;
import io.wispforest.owo.ui.container.Containers;
import io.wispforest.owo.ui.container.FlowLayout;
import io.wispforest.owo.ui.core.*;
import java.util.ArrayList;
import java.util.List;
import net.minecraft.class_124;
import net.minecraft.class_2561;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_368;
import net.minecraft.class_374;
/**
 * Improved toast system using builder pattern for easy toast creation
 */
public class PackCoreToast extends BaseOwoToast<FlowLayout> {

    // Constants
    private static final int DEFAULT_WIDTH = 280;
    private static final int DEFAULT_PADDING = 12;
    private static final long DEFAULT_DURATION_MS = 10000;

    /**
     * Toast type enum for common toast styles
     */
    public enum ToastType {
        SUCCESS(0xFF_55FF55, "✓"),
        WARNING(0xFF_FFD700, "⚠"),
        ERROR(0xFF_FF5555, "✗"),
        INFO(0xFF_5555FF, "ℹ"),
        UPDATE(0xFF_FFD700, "↑");

        private final int borderColor;
        private final String icon;

        ToastType(int borderColor, String icon) {
            this.borderColor = borderColor;
            this.icon = icon;
        }
    }

    private PackCoreToast(Builder builder) {
        super(
                () -> createContent(builder),
                createTimeoutPredicate(System.currentTimeMillis(), builder.duration)
        );
    }

    /**
     * Create the timeout predicate
     */
    private static VisibilityPredicate<FlowLayout> createTimeoutPredicate(long startTime, long duration) {
        return (toast, time) -> {
            long elapsed = System.currentTimeMillis() - startTime;
            return elapsed < duration ? class_368.class_369.field_2210 : class_368.class_369.field_2209;
        };
    }

    /**
     * Create the toast content from builder
     */
    private static FlowLayout createContent(Builder builder) {
        FlowLayout container = Containers.verticalFlow(
                Sizing.fixed(builder.width),
                Sizing.content()
        );

        // Add icon if present
        if (builder.icon != null) {
            FlowLayout iconRow = (FlowLayout) Containers.horizontalFlow(Sizing.fill(100), Sizing.content())
                    .gap(6)
                    .verticalAlignment(VerticalAlignment.CENTER);

            // Icon texture or text
            if (builder.iconTexture != null) {
                iconRow.child(Components.texture(
                        builder.iconTexture,
                        0, 0, builder.iconSize, builder.iconSize,
                        builder.iconSize, builder.iconSize
                ));
            } else if (builder.icon != null) {
                iconRow.child(Components.label(
                        class_2561.method_43470(builder.icon).method_27692(class_124.field_1067)
                ).color(Color.ofRgb(builder.borderColor)));
            }

            // Title next to icon
            if (builder.title != null) {
                iconRow.child(builder.title);
            }

            container.child(iconRow);
        } else if (builder.title != null) {
            // Title without icon
            container.child(builder.title);
        }

        // Add all lines
        for (LabelComponent line : builder.lines) {
            container.child(line);
        }

        // Apply styling
        container
                .gap(builder.gap)
                .padding(Insets.of(builder.padding))
                .surface(Surface.flat(builder.backgroundColor))
                .horizontalAlignment(builder.horizontalAlignment)
                .verticalAlignment(VerticalAlignment.TOP);

        // Add border if specified
        if (builder.borderColor != 0) {
            FlowLayout borderContainer = Containers.verticalFlow(
                    Sizing.content(),
                    Sizing.content()
            );
            borderContainer
                    .child(container)
                    .surface(Surface.outline(builder.borderColor))
                    .padding(Insets.of(1));
            return borderContainer;
        }

        return container;
    }

    /**
     * Show this toast
     */
    public void show() {
        class_374 toastManager = class_310.method_1551().method_1566();
        toastManager.method_1999(this);
    }

    /**
     * Builder class for creating toasts
     */
    public static class Builder {
        // Required
        private LabelComponent title;

        // Optional with defaults
        private final List<LabelComponent> lines = new ArrayList<>();
        private ToastType type = ToastType.INFO;
        private int borderColor = ToastType.INFO.borderColor;
        private String icon = null;
        private class_2960 iconTexture = null;
        private int iconSize = 16;
        private int width = DEFAULT_WIDTH;
        private int padding = DEFAULT_PADDING;
        private int gap = 2;
        private long duration = DEFAULT_DURATION_MS;
        private int backgroundColor = 0xC0_000000;
        private HorizontalAlignment horizontalAlignment = HorizontalAlignment.LEFT;

        /**
         * Set the toast type (applies default styling)
         */
        public Builder type(ToastType type) {
            this.type = type;
            this.borderColor = type.borderColor;
            this.icon = type.icon;
            return this;
        }

        /**
         * Set the title text
         */
        public Builder title(String text) {
            this.title = Components.label(class_2561.method_43470(text).method_27695(class_124.field_1068, class_124.field_1067));
            this.title.horizontalTextAlignment(HorizontalAlignment.LEFT);
            return this;
        }

        /**
         * Set the title text with formatting
         */
        public Builder title(class_2561 text) {
            this.title = Components.label(text);
            this.title.horizontalTextAlignment(HorizontalAlignment.LEFT);
            return this;
        }

        /**
         * Add a line of text
         */
        public Builder line(String text) {
            return line(text, class_124.field_1080);
        }

        /**
         * Add a line of text with formatting
         */
        public Builder line(String text, class_124... formatting) {
            LabelComponent line = Components.label(class_2561.method_43470(text).method_27695(formatting));
            line.horizontalTextAlignment(HorizontalAlignment.LEFT);
            this.lines.add(line);
            return this;
        }

        /**
         * Add a Text component as a line
         */
        public Builder line(class_2561 text) {
            LabelComponent line = Components.label(text);
            line.horizontalTextAlignment(HorizontalAlignment.LEFT);
            this.lines.add(line);
            return this;
        }

        /**
         * Set custom icon
         */
        public Builder icon(String icon) {
            this.icon = icon;
            this.iconTexture = null;
            return this;
        }

        /**
         * Set icon texture
         */
        public Builder iconTexture(class_2960 texture, int size) {
            this.iconTexture = texture;
            this.iconSize = size;
            this.icon = null;
            return this;
        }

        /**
         * Set border color
         */
        public Builder borderColor(int color) {
            this.borderColor = color;
            return this;
        }

        /**
         * Set background color
         */
        public Builder backgroundColor(int color) {
            this.backgroundColor = color;
            return this;
        }

        /**
         * Set duration in milliseconds
         */
        public Builder duration(long durationMs) {
            this.duration = durationMs;
            return this;
        }

        /**
         * Set width
         */
        public Builder width(int width) {
            this.width = width;
            return this;
        }

        /**
         * Set padding
         */
        public Builder padding(int padding) {
            this.padding = padding;
            return this;
        }

        /**
         * Set gap between elements
         */
        public Builder gap(int gap) {
            this.gap = gap;
            return this;
        }

        /**
         * Set horizontal alignment
         */
        public Builder alignment(HorizontalAlignment alignment) {
            this.horizontalAlignment = alignment;
            return this;
        }

        /**
         * Build the toast
         */
        public PackCoreToast build() {
            if (title == null) {
                throw new IllegalStateException("Toast must have a title");
            }
            return new PackCoreToast(this);
        }

        /**
         * Build and show the toast immediately
         */
        public void show() {
            build().show();
        }
    }

    // ===== Static factory methods for common toasts =====

    /**
     * Create an update available toast
     */
    public static void showUpdateAvailable(String currentVersion, String newVersion, String modpackName) {
        new Builder()
                .type(ToastType.UPDATE)
                .title(class_2561.method_43470("Update Available for ")
                        .method_10852(class_2561.method_43470(modpackName).method_27695(class_124.field_1065, class_124.field_1067))
                        .method_10852(class_2561.method_43470("!")))
                .line(class_2561.method_43470(currentVersion).method_27692(class_124.field_1080)
                        .method_10852(class_2561.method_43470(" → ").method_27692(class_124.field_1062))
                        .method_10852(class_2561.method_43470(newVersion).method_27692(class_124.field_1065)))
                .line("Update the pack in your launcher", class_124.field_1080, class_124.field_1056)
                .duration(12000)
                .show();
    }

    /**
     * Create an export completion toast
     */
    public static void showExportComplete(String configName, String fileName) {
        new Builder()
                .type(ToastType.SUCCESS)
                .title("Export Complete!")
                .line(class_2561.method_43470("Config: ").method_27692(class_124.field_1080)
                        .method_10852(class_2561.method_43470(configName).method_27695(class_124.field_1060, class_124.field_1067)))
                .line(class_2561.method_43470("Saved as: ").method_27692(class_124.field_1080)
                        .method_10852(class_2561.method_43470(fileName).method_27692(class_124.field_1054)))
                .line("You can now share or import this config", class_124.field_1080, class_124.field_1056)
                .show();
    }

    /**
     * Create a backup completion toast
     */
    public static void showBackupComplete(String backupName, String fileName, boolean isRestore) {
        String title = isRestore ? "Restore Complete!" : "Backup Complete!";
        String message = isRestore ? "Configuration restored successfully." : "Backup saved successfully.";

        new Builder()
                .type(ToastType.SUCCESS)
                .borderColor(isRestore ? 0xFF_5555FF : 0xFF_55FF55)
                .title(title)
                .line(class_2561.method_43470("Name: ").method_27692(class_124.field_1080)
                        .method_10852(class_2561.method_43470(backupName).method_27695(class_124.field_1060, class_124.field_1067)))
                .line(class_2561.method_43470("File: ").method_27692(class_124.field_1080)
                        .method_10852(class_2561.method_43470(fileName).method_27692(class_124.field_1054)))
                .line(message, class_124.field_1080, class_124.field_1056)
                .show();
    }

    /**
     * Create an error toast
     */
    public static void showError(String title, String message) {
        new Builder()
                .type(ToastType.ERROR)
                .title(title)
                .line(message, class_124.field_1061)
                .duration(8000)
                .show();
    }

    /**
     * Create a warning toast
     */
    public static void showWarning(String title, String message) {
        new Builder()
                .type(ToastType.WARNING)
                .title(title)
                .line(message, class_124.field_1054)
                .show();
    }

    /**
     * Create an info toast
     */
    public static void showInfo(String title, String message) {
        new Builder()
                .type(ToastType.INFO)
                .title(title)
                .line(message, class_124.field_1075)
                .show();
    }

    /**
     * Create a simple success toast
     */
    public static void showSuccess(String message) {
        new Builder()
                .type(ToastType.SUCCESS)
                .title("Success!")
                .line(message, class_124.field_1060)
                .duration(5000)
                .show();
    }
}