/*
 * Decompiled with CFR 0.152.
 */
package com.github.kd_gaming1.packcore.scamshield.detector;

import com.github.kd_gaming1.packcore.PackCore;
import com.github.kd_gaming1.packcore.config.PackCoreConfig;
import com.github.kd_gaming1.packcore.scamshield.context.ConversationContext;
import com.github.kd_gaming1.packcore.scamshield.conversation.ConversationStage;
import com.github.kd_gaming1.packcore.scamshield.detector.DetectionResult;
import com.github.kd_gaming1.packcore.scamshield.detector.LegitimateTradeContext;
import com.github.kd_gaming1.packcore.scamshield.detector.cache.MessageAnalysisCache;
import com.github.kd_gaming1.packcore.scamshield.detector.types.CommandInstructionScam;
import com.github.kd_gaming1.packcore.scamshield.detector.types.JsonBasedScamType;
import com.github.kd_gaming1.packcore.scamshield.detector.types.PhishingLanguageScam;
import com.github.kd_gaming1.packcore.scamshield.detector.types.ScamType;
import com.github.kd_gaming1.packcore.scamshield.tracker.UserSuspicionTracker;
import java.io.IOException;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import net.fabricmc.loader.api.FabricLoader;

public class ScamDetector {
    private static final ScamDetector INSTANCE = new ScamDetector();
    private final List<ScamType> scamTypes = new CopyOnWriteArrayList<ScamType>();
    private final ConversationContext context = ConversationContext.getInstance();
    private final UserSuspicionTracker suspicionTracker = UserSuspicionTracker.getInstance();
    private final MessageAnalysisCache cache;
    private final Path scamShieldDir;
    private final Map<String, MessageRepetition> recentMessages = new ConcurrentHashMap<String, MessageRepetition>();

    private ScamDetector() {
        Path gameDir = FabricLoader.getInstance().getGameDir();
        this.scamShieldDir = gameDir.resolve("packcore/scamshield");
        this.cache = new MessageAnalysisCache(PackCoreConfig.scamShieldCacheSize, (long)PackCoreConfig.scamShieldCacheTTLSeconds * 1000L);
        this.initializeScamTypes();
    }

    public static ScamDetector getInstance() {
        return INSTANCE;
    }

    private void initializeScamTypes() {
        this.scamTypes.clear();
        this.scamTypes.add(new PhishingLanguageScam());
        this.scamTypes.add(new CommandInstructionScam());
        this.loadJsonScamTypes();
        PackCore.LOGGER.info("[ScamShield] Loaded {} total scam detectors ({} JSON-based)", (Object)this.scamTypes.size(), (Object)(this.scamTypes.size() - 2));
    }

    private void loadJsonScamTypes() {
        try {
            String fileName;
            if (!Files.exists(this.scamShieldDir, new LinkOption[0])) {
                Files.createDirectories(this.scamShieldDir, new FileAttribute[0]);
                PackCore.LOGGER.info("[ScamShield] Created scamshield directory: {}", (Object)this.scamShieldDir);
            }
            ArrayList<String> discoveredFiles = new ArrayList<String>();
            try (DirectoryStream<Path> stream = Files.newDirectoryStream(this.scamShieldDir, "scamtype-*.json");){
                for (Path file : stream) {
                    fileName = file.getFileName().toString();
                    discoveredFiles.add(fileName);
                }
            }
            Collections.sort(discoveredFiles);
            if (discoveredFiles.isEmpty()) {
                PackCore.LOGGER.warn("[ScamShield] No scamtype-*.json files found in {}", (Object)this.scamShieldDir);
                PackCore.LOGGER.warn("[ScamShield] Creating default scam type files...");
                this.createDefaultScamTypes();
                stream = Files.newDirectoryStream(this.scamShieldDir, "scamtype-*.json");
                try {
                    for (Path file : stream) {
                        fileName = file.getFileName().toString();
                        discoveredFiles.add(fileName);
                    }
                }
                finally {
                    if (stream != null) {
                        stream.close();
                    }
                }
                Collections.sort(discoveredFiles);
            }
            for (String fileName2 : discoveredFiles) {
                try {
                    JsonBasedScamType scamType = new JsonBasedScamType(fileName2);
                    this.scamTypes.add(scamType);
                    PackCore.LOGGER.info("[ScamShield] \u2713 Loaded: {} ({})", (Object)scamType.getDisplayName(), (Object)fileName2);
                }
                catch (Exception e) {
                    PackCore.LOGGER.error("[ScamShield] \u2717 Failed to load: {}", (Object)fileName2, (Object)e);
                }
            }
            if (discoveredFiles.isEmpty()) {
                PackCore.LOGGER.error("[ScamShield] WARNING: No scam type files loaded! Detection will be limited.");
            }
        }
        catch (IOException e) {
            PackCore.LOGGER.error("[ScamShield] Failed to scan for scam type files", (Throwable)e);
        }
    }

    private void createDefaultScamTypes() {
        String[] defaultFiles;
        for (String fileName : defaultFiles = new String[]{"scamtype-discord-verify.json", "scamtype-free-rank.json", "scamtype-island-theft.json", "scamtype-trade-manipulation.json"}) {
            try {
                new JsonBasedScamType(fileName);
                PackCore.LOGGER.info("[ScamShield] Created default file: {}", (Object)fileName);
            }
            catch (Exception e) {
                PackCore.LOGGER.error("[ScamShield] Failed to create default file: {}", (Object)fileName, (Object)e);
            }
        }
    }

    public DetectionResult analyze(String message, String sender) {
        int currentScore;
        int progressionBonus;
        DetectionResult cached;
        if (!PackCoreConfig.enableScamShield) {
            return DetectionResult.SAFE;
        }
        String normalizedMessage = this.normalizeMessage(message);
        String senderKey = sender != null ? sender.toLowerCase() : "";
        MessageRepetition rep = this.recentMessages.computeIfAbsent(senderKey, k -> new MessageRepetition(normalizedMessage));
        if (rep.normalizedMessage.equals(normalizedMessage)) {
            ++rep.count;
            if (rep.count >= 3 && LegitimateTradeContext.isLegitimateTradeAd(normalizedMessage)) {
                rep.markedAsLegit = true;
            }
            if (rep.markedAsLegit) {
                if (PackCoreConfig.enableScamShieldDebugging) {
                    PackCore.LOGGER.info("[ScamShield] Repetition-whitelisted: {} (repeated {}x)", (Object)sender, (Object)rep.count);
                }
                return DetectionResult.SAFE;
            }
        } else {
            this.recentMessages.put(senderKey, new MessageRepetition(normalizedMessage));
        }
        if ((cached = this.cache.get(normalizedMessage)) != null) {
            if (PackCoreConfig.enableScamShieldDebugging) {
                PackCore.LOGGER.info("[ScamShield] Cache hit for: {}", (Object)sender);
            }
            return cached;
        }
        if (PackCoreConfig.enableScamShieldDebugging) {
            PackCore.LOGGER.info("[ScamShield] Analyzing: [{}] '{}'", (Object)sender, (Object)normalizedMessage);
        }
        long startTime = System.nanoTime();
        ConversationStage currentStage = this.getCurrentStage(sender);
        int threshold = PackCoreConfig.scamShieldTriggerThreshold;
        DetectionResult.Builder resultBuilder = new DetectionResult.Builder(message, sender, threshold);
        resultBuilder.setConversationStage(currentStage);
        double legitimacyMultiplier = LegitimateTradeContext.getScoreMultiplier(normalizedMessage);
        if (legitimacyMultiplier < 1.0 && PackCoreConfig.enableScamShieldDebugging) {
            PackCore.LOGGER.info("[ScamShield]   Legitimate trade context detected, applying {}x multiplier", (Object)legitimacyMultiplier);
        }
        for (ScamType scamType : this.scamTypes) {
            if (!scamType.isEnabled()) continue;
            scamType.analyze(normalizedMessage, message, sender, this.context, resultBuilder);
        }
        if (legitimacyMultiplier < 1.0) {
            resultBuilder.applyMultiplier(legitimacyMultiplier);
        }
        if ((progressionBonus = this.suspicionTracker.recordAndAnalyze(sender, message, currentScore = resultBuilder.getCurrentTotalScore(), resultBuilder.build())) > 0) {
            resultBuilder.addProgressionBonus(progressionBonus);
        }
        DetectionResult result = resultBuilder.build();
        long durationMs = (System.nanoTime() - startTime) / 1000000L;
        if (PackCoreConfig.enableScamShieldDebugging) {
            PackCore.LOGGER.info("[ScamShield] Analysis complete: total={} (type={}, prog={}) | stage={} | threshold={} | triggered={} | {}ms", new Object[]{result.getTotalScore(), result.getScamTypeScore(), result.getProgressionScore(), currentStage.getDisplayName(), threshold, result.isTriggered(), durationMs});
        }
        this.cache.put(normalizedMessage, result);
        long now = System.currentTimeMillis();
        this.recentMessages.entrySet().removeIf(e -> now - ((MessageRepetition)e.getValue()).firstSeen > 300000L);
        return result;
    }

    private ConversationStage getCurrentStage(String sender) {
        if (sender == null || sender.isEmpty()) {
            return ConversationStage.INITIAL;
        }
        String senderKey = sender.toLowerCase();
        UserSuspicionTracker.EnhancedConversationHistory history = this.suspicionTracker.conversations.get(senderKey);
        return history != null ? history.getCurrentStage() : ConversationStage.INITIAL;
    }

    private String normalizeMessage(String message) {
        if (message == null || message.isEmpty()) {
            return "";
        }
        String lower = message.toLowerCase();
        StringBuilder result = new StringBuilder(lower.length());
        boolean lastWasSpace = true;
        for (int i = 0; i < lower.length(); ++i) {
            char c = lower.charAt(i);
            if (Character.isLetterOrDigit(c)) {
                result.append(c);
                lastWasSpace = false;
                continue;
            }
            if (!Character.isWhitespace(c) || lastWasSpace) continue;
            result.append(' ');
            lastWasSpace = true;
        }
        int len = result.length();
        if (len > 0 && result.charAt(len - 1) == ' ') {
            result.setLength(len - 1);
        }
        return result.toString();
    }

    public int reloadScamTypes() {
        PackCore.LOGGER.info("[ScamShield] Reloading all scam types...");
        this.cache.clear();
        int previousCount = this.scamTypes.size();
        this.initializeScamTypes();
        int newCount = this.scamTypes.size();
        PackCore.LOGGER.info("[ScamShield] Reload complete! {} scam detectors active (previously: {})", (Object)newCount, (Object)previousCount);
        if (newCount > previousCount) {
            PackCore.LOGGER.info("[ScamShield] {} NEW scam types discovered!", (Object)(newCount - previousCount));
        } else if (newCount < previousCount) {
            PackCore.LOGGER.info("[ScamShield] {} scam types removed", (Object)(previousCount - newCount));
        }
        return newCount;
    }

    public void shutdown() {
        this.cache.clear();
        PackCore.LOGGER.info("[ScamShield] Shutdown complete");
    }

    public List<ScamType> getScamTypes() {
        return new ArrayList<ScamType>(this.scamTypes);
    }

    public ConversationContext getContext() {
        return this.context;
    }

    public UserSuspicionTracker getSuspicionTracker() {
        return this.suspicionTracker;
    }

    public Path getScamShieldDirectory() {
        return this.scamShieldDir;
    }

    private static class MessageRepetition {
        String normalizedMessage;
        int count;
        long firstSeen;
        boolean markedAsLegit;

        MessageRepetition(String msg) {
            this.normalizedMessage = msg;
            this.count = 1;
            this.firstSeen = System.currentTimeMillis();
            this.markedAsLegit = false;
        }
    }
}

