/*
 * Copyright (c) 2019-2025 TagnumElite
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package com.tagnumelite.projecteintegration.datagen;

import com.tagnumelite.projecteintegration.PEIntegration;
import com.tagnumelite.projecteintegration.api.datagen.APEIItemTagsProvider;
import net.minecraft.core.HolderLookup;
import net.minecraft.data.PackOutput;
import net.minecraft.data.tags.ItemTagsProvider;
import net.minecraft.world.level.block.Block;
import net.neoforged.fml.ModList;
import net.neoforged.neoforge.common.data.ExistingFileHelper;
import net.neoforged.neoforgespi.language.ModFileScanData;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.objectweb.asm.Type;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;

public class PEIItemTagsProvider extends ItemTagsProvider {
    private static final Type PROVIDER_TYPE = Type.getType(com.tagnumelite.projecteintegration.api.datagen.PEIItemTagsProvider.class);

    public PEIItemTagsProvider(PackOutput output, CompletableFuture<HolderLookup.Provider> lookupProvider, CompletableFuture<TagLookup<Block>> blockTags, @Nullable ExistingFileHelper existingFileHelper) {
        super(output, lookupProvider, blockTags, PEIntegration.MODID, existingFileHelper);
    }

    @Override
    protected void addTags(HolderLookup.@NotNull Provider provider) {
        for (APEIItemTagsProvider tagProvider : getItemTagProviders()) {
            tagProvider.addTags(provider, this::tag);
        }
    }

    private List<APEIItemTagsProvider> getItemTagProviders() {
        ModList modList = ModList.get();
        List<APEIItemTagsProvider> providers = new ArrayList<>();

        for (ModFileScanData scanData : modList.getAllScanData()) {
            for (ModFileScanData.AnnotationData data : scanData.getAnnotations()) {
                if (PROVIDER_TYPE.equals(data.annotationType())) {
                    APEIItemTagsProvider instance = createInstance(data.memberName());
                    if (instance != null) {
                        providers.add(instance);
                    }
                }
            }
        }

        return providers;
    }

    private APEIItemTagsProvider createInstance(String className) {
        try {
            return Class.forName(className).asSubclass(APEIItemTagsProvider.class).newInstance();
        } catch (ClassNotFoundException | InstantiationException | IllegalAccessException e) {
            PEIntegration.LOGGER.error("Failed to load item tags provider: {}", className, e);
        }
        return null;
    }
}
