/*
 * Decompiled with CFR 0.152.
 */
package dev.turjo.easyshopgui.marketplace;

import dev.turjo.easyshopgui.EasyShopGUI;
import dev.turjo.easyshopgui.models.ShopItem;
import dev.turjo.easyshopgui.models.ShopSection;
import dev.turjo.easyshopgui.utils.Logger;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.concurrent.ConcurrentHashMap;
import org.bukkit.Bukkit;
import org.bukkit.plugin.Plugin;
import org.bukkit.scheduler.BukkitRunnable;

public class AIMarketplace {
    private final EasyShopGUI plugin;
    private final Map<String, MarketData> marketData = new ConcurrentHashMap<String, MarketData>();
    private final Map<String, PriceHistory> priceHistory = new ConcurrentHashMap<String, PriceHistory>();
    private final Map<String, StockData> stockData = new ConcurrentHashMap<String, StockData>();
    private final Random random = new Random();
    private final double PRICE_VOLATILITY;
    private final double DEMAND_INFLUENCE = 0.8;
    private final double SUPPLY_INFLUENCE = 0.6;
    private final double TREND_INFLUENCE = 0.4;
    private static final int HISTORY_SIZE = 100;
    private final int UPDATE_INTERVAL;

    public AIMarketplace(EasyShopGUI plugin) {
        this.plugin = plugin;
        this.PRICE_VOLATILITY = plugin.getConfigManager().getMainConfig().getDouble("ai-marketplace.price-volatility", 0.15);
        this.UPDATE_INTERVAL = plugin.getConfigManager().getMainConfig().getInt("ai-marketplace.update-interval", 300);
        if (plugin.getConfigManager().getMainConfig().getBoolean("ai-marketplace.enabled", true)) {
            this.initializeMarketData();
            this.startAIEngine();
        }
    }

    private void initializeMarketData() {
        try {
            Bukkit.getScheduler().runTaskLater((Plugin)this.plugin, () -> {
                try {
                    Map<String, ShopSection> sections = this.plugin.getGuiManager().getSections();
                    if (sections == null || sections.isEmpty()) {
                        Logger.warn("No sections available for AI Marketplace initialization");
                        return;
                    }
                    for (ShopSection section : sections.values()) {
                        for (ShopItem item : section.getItems()) {
                            String itemId = item.getId();
                            MarketData data = new MarketData(item.getBuyPrice(), item.getSellPrice(), item.getStock() == -1 ? 1000 : item.getStock(), this.calculateInitialDemand(item), 0, 0, System.currentTimeMillis());
                            this.marketData.put(itemId, data);
                            this.priceHistory.put(itemId, new PriceHistory());
                            StockData stock = new StockData(item.getStock() == -1 ? 1000 : item.getStock(), item.getStock() == -1 ? 1000 : item.getStock(), System.currentTimeMillis());
                            this.stockData.put(itemId, stock);
                            Logger.debug("Initialized market data for: " + itemId);
                        }
                    }
                    Logger.info("AI Marketplace initialized with " + this.marketData.size() + " items");
                }
                catch (Exception e) {
                    Logger.error("Error initializing AI Marketplace: " + e.getMessage());
                    e.printStackTrace();
                }
            }, 20L);
        }
        catch (Exception e) {
            Logger.error("Error scheduling AI Marketplace initialization: " + e.getMessage());
            e.printStackTrace();
        }
    }

    private void startAIEngine() {
        new BukkitRunnable(){

            public void run() {
                try {
                    AIMarketplace.this.adjustPricesWithAI();
                }
                catch (Exception e) {
                    Logger.error("Error in AI price adjustment: " + e.getMessage());
                }
            }
        }.runTaskTimerAsynchronously((Plugin)this.plugin, 600L, 600L);
        new BukkitRunnable(){

            public void run() {
                try {
                    AIMarketplace.this.manageStockWithAI();
                }
                catch (Exception e) {
                    Logger.error("Error in AI stock management: " + e.getMessage());
                }
            }
        }.runTaskTimerAsynchronously((Plugin)this.plugin, 1200L, 1200L);
        new BukkitRunnable(){

            public void run() {
                try {
                    AIMarketplace.this.analyzeMarketTrends();
                }
                catch (Exception e) {
                    Logger.error("Error in market analysis: " + e.getMessage());
                }
            }
        }.runTaskTimerAsynchronously((Plugin)this.plugin, 6000L, 6000L);
        Logger.info("AI Marketplace engines started - Real-time pricing active!");
    }

    private void adjustPricesWithAI() {
        for (Map.Entry<String, MarketData> entry : this.marketData.entrySet()) {
            String itemId = entry.getKey();
            MarketData data = entry.getValue();
            try {
                double demandFactor = this.calculateDemandFactor(data);
                double supplyFactor = this.calculateSupplyFactor(data);
                double trendFactor = this.calculateTrendFactor(itemId);
                double volatilityFactor = this.calculateVolatilityFactor();
                double priceMultiplier = 1.0 + demandFactor * 0.8 + supplyFactor * 0.6 + trendFactor * 0.4 + volatilityFactor * 0.1;
                priceMultiplier = Math.max(0.3, Math.min(3.0, priceMultiplier));
                double newBuyPrice = data.getBaseBuyPrice() * priceMultiplier;
                double newSellPrice = data.getBaseSellPrice() * priceMultiplier;
                data.setCurrentBuyPrice(newBuyPrice);
                data.setCurrentSellPrice(newSellPrice);
                PriceHistory history = this.priceHistory.get(itemId);
                if (history != null) {
                    history.addPrice(newBuyPrice);
                }
                this.updateShopItemPrice(itemId, newBuyPrice, newSellPrice);
                Logger.debug("AI adjusted price for " + itemId + ": Buy $" + String.format("%.2f", newBuyPrice) + ", Sell $" + String.format("%.2f", newSellPrice));
            }
            catch (Exception e) {
                Logger.debug("Error adjusting price for " + itemId + ": " + e.getMessage());
            }
        }
    }

    private void manageStockWithAI() {
        for (Map.Entry<String, StockData> entry : this.stockData.entrySet()) {
            String itemId = entry.getKey();
            StockData stock = entry.getValue();
            MarketData market = this.marketData.get(itemId);
            if (market == null) continue;
            try {
                double demandRate = this.calculateDemandRate(market);
                double stockLevel = (double)stock.getCurrentStock() / (double)Math.max(1, stock.getMaxStock());
                long timeSinceRestock = System.currentTimeMillis() - stock.getLastRestockTime();
                boolean shouldRestock = false;
                int restockAmount = 0;
                if (stockLevel < 0.1) {
                    restockAmount = (int)((double)stock.getMaxStock() * 0.8);
                    shouldRestock = true;
                    Logger.debug("Emergency restock triggered for " + itemId);
                } else if (stockLevel < 0.3 && demandRate > 0.6) {
                    restockAmount = (int)((double)stock.getMaxStock() * 0.6);
                    shouldRestock = true;
                    Logger.debug("High demand restock triggered for " + itemId);
                } else if (stockLevel < 0.5 && timeSinceRestock > 1800000L) {
                    restockAmount = (int)((double)stock.getMaxStock() * 0.4);
                    shouldRestock = true;
                    Logger.debug("Regular restock triggered for " + itemId);
                }
                if (!shouldRestock) continue;
                int newStock = Math.min(stock.getCurrentStock() + restockAmount, stock.getMaxStock());
                stock.setCurrentStock(newStock);
                stock.setLastRestockTime(System.currentTimeMillis());
                this.updateShopItemStock(itemId, newStock);
                Logger.debug("AI restocked " + itemId + " to " + newStock + " units");
            }
            catch (Exception e) {
                Logger.debug("Error managing stock for " + itemId + ": " + e.getMessage());
            }
        }
    }

    public boolean processPurchase(String itemId, int quantity) {
        StockData stock = this.stockData.get(itemId);
        MarketData market = this.marketData.get(itemId);
        if (stock == null || market == null) {
            return false;
        }
        if (stock.getCurrentStock() != -1 && stock.getCurrentStock() < quantity) {
            return false;
        }
        if (stock.getCurrentStock() != -1) {
            stock.setCurrentStock(Math.max(0, stock.getCurrentStock() - quantity));
            this.updateShopItemStock(itemId, stock.getCurrentStock());
        }
        market.setBuyTransactions(market.getBuyTransactions() + quantity);
        market.setLastTransactionTime(System.currentTimeMillis());
        this.adjustPriceAfterTransaction(itemId, "BUY", quantity);
        Logger.debug("Processed purchase: " + quantity + "x " + itemId + ", Stock: " + stock.getCurrentStock());
        return true;
    }

    public void processSale(String itemId, int quantity) {
        StockData stock = this.stockData.get(itemId);
        MarketData market = this.marketData.get(itemId);
        if (stock == null || market == null) {
            return;
        }
        if (stock.getCurrentStock() != -1) {
            stock.setCurrentStock(Math.min(stock.getCurrentStock() + quantity, stock.getMaxStock()));
            this.updateShopItemStock(itemId, stock.getCurrentStock());
        }
        market.setSellTransactions(market.getSellTransactions() + quantity);
        market.setLastTransactionTime(System.currentTimeMillis());
        this.adjustPriceAfterTransaction(itemId, "SELL", quantity);
        Logger.debug("Processed sale: " + quantity + "x " + itemId + ", Stock: " + stock.getCurrentStock());
    }

    private void adjustPriceAfterTransaction(String itemId, String type, int quantity) {
        MarketData data = this.marketData.get(itemId);
        if (data == null) {
            return;
        }
        try {
            double priceChange = 0.0;
            if (type.equals("BUY")) {
                priceChange = (double)quantity / 64.0 * 0.02;
            } else if (type.equals("SELL")) {
                priceChange = -((double)quantity / 64.0) * 0.015;
            }
            double currentMultiplier = data.getCurrentBuyPrice() / data.getBaseBuyPrice();
            double newMultiplier = Math.max(0.5, Math.min(2.0, currentMultiplier + priceChange));
            double newBuyPrice = data.getBaseBuyPrice() * newMultiplier;
            double newSellPrice = data.getBaseSellPrice() * newMultiplier;
            data.setCurrentBuyPrice(newBuyPrice);
            data.setCurrentSellPrice(newSellPrice);
            this.updateShopItemPrice(itemId, newBuyPrice, newSellPrice);
            Logger.debug("Immediate price adjustment for " + itemId + ": " + (priceChange > 0.0 ? "+" : "") + String.format("%.1f", priceChange * 100.0) + "%");
        }
        catch (Exception e) {
            Logger.debug("Error in immediate price adjustment: " + e.getMessage());
        }
    }

    private double calculateTrendFactor(String itemId) {
        try {
            PriceHistory history = this.priceHistory.get(itemId);
            if (history == null) {
                return 0.0;
            }
            List<Double> prices = history.getPrices();
            if (prices == null || prices.size() < 6) {
                return 0.0;
            }
            int size = prices.size();
            int recentStart = Math.max(0, size - 3);
            int recentEnd = size;
            int olderStart = Math.max(0, size - 6);
            int olderEnd = Math.max(0, size - 3);
            if (recentStart >= recentEnd || olderStart >= olderEnd) {
                return 0.0;
            }
            List<Double> recentPrices = prices.subList(recentStart, recentEnd);
            List<Double> olderPrices = prices.subList(olderStart, olderEnd);
            if (recentPrices.isEmpty() || olderPrices.isEmpty()) {
                return 0.0;
            }
            double recentAvg = recentPrices.stream().mapToDouble(Double::doubleValue).average().orElse(0.0);
            double olderAvg = olderPrices.stream().mapToDouble(Double::doubleValue).average().orElse(0.0);
            if (olderAvg == 0.0) {
                return 0.0;
            }
            return (recentAvg - olderAvg) / olderAvg * 0.5;
        }
        catch (Exception e) {
            Logger.debug("Error calculating trend factor for " + itemId + ": " + e.getMessage());
            return 0.0;
        }
    }

    private double calculateDemandFactor(MarketData data) {
        try {
            long timeSinceLastTransaction = System.currentTimeMillis() - data.getLastTransactionTime();
            double timeDecay = Math.exp((double)(-timeSinceLastTransaction) / 3600000.0);
            double buyDemand = (double)data.getBuyTransactions() * timeDecay;
            double sellSupply = (double)data.getSellTransactions() * timeDecay;
            return Math.tanh((buyDemand - sellSupply) / 100.0) * this.PRICE_VOLATILITY;
        }
        catch (Exception e) {
            Logger.debug("Error calculating demand factor: " + e.getMessage());
            return 0.0;
        }
    }

    private double calculateSupplyFactor(MarketData data) {
        try {
            StockData stock = this.stockData.get(data.toString());
            if (stock == null) {
                return 0.0;
            }
            double stockRatio = (double)stock.getCurrentStock() / (double)Math.max(1, stock.getMaxStock());
            return (0.5 - stockRatio) * this.PRICE_VOLATILITY;
        }
        catch (Exception e) {
            Logger.debug("Error calculating supply factor: " + e.getMessage());
            return 0.0;
        }
    }

    private double calculateVolatilityFactor() {
        return this.random.nextGaussian() * 0.02;
    }

    private void analyzeMarketTrends() {
        Logger.debug("AI analyzing market trends...");
        for (Map.Entry<String, MarketData> entry : this.marketData.entrySet()) {
            String itemId = entry.getKey();
            MarketData data = entry.getValue();
            try {
                PriceHistory history = this.priceHistory.get(itemId);
                String trend = this.analyzePriceTrend(history);
                data.setTrend(trend);
                String sentiment = this.calculateMarketSentiment(data);
                data.setSentiment(sentiment);
                Logger.debug("Item " + itemId + " - Trend: " + trend + ", Sentiment: " + sentiment);
            }
            catch (Exception e) {
                Logger.debug("Error analyzing trends for " + itemId + ": " + e.getMessage());
            }
        }
    }

    private String analyzePriceTrend(PriceHistory history) {
        try {
            if (history == null) {
                return "STABLE";
            }
            List<Double> prices = history.getPrices();
            if (prices == null || prices.size() < 3) {
                return "STABLE";
            }
            int size = prices.size();
            if (size < 3) {
                return "STABLE";
            }
            double recent = prices.get(size - 1);
            double older = prices.get(Math.max(0, size - 3));
            if (older == 0.0) {
                return "STABLE";
            }
            double change = (recent - older) / older;
            if (change > 0.05) {
                return "RISING";
            }
            if (change < -0.05) {
                return "FALLING";
            }
            return "STABLE";
        }
        catch (Exception e) {
            Logger.debug("Error analyzing price trend: " + e.getMessage());
            return "STABLE";
        }
    }

    private String calculateMarketSentiment(MarketData data) {
        try {
            double demandRatio = this.calculateDemandRate(data);
            if (demandRatio > 0.7) {
                return "BULLISH";
            }
            if (demandRatio < 0.3) {
                return "BEARISH";
            }
            return "NEUTRAL";
        }
        catch (Exception e) {
            return "NEUTRAL";
        }
    }

    private double calculateDemandRate(MarketData data) {
        try {
            long timeSinceLastTransaction = System.currentTimeMillis() - data.getLastTransactionTime();
            if (timeSinceLastTransaction > 3600000L) {
                return 0.1;
            }
            return Math.min(1.0, (double)(data.getBuyTransactions() + data.getSellTransactions()) / 50.0);
        }
        catch (Exception e) {
            return 0.1;
        }
    }

    private double calculateInitialDemand(ShopItem item) {
        String materialName = item.getMaterial().toString();
        if (materialName.contains("DIAMOND") || materialName.contains("EMERALD") || materialName.contains("NETHERITE")) {
            return 0.9;
        }
        if (materialName.contains("IRON") || materialName.contains("GOLD") || materialName.contains("REDSTONE")) {
            return 0.7;
        }
        if (materialName.contains("STONE") || materialName.contains("DIRT") || materialName.contains("COBBLESTONE")) {
            return 0.3;
        }
        if (materialName.contains("BREAD") || materialName.contains("APPLE") || materialName.contains("BEEF") || materialName.contains("CHICKEN")) {
            return 0.6;
        }
        return 0.5;
    }

    private void updateShopItemPrice(String itemId, double buyPrice, double sellPrice) {
        try {
            Map<String, ShopSection> sections = this.plugin.getGuiManager().getSections();
            for (ShopSection section : sections.values()) {
                ShopItem item = section.getItem(itemId);
                if (item == null) continue;
                item.setBuyPrice(buyPrice);
                item.setSellPrice(sellPrice);
                break;
            }
        }
        catch (Exception e) {
            Logger.debug("Error updating shop item price: " + e.getMessage());
        }
    }

    private void updateShopItemStock(String itemId, int stock) {
        try {
            Map<String, ShopSection> sections = this.plugin.getGuiManager().getSections();
            for (ShopSection section : sections.values()) {
                ShopItem item = section.getItem(itemId);
                if (item == null) continue;
                item.setStock(stock);
                break;
            }
        }
        catch (Exception e) {
            Logger.debug("Error updating shop item stock: " + e.getMessage());
        }
    }

    public double getCurrentBuyPrice(String itemId) {
        MarketData data = this.marketData.get(itemId);
        return data != null ? data.getCurrentBuyPrice() : 0.0;
    }

    public double getCurrentSellPrice(String itemId) {
        MarketData data = this.marketData.get(itemId);
        return data != null ? data.getCurrentSellPrice() : 0.0;
    }

    public int getCurrentStock(String itemId) {
        StockData stock = this.stockData.get(itemId);
        return stock != null ? stock.getCurrentStock() : -1;
    }

    public void recordTransaction(String itemId, String type, int quantity, double price) {
        try {
            if (type.equals("BUY")) {
                this.processPurchase(itemId, quantity);
            } else if (type.equals("SELL")) {
                this.processSale(itemId, quantity);
            }
        }
        catch (Exception e) {
            Logger.debug("Error recording transaction: " + e.getMessage());
        }
    }

    private static class MarketData {
        private final double baseBuyPrice;
        private final double baseSellPrice;
        private double currentBuyPrice;
        private double currentSellPrice;
        private int buyTransactions;
        private int sellTransactions;
        private long lastTransactionTime;
        private String trend = "STABLE";
        private String sentiment = "NEUTRAL";

        public MarketData(double baseBuyPrice, double baseSellPrice, int maxStock, double demand, int buyTransactions, int sellTransactions, long lastTransactionTime) {
            this.baseBuyPrice = baseBuyPrice;
            this.baseSellPrice = baseSellPrice;
            this.currentBuyPrice = baseBuyPrice;
            this.currentSellPrice = baseSellPrice;
            this.buyTransactions = buyTransactions;
            this.sellTransactions = sellTransactions;
            this.lastTransactionTime = lastTransactionTime;
        }

        public double getBaseBuyPrice() {
            return this.baseBuyPrice;
        }

        public double getBaseSellPrice() {
            return this.baseSellPrice;
        }

        public double getCurrentBuyPrice() {
            return this.currentBuyPrice;
        }

        public void setCurrentBuyPrice(double currentBuyPrice) {
            this.currentBuyPrice = currentBuyPrice;
        }

        public double getCurrentSellPrice() {
            return this.currentSellPrice;
        }

        public void setCurrentSellPrice(double currentSellPrice) {
            this.currentSellPrice = currentSellPrice;
        }

        public int getBuyTransactions() {
            return this.buyTransactions;
        }

        public void setBuyTransactions(int buyTransactions) {
            this.buyTransactions = buyTransactions;
        }

        public int getSellTransactions() {
            return this.sellTransactions;
        }

        public void setSellTransactions(int sellTransactions) {
            this.sellTransactions = sellTransactions;
        }

        public long getLastTransactionTime() {
            return this.lastTransactionTime;
        }

        public void setLastTransactionTime(long lastTransactionTime) {
            this.lastTransactionTime = lastTransactionTime;
        }

        public String getTrend() {
            return this.trend;
        }

        public void setTrend(String trend) {
            this.trend = trend;
        }

        public String getSentiment() {
            return this.sentiment;
        }

        public void setSentiment(String sentiment) {
            this.sentiment = sentiment;
        }
    }

    private static class PriceHistory {
        private final LinkedList<Double> prices = new LinkedList();

        private PriceHistory() {
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void addPrice(double price) {
            LinkedList<Double> linkedList = this.prices;
            synchronized (linkedList) {
                this.prices.addLast(price);
                if (this.prices.size() > 100) {
                    this.prices.removeFirst();
                }
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public List<Double> getPrices() {
            LinkedList<Double> linkedList = this.prices;
            synchronized (linkedList) {
                return new ArrayList<Double>(this.prices);
            }
        }
    }

    private static class StockData {
        private int currentStock;
        private final int maxStock;
        private long lastRestockTime;

        public StockData(int currentStock, int maxStock, long lastRestockTime) {
            this.currentStock = currentStock;
            this.maxStock = maxStock;
            this.lastRestockTime = lastRestockTime;
        }

        public int getCurrentStock() {
            return this.currentStock;
        }

        public void setCurrentStock(int currentStock) {
            this.currentStock = currentStock;
        }

        public int getMaxStock() {
            return this.maxStock;
        }

        public long getLastRestockTime() {
            return this.lastRestockTime;
        }

        public void setLastRestockTime(long lastRestockTime) {
            this.lastRestockTime = lastRestockTime;
        }
    }
}

