/*
 * Decompiled with CFR 0.152.
 */
package com.github.mizosoft.methanol.internal.decoder;

import com.github.mizosoft.methanol.decoder.AsyncDecoder;
import com.github.mizosoft.methanol.internal.decoder.InflaterUtils;
import java.io.EOFException;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.zip.CRC32;
import java.util.zip.Inflater;
import java.util.zip.ZipException;

final class GzipDecoder
implements AsyncDecoder {
    static final String ENCODING = "gzip";
    private static final int GZIP_MAGIC = 35615;
    private static final int CM_DEFLATE = 8;
    private static final int HEADER_SIZE = 10;
    private static final int HEADER_SKIPPED_SIZE = 6;
    private static final int TRAILER_SIZE = 8;
    private static final int TEMP_BUFFER_SIZE = 10;
    private static final int MIN_GZIP_MEMBER_SIZE = 20;
    private static final int BYTE_MASK = 255;
    private static final int SHORT_MASK = 65535;
    private static final long INT_MASK = 0xFFFFFFFFL;
    private final Inflater inflater = new Inflater(true);
    private final ByteBuffer tempBuffer = ByteBuffer.allocate(10).order(ByteOrder.LITTLE_ENDIAN);
    private final CRC32 crc = new CRC32();
    private boolean computeCrc;
    private State state = State.BEGIN;
    private int flags;
    private int fieldLength;
    private int fieldPosition;

    GzipDecoder() {
    }

    @Override
    public String encoding() {
        return ENCODING;
    }

    @Override
    public void decode(AsyncDecoder.ByteSource source, AsyncDecoder.ByteSink sink) throws IOException {
        block13: while (this.state != State.END) {
            switch (this.state) {
                case BEGIN: {
                    this.state = State.HEADER.prepare(this);
                }
                case HEADER: {
                    if (source.remaining() < 10L) break block13;
                    this.readHeader(source);
                    this.state = State.fromFlags(this);
                    continue block13;
                }
                case FLG_EXTRA_LEN: {
                    if (source.remaining() < 2L) break block13;
                    this.fieldLength = this.getUShort(source);
                    this.state = State.FLG_EXTRA_DATA.prepare(this);
                }
                case FLG_EXTRA_DATA: {
                    if (!this.trySkipExtraField(source)) break block13;
                    this.state = State.fromFlags(this);
                    continue block13;
                }
                case FLG_ZERO_TERMINATED: {
                    if (!this.tryConsumeToZeroByte(source)) break block13;
                    this.state = State.fromFlags(this);
                    continue block13;
                }
                case FLG_HCRC: {
                    if (source.remaining() < 2L) break block13;
                    long crc16 = this.crc.getValue() & 0xFFFFL;
                    GzipDecoder.checkValue(crc16, this.getUShort(source), "corrupt gzip header");
                    this.state = State.DEFLATED.prepare(this);
                }
                case DEFLATED: {
                    InflaterUtils.inflateSourceWithChecksum(this.inflater, source, sink, this.crc);
                    if (!this.inflater.finished()) break block13;
                    this.state = State.TRAILER.prepare(this);
                }
                case TRAILER: {
                    if (source.remaining() < 8L) break block13;
                    this.readTrailer(source);
                    this.state = State.CONCAT_INSPECTION;
                }
                case CONCAT_INSPECTION: {
                    IOException failedToReadHeader = null;
                    if (source.remaining() < 20L) {
                        if (!source.finalSource()) break block13;
                        this.state = State.END;
                    } else {
                        State.HEADER.prepare(this);
                        try {
                            this.readHeader(source);
                            this.state = State.fromFlags(this);
                        }
                        catch (IOException e) {
                            failedToReadHeader = e;
                            this.state = State.END;
                        }
                    }
                    if (this.state != State.END || !source.hasRemaining()) continue block13;
                    IOException streamFinishedPrematurely = new IOException("gzip stream finished prematurely");
                    if (failedToReadHeader != null) {
                        streamFinishedPrematurely.addSuppressed(failedToReadHeader);
                    }
                    throw streamFinishedPrematurely;
                }
                default: {
                    throw new AssertionError((Object)("unexpected state: " + this.state));
                }
            }
        }
        if (this.state != State.END && source.finalSource()) {
            throw new EOFException("unexpected end of gzip stream");
        }
    }

    @Override
    public void close() {
        this.inflater.end();
    }

    private ByteBuffer fillTempBuffer(AsyncDecoder.ByteSource source, int bytes) {
        source.pullBytes(this.tempBuffer.rewind().limit(bytes));
        assert (!this.tempBuffer.hasRemaining());
        if (this.computeCrc) {
            this.crc.update(this.tempBuffer.rewind());
        }
        return this.tempBuffer.rewind();
    }

    private int getUByte(AsyncDecoder.ByteSource source) {
        return this.fillTempBuffer(source, 1).get() & 0xFF;
    }

    private int getUShort(AsyncDecoder.ByteSource source) {
        return this.fillTempBuffer(source, 2).getShort() & 0xFFFF;
    }

    private long getUInt(AsyncDecoder.ByteSource source) {
        return (long)this.fillTempBuffer(source, 4).getInt() & 0xFFFFFFFFL;
    }

    private void readHeader(AsyncDecoder.ByteSource source) throws IOException {
        GzipDecoder.checkValue(35615L, this.getUShort(source), "not in gzip format");
        GzipDecoder.checkValue(8L, this.getUByte(source), "unsupported compression method");
        int flags = this.getUByte(source);
        if (FlagOption.RESERVED.isEnabled(flags)) {
            throw new ZipException(String.format("unsupported flags: %#x", flags));
        }
        if (!FlagOption.HCRC.isEnabled(flags)) {
            this.computeCrc = false;
        }
        this.flags = flags;
        this.fillTempBuffer(source, 6);
    }

    private void readTrailer(AsyncDecoder.ByteSource source) throws IOException {
        GzipDecoder.checkValue(this.crc.getValue(), this.getUInt(source), "corrupt gzip stream (CRC32)");
        GzipDecoder.checkValue(this.inflater.getBytesWritten() & 0xFFFFFFFFL, this.getUInt(source), "corrupt gzip stream (ISIZE)");
    }

    private boolean trySkipExtraField(AsyncDecoder.ByteSource source) {
        while (source.hasRemaining() && this.fieldPosition < this.fieldLength) {
            ByteBuffer in = source.currentSource();
            int skipped = Math.min(in.remaining(), this.fieldLength - this.fieldPosition);
            int skipPosition = in.position() + skipped;
            if (this.computeCrc) {
                int originalLimit = in.limit();
                this.crc.update(in.limit(skipPosition));
                in.limit(originalLimit);
            } else {
                in.position(skipPosition);
            }
            this.fieldPosition += skipped;
        }
        return this.fieldPosition >= this.fieldLength;
    }

    private boolean tryConsumeToZeroByte(AsyncDecoder.ByteSource source) {
        while (source.hasRemaining()) {
            ByteBuffer in = source.currentSource();
            while (in.hasRemaining()) {
                byte currentByte = in.get();
                if (this.computeCrc) {
                    this.crc.update(currentByte);
                }
                if (currentByte != 0) continue;
                return true;
            }
        }
        return false;
    }

    private static void checkValue(long expected, long found, String msg) throws IOException {
        if (expected != found) {
            throw new ZipException(String.format("%s; expected: %#x, found: %#x", msg, expected, found));
        }
    }

    private static enum State {
        BEGIN,
        HEADER{

            @Override
            void onPrepare(GzipDecoder dec) {
                dec.crc.reset();
                dec.computeCrc = true;
            }
        }
        ,
        FLG_EXTRA_LEN{

            @Override
            void onPrepare(GzipDecoder dec) {
                dec.fieldLength = 0;
            }
        }
        ,
        FLG_EXTRA_DATA{

            @Override
            void onPrepare(GzipDecoder dec) {
                dec.fieldPosition = 0;
            }
        }
        ,
        FLG_ZERO_TERMINATED,
        FLG_HCRC{

            @Override
            void onPrepare(GzipDecoder dec) {
                dec.computeCrc = false;
            }
        }
        ,
        DEFLATED{

            @Override
            void onPrepare(GzipDecoder dec) {
                dec.inflater.reset();
                dec.crc.reset();
            }
        }
        ,
        TRAILER,
        CONCAT_INSPECTION{

            @Override
            void onPrepare(GzipDecoder dec) {
                HEADER.onPrepare(dec);
            }
        }
        ,
        END;


        void onPrepare(GzipDecoder dec) {
        }

        final State prepare(GzipDecoder dec) {
            this.onPrepare(dec);
            return this;
        }

        private static State fromFlags(GzipDecoder dec) {
            FlagOption option = FlagOption.nextEnabled(dec.flags);
            dec.flags = option.clear(dec.flags);
            return option.forward(dec);
        }
    }

    private static enum FlagOption {
        RESERVED(224, State.END){

            @Override
            boolean isEnabled(int flags) {
                return (flags & this.value) != 0;
            }
        }
        ,
        EXTRA(4, State.FLG_EXTRA_LEN),
        NAME(8, State.FLG_ZERO_TERMINATED),
        COMMENT(16, State.FLG_ZERO_TERMINATED),
        HCRC(2, State.FLG_HCRC),
        TEXT(1, State.DEFLATED),
        NONE(0, State.DEFLATED){

            @Override
            boolean isEnabled(int flags) {
                return flags == this.value;
            }
        };

        final int value;
        final State state;

        private FlagOption(int value2, State state) {
            this.value = value2;
            this.state = state;
        }

        boolean isEnabled(int flags) {
            return (flags & this.value) == this.value;
        }

        int clear(int flags) {
            return flags & ~this.value;
        }

        State forward(GzipDecoder ctx) {
            return this.state.prepare(ctx);
        }

        static FlagOption nextEnabled(int flags) {
            for (FlagOption option : FlagOption.values()) {
                if (!option.isEnabled(flags)) continue;
                return option;
            }
            throw new AssertionError((Object)("couldn't get FlgOption for: " + Integer.toHexString(flags)));
        }
    }
}

