/*
 * Decompiled with CFR 0.152.
 */
package net.flectone.pulse.data.database;

import com.github.retrooper.packetevents.manager.server.ServerVersion;
import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.Path;
import java.text.SimpleDateFormat;
import java.util.Date;
import javax.sql.DataSource;
import net.flectone.pulse.config.Config;
import net.flectone.pulse.data.database.dao.FPlayerDAO;
import net.flectone.pulse.data.database.dao.SettingDAO;
import net.flectone.pulse.library.apache.lang3.Strings;
import net.flectone.pulse.library.guice.Inject;
import net.flectone.pulse.library.guice.Injector;
import net.flectone.pulse.library.guice.Singleton;
import net.flectone.pulse.library.guice.name.Named;
import net.flectone.pulse.library.hikaricp.hikari.HikariConfig;
import net.flectone.pulse.library.hikaricp.hikari.HikariDataSource;
import net.flectone.pulse.library.hikaricp.hikari.pool.HikariPool;
import net.flectone.pulse.library.jdbi3.v3.core.Jdbi;
import net.flectone.pulse.library.jdbi3.v3.core.mapper.reflect.ConstructorMapper;
import net.flectone.pulse.library.jdbi3.v3.core.spi.JdbiPlugin;
import net.flectone.pulse.library.jdbi3.v3.core.statement.SqlStatements;
import net.flectone.pulse.library.jdbi3.v3.sqlobject.SqlObjectPlugin;
import net.flectone.pulse.library.libby.Library;
import net.flectone.pulse.model.FColor;
import net.flectone.pulse.model.entity.FPlayer;
import net.flectone.pulse.model.util.Moderation;
import net.flectone.pulse.module.command.ignore.model.Ignore;
import net.flectone.pulse.module.command.mail.model.Mail;
import net.flectone.pulse.platform.adapter.PlatformServerAdapter;
import net.flectone.pulse.platform.provider.PacketProvider;
import net.flectone.pulse.processing.resolver.FileResolver;
import net.flectone.pulse.processing.resolver.ReflectionResolver;
import net.flectone.pulse.processing.resolver.SystemVariableResolver;
import net.flectone.pulse.util.logging.FLogger;
import org.jetbrains.annotations.NotNull;

@Singleton
public class Database {
    private final Config.Database config;
    private final Injector injector;
    private final FileResolver fileResolver;
    private final Path projectPath;
    private final SystemVariableResolver systemVariableResolver;
    private final PlatformServerAdapter platformServerAdapter;
    private final FLogger fLogger;
    private final PacketProvider packetProvider;
    private final ReflectionResolver reflectionResolver;
    private HikariDataSource dataSource;
    private Jdbi jdbi;

    @Inject
    public Database(FileResolver fileResolver, Injector injector, @Named(value="projectPath") Path projectPath, SystemVariableResolver systemVariableResolver, PlatformServerAdapter platformServerAdapter, FLogger fLogger, PacketProvider packetProvider, ReflectionResolver reflectionResolver) {
        this.config = fileResolver.getConfig().getDatabase();
        this.injector = injector;
        this.fileResolver = fileResolver;
        this.projectPath = projectPath;
        this.systemVariableResolver = systemVariableResolver;
        this.platformServerAdapter = platformServerAdapter;
        this.fLogger = fLogger;
        this.packetProvider = packetProvider;
        this.reflectionResolver = reflectionResolver;
    }

    public void connect() throws IOException {
        if (this.packetProvider.getServerVersion().isOlderThanOrEquals(ServerVersion.V_1_10_2) && this.config.getType() == Type.SQLITE) {
            this.fLogger.warning("SQLite database is not supported on this version of Minecraft");
            this.fLogger.warning("H2 Database will be used");
            this.config.setType(Type.H2);
        }
        HikariConfig hikariConfig = this.createHikaryConfig();
        try {
            this.dataSource = new HikariDataSource(hikariConfig);
            this.jdbi = Jdbi.create((DataSource)this.dataSource);
            this.jdbi.installPlugin((JdbiPlugin)new SqlObjectPlugin());
            if (this.config.getType() == Type.POSTGRESQL) {
                ((SqlStatements)this.jdbi.getConfig(SqlStatements.class)).setTemplateEngine((sql, ctx) -> Strings.CS.replace(sql, "`", "\""));
            }
            this.jdbi.registerRowMapper(ConstructorMapper.factory(FColor.class));
            this.jdbi.registerRowMapper(ConstructorMapper.factory(FPlayerDAO.PlayerInfo.class));
            this.jdbi.registerRowMapper(ConstructorMapper.factory(Ignore.class));
            this.jdbi.registerRowMapper(ConstructorMapper.factory(Mail.class));
            this.jdbi.registerRowMapper(ConstructorMapper.factory(Moderation.class));
        }
        catch (HikariPool.PoolInitializationException e) {
            throw new RuntimeException(e);
        }
        InputStream sqlFile = this.platformServerAdapter.getResource("sqls/" + this.config.getType().name().toLowerCase() + ".sql");
        this.executeSQLFile(sqlFile);
        if (this.fileResolver.isVersionOlderThan(this.fileResolver.getPreInitVersion(), "0.9.0")) {
            this.MIGRATION_0_9_0();
        }
        if (this.fileResolver.isVersionOlderThan(this.fileResolver.getPreInitVersion(), "1.3.0")) {
            this.MIGRATION_1_3_0();
        }
        if (this.config.getType() == Type.SQLITE) {
            ((FPlayerDAO)this.injector.getInstance(FPlayerDAO.class)).updateAllToOffline();
        }
        this.init();
    }

    @NotNull
    public Jdbi getJdbi() throws IllegalStateException {
        if (this.jdbi == null) {
            throw new IllegalStateException("JDBI not initialized");
        }
        return this.jdbi;
    }

    public void init() {
        this.fLogger.info(String.valueOf((Object)this.config.getType()) + " database connected");
    }

    public void disconnect() {
        if (this.dataSource != null) {
            this.dataSource.getHikariPoolMXBean().softEvictConnections();
            this.dataSource.close();
            this.fLogger.info("Database disconnected");
        }
    }

    private HikariConfig createHikaryConfig() {
        HikariConfig hikariConfig = new HikariConfig();
        String connectionURL = "jdbc:" + this.config.getType().name().toLowerCase() + ":";
        switch (this.config.getType().ordinal()) {
            case 0: {
                this.reflectionResolver.hasClassOrElse("org.postgresql.Driver", libraryResolver -> libraryResolver.loadLibrary(Library.builder().groupId("org{}postgresql").artifactId("postgresql").version("42.7.7").build()));
                connectionURL = connectionURL + "//" + this.systemVariableResolver.substituteEnvVars(this.config.getHost()) + ":" + this.systemVariableResolver.substituteEnvVars(this.config.getPort()) + "/" + this.systemVariableResolver.substituteEnvVars(this.config.getName()) + this.config.getParameters();
                hikariConfig.setDriverClassName("org.postgresql.Driver");
                hikariConfig.setUsername(this.systemVariableResolver.substituteEnvVars(this.config.getUser()));
                hikariConfig.setPassword(this.systemVariableResolver.substituteEnvVars(this.config.getPassword()));
                hikariConfig.setMaximumPoolSize(8);
                hikariConfig.setMinimumIdle(2);
                hikariConfig.addDataSourceProperty("prepStmtCacheSize", (Object)"500");
                hikariConfig.addDataSourceProperty("prepStmtCacheSqlLimit", (Object)"4096");
                break;
            }
            case 1: {
                this.reflectionResolver.hasClassOrElse("org.h2.Driver", libraryResolver -> libraryResolver.loadLibrary(Library.builder().groupId("com{}h2database").artifactId("h2").version("2.3.232").build()));
                connectionURL = connectionURL + "file:./" + this.projectPath.toString() + File.separator + this.systemVariableResolver.substituteEnvVars(this.config.getName()) + ".h2;DB_CLOSE_DELAY=-1;MODE=MySQL";
                hikariConfig.setDriverClassName("org.h2.Driver");
                hikariConfig.setMaximumPoolSize(5);
                hikariConfig.setMinimumIdle(1);
                hikariConfig.setConnectionTimeout(30000L);
                hikariConfig.addDataSourceProperty("cachePrepStmts", (Object)"true");
                hikariConfig.addDataSourceProperty("prepStmtCacheSize", (Object)"500");
                hikariConfig.addDataSourceProperty("prepStmtCacheSqlLimit", (Object)"4096");
                break;
            }
            case 2: {
                connectionURL = connectionURL + this.projectPath.toString() + File.separator + this.systemVariableResolver.substituteEnvVars(this.config.getName()) + ".db";
                hikariConfig.setMaximumPoolSize(5);
                hikariConfig.setMinimumIdle(1);
                hikariConfig.setConnectionTimeout(30000L);
                hikariConfig.addDataSourceProperty("busy_timeout", (Object)30000);
                hikariConfig.addDataSourceProperty("journal_mode", (Object)"WAL");
                hikariConfig.addDataSourceProperty("synchronous", (Object)"NORMAL");
                hikariConfig.addDataSourceProperty("journal_size_limit", (Object)"6144000");
                break;
            }
            case 3: {
                this.reflectionResolver.hasClassOrElse("com.mysql.jdbc.Driver", libraryResolver -> libraryResolver.loadLibrary(Library.builder().groupId("com{}mysql").artifactId("mysql-connector-j").version("9.4.0").build()));
                connectionURL = connectionURL + "//" + this.systemVariableResolver.substituteEnvVars(this.config.getHost()) + ":" + this.systemVariableResolver.substituteEnvVars(this.config.getPort()) + "/" + this.systemVariableResolver.substituteEnvVars(this.config.getName()) + this.config.getParameters();
                hikariConfig.setUsername(this.systemVariableResolver.substituteEnvVars(this.config.getUser()));
                hikariConfig.setPassword(this.systemVariableResolver.substituteEnvVars(this.config.getPassword()));
                hikariConfig.setMaximumPoolSize(8);
                hikariConfig.setMinimumIdle(2);
                hikariConfig.addDataSourceProperty("cachePrepStmts", (Object)"true");
                hikariConfig.addDataSourceProperty("prepStmtCacheSize", (Object)"500");
                hikariConfig.addDataSourceProperty("prepStmtCacheSqlLimit", (Object)"4096");
                hikariConfig.addDataSourceProperty("useServerPrepStmts", (Object)"true");
                hikariConfig.addDataSourceProperty("rewriteBatchedStatements", (Object)"true");
                break;
            }
            default: {
                throw new IllegalStateException(String.valueOf((Object)this.config.getType()) + " not supported");
            }
        }
        hikariConfig.setJdbcUrl(connectionURL);
        hikariConfig.setPoolName("FlectonePulseDatabase");
        return hikariConfig;
    }

    private void executeSQLFile(InputStream inputStream) throws IOException {
        String line;
        BufferedReader bufferedReader = new BufferedReader(new InputStreamReader(inputStream));
        StringBuilder builder = new StringBuilder();
        while ((line = bufferedReader.readLine()) != null) {
            if ((line = line.trim()).isEmpty() || line.startsWith("--")) continue;
            builder.append(line);
            if (!line.endsWith(";")) continue;
            String sql = builder.toString();
            this.getJdbi().useHandle(handle -> handle.execute(sql, new Object[0]));
            builder.setLength(0);
        }
    }

    private void MIGRATION_0_9_0() {
        this.backupDatabase();
        SettingDAO settingDAO = (SettingDAO)this.injector.getInstance(SettingDAO.class);
        ((FPlayerDAO)this.injector.getInstance(FPlayerDAO.class)).getFPlayers().forEach(fPlayer -> {
            if (fPlayer.isUnknown()) {
                return;
            }
            fPlayer.setSetting(FPlayer.Setting.ANON);
            settingDAO.insertOrUpdate((FPlayer)fPlayer, FPlayer.Setting.ANON);
        });
    }

    private void MIGRATION_1_3_0() {
        this.backupDatabase();
        try {
            InputStream sqlFile = this.platformServerAdapter.getResource("sqls/migrations/1_3_0.sql");
            this.executeSQLFile(sqlFile);
        }
        catch (IOException e) {
            this.fLogger.warning(e);
        }
    }

    private void backupDatabase() {
        if (this.config.getType() == Type.SQLITE) {
            String databaseName = this.systemVariableResolver.substituteEnvVars(this.config.getName()) + ".db";
            String timeStamp = new SimpleDateFormat("yyyy-MM-dd_HH.mm.ss").format(new Date());
            String copiedDatabaseName = databaseName + "_backup_" + timeStamp;
            try {
                Files.copy(this.projectPath.resolve(databaseName), this.projectPath.resolve(copiedDatabaseName), new CopyOption[0]);
            }
            catch (IOException e) {
                this.fLogger.warning(e);
            }
        }
    }

    public static enum Type {
        POSTGRESQL,
        H2,
        SQLITE,
        MYSQL;

    }
}

