package de.keksuccino.fancymenu.customization.panorama;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.UUID;
import net.minecraft.class_10366;
import net.minecraft.class_1060;
import net.minecraft.class_10799;
import net.minecraft.class_11279;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_332;
import net.minecraft.class_4068;
import net.minecraft.class_751;
import net.minecraft.class_9848;
import com.mojang.blaze3d.systems.RenderSystem;
import de.keksuccino.fancymenu.util.ScreenUtils;
import de.keksuccino.fancymenu.util.resource.ResourceSource;
import de.keksuccino.fancymenu.util.resource.ResourceSourceType;
import de.keksuccino.fancymenu.util.resource.ResourceSupplier;
import de.keksuccino.fancymenu.util.resource.resources.texture.ITexture;
import de.keksuccino.konkrete.math.MathUtils;
import de.keksuccino.fancymenu.util.properties.PropertyContainer;
import de.keksuccino.fancymenu.util.properties.PropertiesParser;
import de.keksuccino.fancymenu.util.properties.PropertyContainerSet;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

@SuppressWarnings("unused")
public class LocalTexturePanoramaRenderer implements class_4068, AutoCloseable {

	private static final Logger LOGGER = LogManager.getLogger();

	@NotNull
	public File propertiesFile;
	@NotNull
	public File panoramaImageDir;
	@Nullable
	public File overlayImageFile;
	protected String name = null;
	public final List<ResourceSupplier<ITexture>> panoramaImageSuppliers = new ArrayList<>();
	@Nullable
	public ResourceSupplier<ITexture> overlayTextureSupplier;
	protected float speed = 1.0F;
	protected double fov = 85.0D;
	protected float angle = 25.0F;
	public float opacity = 1.0F;
	protected volatile boolean tickerThreadRunning = false;
	protected volatile float currentRotation = 0.0F; //0 - 360
	protected volatile long lastRenderCall = -1L;
	@Nullable
	private class_751 cubeMap = null;
	@Nullable
	private PanoramaCubeMapTexture panoramaCubeMapTexture = null;
	@Nullable
	private class_2960 cubeMapLocation = null;
	@Nullable
	private class_11279 projectionMatrixBuffer = null;
	private final String uniqueId = UUID.randomUUID().toString();

	@Nullable
	public static LocalTexturePanoramaRenderer build(@NotNull File propertiesFile, @NotNull File panoramaImageDir, @Nullable File overlayImageFile) {
		LocalTexturePanoramaRenderer renderer = new LocalTexturePanoramaRenderer(propertiesFile, panoramaImageDir, overlayImageFile);
		try {
			if (renderer.propertiesFile.isFile() && renderer.panoramaImageDir.isDirectory()) {
				PropertyContainerSet panoProperties = PropertiesParser.deserializeSetFromFile(renderer.propertiesFile.getAbsolutePath());
				if (panoProperties != null) {
					PropertyContainer panoMeta = panoProperties.getFirstContainerOfType("panorama-meta");
					if (panoMeta != null) {
						renderer.name = panoMeta.getValue("name");
						if (renderer.name == null) {
							LOGGER.error("[FANCYMENU] Unable to load panorama! Missing 'name' value in panorama meta section: " + renderer.propertiesFile.getAbsolutePath(), new NullPointerException());
							return null;
						}
						String sp = panoMeta.getValue("speed");
						if ((sp != null) && MathUtils.isFloat(sp)) {
							renderer.speed = Float.parseFloat(sp);
						}
						String fo = panoMeta.getValue("fov");
						if ((fo != null) && MathUtils.isDouble(fo)) {
							renderer.fov = Double.parseDouble(fo);
						}
						String an = panoMeta.getValue("angle");
						if ((an != null) && MathUtils.isFloat(an)) {
							renderer.angle = Float.parseFloat(an);
						}
						String rot = panoMeta.getValue("start_rotation");
						if ((rot != null) && MathUtils.isFloat(rot)) {
							renderer.currentRotation = Float.parseFloat(rot);
							if ((renderer.currentRotation > 360.0F) || (renderer.currentRotation < 0.0F)) {
								renderer.currentRotation = 0;
							}
						}
						renderer.prepare();
						renderer.initialize();
						return renderer;
					} else {
						LOGGER.error("[FANCYMENU] Unable to load panorama! Missing 'panorama-meta' section in properties instance: " + renderer.propertiesFile.getAbsolutePath(), new NullPointerException());
					}
				} else {
					LOGGER.error("[FANCYMENU] Unable to load panorama! Parsed properties instance was NULL: " + renderer.propertiesFile.getAbsolutePath(), new NullPointerException());
				}
			}
		} catch (Exception ex) {
			LOGGER.error("[FANCYMENU] An error happened while trying to build a cubic panorama!", ex);
		}
		return null;
	}

	protected LocalTexturePanoramaRenderer(@NotNull File propertiesFile, @NotNull File panoramaImageDir, @Nullable File overlayImageFile) {
		this.propertiesFile = Objects.requireNonNull(propertiesFile);
		this.panoramaImageDir = Objects.requireNonNull(panoramaImageDir);
		this.overlayImageFile = overlayImageFile;
	}

	protected void prepare() {

		this.panoramaImageSuppliers.clear();
		this.overlayTextureSupplier = null;

		for (int i = 0; i < 6; i++) {
			File panoImage = new File(this.panoramaImageDir, "panorama_" + i + ".png");
			if (panoImage.isFile()) {
				this.panoramaImageSuppliers.add(ResourceSupplier.image(ResourceSource.of(panoImage.getAbsolutePath(), ResourceSourceType.LOCAL).getSourceWithPrefix()));
			} else {
				LOGGER.error("[FANCYMENU] Unable to load panorama! Missing panorama image 'panorama_" + i + ".png': " + this.name);
				return;
			}
		}

		if ((this.overlayImageFile != null) && this.overlayImageFile.isFile()) {
			this.overlayTextureSupplier = ResourceSupplier.image(ResourceSource.of(this.overlayImageFile.getAbsolutePath(), ResourceSourceType.LOCAL).getSourceWithPrefix());
		}

	}

	protected void initialize() {
		// Create custom cube map texture
		this.panoramaCubeMapTexture = new PanoramaCubeMapTexture(this.name != null ? this.name : "unnamed", this.panoramaImageSuppliers);
		
		// Register the texture with a unique ResourceLocation
		this.cubeMapLocation = class_2960.method_60655("fancymenu", "panorama_" + this.uniqueId);
		class_1060 textureManager = class_310.method_1551().method_1531();
		textureManager.method_4616(this.cubeMapLocation, this.panoramaCubeMapTexture);
		
		// Create CubeMap instance for rendering
		this.cubeMap = new class_751(this.cubeMapLocation);
		
		// Create projection matrix buffer
		this.projectionMatrixBuffer = new class_11279("fancymenu_panorama_" + this.uniqueId, 0.05F, 10.0F);
	}

	@SuppressWarnings("all")
	protected void startTickerThreadIfNeeded() {

		if (this.tickerThreadRunning) return;

		this.lastRenderCall = System.currentTimeMillis();
		this.tickerThreadRunning = true;

		new Thread(() -> {
			while ((this.lastRenderCall + 5000L) > System.currentTimeMillis()) {
				try {
					this.currentRotation += 0.03F;
					if (this.currentRotation >= 360) {
						this.currentRotation = 0;
					}
				} catch (Exception ex) {
					LOGGER.error("[FANCYMENU] Error while ticking panorama!", ex);
				}
				try {
					Thread.sleep(Math.max(2, (int)(20 / this.speed)));
				} catch (Exception ex) {
					LOGGER.error("[FANCYMENU] Error while ticking panorama!", ex);
				}
			}
			this.tickerThreadRunning = false;
		}, "FancyMenu Panorama Ticker Thread").start();

	}

	@Override
	public void method_25394(@NotNull class_332 graphics, int mouseX, int mouseY, float partial) {

		this.lastRenderCall = System.currentTimeMillis();
		this.startTickerThreadIfNeeded();

		// Check if cube map texture is loaded
		if ((this.panoramaCubeMapTexture == null) || this.panoramaCubeMapTexture.isLoadFailed()) {
			graphics.method_25290(class_10799.field_56883, ITexture.MISSING_TEXTURE_LOCATION, 0, 0, 0.0F, 0.0F, ScreenUtils.getScreenWidth(), ScreenUtils.getScreenHeight(), ScreenUtils.getScreenWidth(), ScreenUtils.getScreenHeight());
			return;
		}

		if (!this.panoramaCubeMapTexture.isLoaded()) {
			// Trigger loading by trying to get the texture view
			this.panoramaCubeMapTexture.method_71659();
			// Show loading screen or previous frame while loading
			return;
		}

		this._render(graphics, class_310.method_1551(), this.opacity);

	}

	private void _render(@NotNull class_332 graphics, class_310 mc, float alpha) {

		int screenW = ScreenUtils.getScreenWidth();
		int screenH = ScreenUtils.getScreenHeight();

		float pitch = this.angle;
		float yaw = -this.currentRotation;
		
		// Render the cube map using vanilla's approach with custom alpha support
		this.renderCubeMapWithAlpha(mc, pitch, yaw, alpha);

		// Render overlay if present
		if (this.overlayTextureSupplier != null) {
			ITexture texture = this.overlayTextureSupplier.get();
			if (texture != null) {
				class_2960 location = texture.getResourceLocation();
				if (location != null) {
					graphics.method_25291(class_10799.field_56883, location, 0, 0, 0.0F, 0.0F, screenW, screenH, screenW, screenH, class_9848.method_61317(this.opacity));
				}
			}
		}

	}

	private void renderCubeMapWithAlpha(class_310 mc, float pitch, float yaw, float alpha) {
		// We can't use CubeMap.render() directly because it doesn't support custom FOV and alpha
		// So we need to set our own projection matrix with custom FOV before rendering
		
		if (this.cubeMap != null && this.projectionMatrixBuffer != null) {
			// Backup current projection matrix
			RenderSystem.backupProjectionMatrix();
			
			// Set our custom projection matrix with desired FOV
			RenderSystem.setProjectionMatrix(
				this.projectionMatrixBuffer.method_71095(mc.method_22683().method_4489(), mc.method_22683().method_4506(), (float)this.fov), 
				class_10366.field_54953
			);
			
			// Apply alpha by modifying shader color
//			RenderSystem.setShaderColor(1.0F, 1.0F, 1.0F, alpha);
			
			// Use vanilla cube map render
			this.cubeMap.method_3156(mc, pitch, yaw);
			
			// Reset shader color
//			RenderSystem.setShaderColor(1.0F, 1.0F, 1.0F, 1.0F);
			
			// Restore original projection matrix
			RenderSystem.restoreProjectionMatrix();
		}
	}

	public String getName() {
		return this.name;
	}

	public void setSpeed(float speed) {
		if (speed < 0.0F) {
			speed = 0.0F;
		}
		this.speed = speed;
	}

	public void setFov(double fov) {
		if (fov > 179.0D) {
			fov = 179.0D;
		}
		this.fov = fov;
	}

	public void setAngle(float angle) {
		this.angle = angle;
	}

	@Override
	public void close() {
		if (this.cubeMap != null) {
			this.cubeMap.close();
			this.cubeMap = null;
		}
		if (this.panoramaCubeMapTexture != null) {
			this.panoramaCubeMapTexture.close();
			this.panoramaCubeMapTexture = null;
		}
		if (this.projectionMatrixBuffer != null) {
			this.projectionMatrixBuffer.close();
			this.projectionMatrixBuffer = null;
		}
		if (this.cubeMapLocation != null) {
			class_310.method_1551().method_1531().method_4615(this.cubeMapLocation);
			this.cubeMapLocation = null;
		}
	}

}
