package de.keksuccino.fancymenu.util.rendering;

import com.mojang.blaze3d.pipeline.RenderPipeline;
import javax.annotation.Nullable;
import net.minecraft.class_11231;
import net.minecraft.class_11244;
import net.minecraft.class_4588;
import net.minecraft.class_8030;
import org.joml.Matrix3x2f;

/**
 * Represents the state required to render a textured quadrilateral (a "blit" operation)
 * with floating-point coordinates.
 * This immutable record is used by the GUI rendering system to batch draw calls.
 *
 * @param pipeline The rendering pipeline to use for this element.
 * @param textureSetup The texture setup, defining which textures are bound.
 * @param transform The transformation matrix to apply to the vertices.
 * @param minX The minimum X coordinate of the quad.
 * @param minY The minimum Y coordinate of the quad.
 * @param maxX The maximum X coordinate of the quad.
 * @param maxY The maximum Y coordinate of the quad.
 * @param minU The minimum U texture coordinate (horizontal).
 * @param maxU The maximum U texture coordinate (horizontal).
 * @param minV The minimum V texture coordinate (vertical).
 * @param maxV The maximum V texture coordinate (vertical).
 * @param color The color tint to apply to the texture, in ARGB format.
 * @param scissorArea The optional scissor rectangle for clipping.
 * @param bounds The pre-calculated bounding box of this element after transformation and clipping, used for culling.
 */
public record FloatBlitRenderState(RenderPipeline pipeline, class_11231 textureSetup, Matrix3x2f transform, float minX, float minY, float maxX, float maxY, float minU, float maxU, float minV, float maxV, int color, @Nullable class_8030 scissorArea, @Nullable class_8030 bounds) implements class_11244 {

    /**
     * Creates a new blit render state, automatically calculating the bounding box.
     *
     * @param pipeline The rendering pipeline to use for this element.
     * @param textureSetup The texture setup, defining which textures are bound.
     * @param transform The transformation matrix to apply to the vertices.
     * @param minX The minimum X coordinate of the quad.
     * @param minY The minimum Y coordinate of the quad.
     * @param maxX The maximum X coordinate of the quad.
     * @param maxY The maximum Y coordinate of the quad.
     * @param minU The minimum U texture coordinate (horizontal).
     * @param maxU The maximum U texture coordinate (horizontal).
     * @param minV The minimum V texture coordinate (vertical).
     * @param maxV The maximum V texture coordinate (vertical).
     * @param color The color tint to apply to the texture, in ARGB format.
     * @param scissorRectangle The optional scissor rectangle for clipping.
     */
    public FloatBlitRenderState(RenderPipeline pipeline, class_11231 textureSetup, Matrix3x2f transform, float minX, float minY, float maxX, float maxY, float minU, float maxU, float minV, float maxV, int color, @Nullable class_8030 scissorRectangle) {
        this(pipeline, textureSetup, transform, minX, minY, maxX, maxY, minU, maxU, minV, maxV, color, scissorRectangle, getBounds(minX, minY, maxX, maxY, transform, scissorRectangle));
    }

    /**
     * Adds the four vertices of the textured quad to the given vertex consumer.
     *
     * @param vertexConsumer The consumer to which the vertices will be added.
     */
    @Override
    public void method_70917(class_4588 vertexConsumer) {
        vertexConsumer.method_70815(this.transform(), this.minX(), this.minY()).method_22913(this.minU(), this.minV()).method_39415(this.color());
        vertexConsumer.method_70815(this.transform(), this.minX(), this.maxY()).method_22913(this.minU(), this.maxV()).method_39415(this.color());
        vertexConsumer.method_70815(this.transform(), this.maxX(), this.maxY()).method_22913(this.maxU(), this.maxV()).method_39415(this.color());
        vertexConsumer.method_70815(this.transform(), this.maxX(), this.minY()).method_22913(this.maxU(), this.minV()).method_39415(this.color());
    }

    /**
     * Calculates the screen-space bounding box for the quad after transformation and clipping.
     *
     * @param minX The minimum X coordinate of the quad.
     * @param minY The minimum Y coordinate of the quad.
     * @param maxX The maximum X coordinate of the quad.
     * @param maxY The maximum Y coordinate of the quad.
     * @param transform The transformation matrix applied to the quad.
     * @param scissorRectangle The scissor rectangle used for clipping.
     * @return The intersected bounding box, or {@code null} if the intersection is empty.
     */
    @Nullable
    private static class_8030 getBounds(float minX, float minY, float maxX, float maxY, Matrix3x2f transform, @Nullable class_8030 scissorRectangle) {
        // Since ScreenRectangle uses integer coordinates, we must cast the floats.
        // This captures the intended area for culling, even though rendering uses floats.
        int x = (int)minX;
        int y = (int)minY;
        int width = (int)(maxX - minX);
        int height = (int)(maxY - minY);

        class_8030 elementBounds = new class_8030(x, y, width, height).method_71523(transform);
        return scissorRectangle != null ? scissorRectangle.method_49701(elementBounds) : elementBounds;
    }

}