package de.keksuccino.fancymenu.util.rendering;

import com.mojang.blaze3d.pipeline.RenderPipeline;
import javax.annotation.Nullable;
import net.minecraft.class_11231;
import net.minecraft.class_11244;
import net.minecraft.class_4588;
import net.minecraft.class_8030;
import org.joml.Matrix3x2f;

/**
 * Represents the state required to render a colored rectangle, which can have a solid color
 * or a vertical gradient. This uses floating-point coordinates for precision.
 * This immutable record is used by the GUI rendering system to batch draw calls.
 *
 * @param pipeline The rendering pipeline to use for this element.
 * @param textureSetup The texture setup, which is typically {@link class_11231#method_70899()} for solid colors.
 * @param transform The transformation matrix to apply to the vertices.
 * @param minX The minimum X coordinate of the rectangle.
 * @param minY The minimum Y coordinate of the rectangle.
 * @param maxX The maximum X coordinate of the rectangle.
 * @param maxY The maximum Y coordinate of the rectangle.
 * @param startColor The color of the top edge of the rectangle (or the entire rectangle if endColor is the same), in ARGB format.
 * @param endColor The color of the bottom edge of the rectangle, for creating gradients, in ARGB format.
 * @param scissorArea The optional scissor rectangle for clipping.
 * @param bounds The pre-calculated bounding box of this element after transformation and clipping, used for culling.
 */
public record FloatColoredRectangleRenderState(RenderPipeline pipeline, class_11231 textureSetup, Matrix3x2f transform, float minX, float minY, float maxX, float maxY, int startColor, int endColor, @Nullable class_8030 scissorArea, @Nullable class_8030 bounds) implements class_11244 {

    /**
     * Creates a new colored rectangle render state, automatically calculating the bounding box.
     *
     * @param pipeline The rendering pipeline to use for this element.
     * @param textureSetup The texture setup.
     * @param transform The transformation matrix to apply to the vertices.
     * @param minX The minimum X coordinate of the rectangle.
     * @param minY The minimum Y coordinate of the rectangle.
     * @param maxX The maximum X coordinate of the rectangle.
     * @param maxY The maximum Y coordinate of the rectangle.
     * @param startColor The color for the top edge.
     * @param endColor The color for the bottom edge.
     * @param scissorRectangle The optional scissor rectangle for clipping.
     */
    public FloatColoredRectangleRenderState(RenderPipeline pipeline, class_11231 textureSetup, Matrix3x2f transform, float minX, float minY, float maxX, float maxY, int startColor, int endColor, @Nullable class_8030 scissorRectangle) {
        this(pipeline, textureSetup, transform, minX, minY, maxX, maxY, startColor, endColor, scissorRectangle, getBounds(minX, minY, maxX, maxY, transform, scissorRectangle));
    }

    /**
     * Adds the four vertices of the colored rectangle to the given vertex consumer.
     *
     * @param vertexConsumer The consumer to which the vertices will be added.
     */
    @Override
    public void method_70917(class_4588 vertexConsumer) {
        // Top-left vertex with startColor
        vertexConsumer.method_70815(this.transform(), this.minX(), this.minY()).method_39415(this.startColor());
        // Bottom-left vertex with endColor
        vertexConsumer.method_70815(this.transform(), this.minX(), this.maxY()).method_39415(this.endColor());
        // Bottom-right vertex with endColor
        vertexConsumer.method_70815(this.transform(), this.maxX(), this.maxY()).method_39415(this.endColor());
        // Top-right vertex with startColor
        vertexConsumer.method_70815(this.transform(), this.maxX(), this.minY()).method_39415(this.startColor());
    }

    /**
     * Calculates the screen-space bounding box for the rectangle after transformation and clipping.
     *
     * @param minX The minimum X coordinate of the rectangle.
     * @param minY The minimum Y coordinate of the rectangle.
     * @param maxX The maximum X coordinate of the rectangle.
     * @param maxY The maximum Y coordinate of the rectangle.
     * @param transform The transformation matrix applied to the rectangle.
     * @param scissorRectangle The scissor rectangle used for clipping.
     * @return The intersected bounding box, or {@code null} if the intersection is empty.
     */
    @Nullable
    private static class_8030 getBounds(float minX, float minY, float maxX, float maxY, Matrix3x2f transform, @Nullable class_8030 scissorRectangle) {
        // Cast to int for ScreenRectangle, which is used for integer-based culling.
        int x = (int)minX;
        int y = (int)minY;
        int width = (int)(maxX - minX);
        int height = (int)(maxY - minY);

        class_8030 elementBounds = new class_8030(x, y, width, height).method_71523(transform);
        return scissorRectangle != null ? scissorRectangle.method_49701(elementBounds) : elementBounds;
    }

}