package de.keksuccino.fancymenu.customization.panorama;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import net.minecraft.class_1159;
import net.minecraft.class_1160;
import net.minecraft.class_287;
import net.minecraft.class_289;
import net.minecraft.class_290;
import net.minecraft.class_293;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_4587;
import net.minecraft.class_757;
import com.mojang.blaze3d.systems.RenderSystem;
import com.mojang.blaze3d.vertex.*;
import de.keksuccino.fancymenu.util.ScreenUtils;
import de.keksuccino.fancymenu.util.rendering.RenderingUtils;
import de.keksuccino.fancymenu.util.rendering.gui.GuiGraphics;
import de.keksuccino.fancymenu.util.rendering.gui.Renderable;
import de.keksuccino.fancymenu.util.resource.ResourceSource;
import de.keksuccino.fancymenu.util.resource.ResourceSourceType;
import de.keksuccino.fancymenu.util.resource.ResourceSupplier;
import de.keksuccino.fancymenu.util.resource.resources.texture.ITexture;
import de.keksuccino.konkrete.math.MathUtils;
import de.keksuccino.fancymenu.util.properties.PropertyContainer;
import de.keksuccino.fancymenu.util.properties.PropertiesParser;
import de.keksuccino.fancymenu.util.properties.PropertyContainerSet;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

@SuppressWarnings("unused")
public class LocalTexturePanoramaRenderer implements Renderable {

	private static final Logger LOGGER = LogManager.getLogger();

	@NotNull
	public File propertiesFile;
	@NotNull
	public File panoramaImageDir;
	@Nullable
	public File overlayImageFile;
	protected String name = null;
	public final List<ResourceSupplier<ITexture>> panoramaImageSuppliers = new ArrayList<>();
	@Nullable
	public ResourceSupplier<ITexture> overlayTextureSupplier;
	protected float speed = 1.0F;
	protected double fov = 85.0D;
	protected float angle = 25.0F;
	public float opacity = 1.0F;
	protected volatile boolean tickerThreadRunning = false;
	protected volatile float currentRotation = 0.0F; //0 - 360
	protected volatile long lastRenderCall = -1L;

	@Nullable
	public static LocalTexturePanoramaRenderer build(@NotNull File propertiesFile, @NotNull File panoramaImageDir, @Nullable File overlayImageFile) {
		LocalTexturePanoramaRenderer renderer = new LocalTexturePanoramaRenderer(propertiesFile, panoramaImageDir, overlayImageFile);
		try {
			if (renderer.propertiesFile.isFile() && renderer.panoramaImageDir.isDirectory()) {
				PropertyContainerSet panoProperties = PropertiesParser.deserializeSetFromFile(renderer.propertiesFile.getAbsolutePath());
				if (panoProperties != null) {
					PropertyContainer panoMeta = panoProperties.getFirstContainerOfType("panorama-meta");
					if (panoMeta != null) {
						renderer.name = panoMeta.getValue("name");
						if (renderer.name == null) {
							LOGGER.error("[FANCYMENU] Unable to load panorama! Missing 'name' value in panorama meta section: " + renderer.propertiesFile.getAbsolutePath(), new NullPointerException());
							return null;
						}
						String sp = panoMeta.getValue("speed");
						if ((sp != null) && MathUtils.isFloat(sp)) {
							renderer.speed = Float.parseFloat(sp);
						}
						String fo = panoMeta.getValue("fov");
						if ((fo != null) && MathUtils.isDouble(fo)) {
							renderer.fov = Double.parseDouble(fo);
						}
						String an = panoMeta.getValue("angle");
						if ((an != null) && MathUtils.isFloat(an)) {
							renderer.angle = Float.parseFloat(an);
						}
						String rot = panoMeta.getValue("start_rotation");
						if ((rot != null) && MathUtils.isFloat(rot)) {
							renderer.currentRotation = Float.parseFloat(rot);
							if ((renderer.currentRotation > 360.0F) || (renderer.currentRotation < 0.0F)) {
								renderer.currentRotation = 0;
							}
						}
						renderer.prepare();
						return renderer;
					} else {
						LOGGER.error("[FANCYMENU] Unable to load panorama! Missing 'panorama-meta' section in properties instance: " + renderer.propertiesFile.getAbsolutePath(), new NullPointerException());
					}
				} else {
					LOGGER.error("[FANCYMENU] Unable to load panorama! Parsed properties instance was NULL: " + renderer.propertiesFile.getAbsolutePath(), new NullPointerException());
				}
			}
		} catch (Exception ex) {
			LOGGER.error("[FANCYMENU] An error happened while trying to build a cubic panorama!", ex);
		}
		return null;
	}

	protected LocalTexturePanoramaRenderer(@NotNull File propertiesFile, @NotNull File panoramaImageDir, @Nullable File overlayImageFile) {
		this.propertiesFile = Objects.requireNonNull(propertiesFile);
		this.panoramaImageDir = Objects.requireNonNull(panoramaImageDir);
		this.overlayImageFile = overlayImageFile;
	}

	protected void prepare() {

		this.panoramaImageSuppliers.clear();
		this.overlayTextureSupplier = null;

		for (int i = 0; i < 6; i++) {
			File panoImage = new File(this.panoramaImageDir, "panorama_" + i + ".png");
			if (panoImage.isFile()) {
				this.panoramaImageSuppliers.add(ResourceSupplier.image(ResourceSource.of(panoImage.getAbsolutePath(), ResourceSourceType.LOCAL).getSourceWithPrefix()));
			} else {
				LOGGER.error("[FANCYMENU] Unable to load panorama! Missing panorama image 'panorama_" + i + ".png': " + this.name);
				return;
			}
		}

		if ((this.overlayImageFile != null) && this.overlayImageFile.isFile()) {
			this.panoramaImageSuppliers.add(ResourceSupplier.image(ResourceSource.of(this.overlayImageFile.getAbsolutePath(), ResourceSourceType.LOCAL).getSourceWithPrefix()));
		}

	}

	@SuppressWarnings("all")
	protected void startTickerThreadIfNeeded() {

		if (this.tickerThreadRunning) return;

		this.lastRenderCall = System.currentTimeMillis();
		this.tickerThreadRunning = true;

		new Thread(() -> {
			while ((this.lastRenderCall + 5000L) > System.currentTimeMillis()) {
				try {
					this.currentRotation += 0.03F;
					if (this.currentRotation >= 360) {
						this.currentRotation = 0;
					}
				} catch (Exception ex) {
					LOGGER.error("[FANCYMENU] Error while ticking panorama!", ex);
				}
				try {
					Thread.sleep(Math.max(2, (int)(20 / this.speed)));
				} catch (Exception ex) {
					LOGGER.error("[FANCYMENU] Error while ticking panorama!", ex);
				}
			}
			this.tickerThreadRunning = false;
		}, "FancyMenu Panorama Ticker Thread").start();

	}

	@Override
	public void render(@NotNull GuiGraphics graphics, int mouseX, int mouseY, float partial) {
		this.lastRenderCall = System.currentTimeMillis();
		this.startTickerThreadIfNeeded();
		if (this.panoramaImageSuppliers.size() < 6) {
			RenderSystem.enableBlend();
			RenderingUtils.resetShaderColor(graphics);
			graphics.blit(ITexture.MISSING_TEXTURE_LOCATION, 0, 0, 0.0F, 0.0F, ScreenUtils.getScreenWidth(), ScreenUtils.getScreenHeight(), ScreenUtils.getScreenWidth(), ScreenUtils.getScreenHeight());
			RenderingUtils.resetShaderColor(graphics);
		} else {
			this.renderRaw(graphics.pose(), this.opacity);
		}
	}

	public void renderRaw(@NotNull class_4587 pose, float panoramaAlpha) {

		class_310 mc = class_310.method_1551();

		int screenW = ScreenUtils.getScreenWidth();
		int screenH = ScreenUtils.getScreenHeight();

		float pitch = this.angle;
		float yaw = -this.currentRotation;

		class_289 tesselator = class_289.method_1348();
		class_287 bufferBuilder = tesselator.method_1349();
		class_1159 matrix4f = class_1159.method_4929(this.fov, (float)mc.method_22683().method_4489() / (float)mc.method_22683().method_4506(), 0.05F, 10.0F);
		RenderSystem.backupProjectionMatrix();
		RenderSystem.setProjectionMatrix(matrix4f);
		class_4587 modelViewStack = RenderSystem.getModelViewStack();
		modelViewStack.method_22903();
		modelViewStack.method_34426();
		modelViewStack.method_22907(class_1160.field_20703.method_23214(180.0f));
		RenderSystem.applyModelViewMatrix();
		RenderSystem.setShader(class_757::method_34543);
		RenderSystem.setShaderColor(1.0f, 1.0f, 1.0f, this.opacity);
		RenderSystem.enableBlend();
		RenderSystem.disableCull();
		RenderSystem.depthMask(false);
		RenderSystem.disableDepthTest();

		for(int j = 0; j < 4; ++j) {
			modelViewStack.method_22903();
			float k = ((float)(j % 2) / 2.0f - 0.5f) / 256.0f;
			float l = ((float)(j / 2) / 2.0f - 0.5f) / 256.0f;
			modelViewStack.method_22904(k, l, 0.0f);
			modelViewStack.method_22907(class_1160.field_20703.method_23214(pitch));
			modelViewStack.method_22907(class_1160.field_20705.method_23214(yaw));
			RenderSystem.applyModelViewMatrix();
			for (int n = 0; n < 6; ++n) {
				class_2960 location = null;
				if (this.panoramaImageSuppliers.size() >= (n + 1)) {
					ResourceSupplier<ITexture> texSupplier = this.panoramaImageSuppliers.get(n);
					ITexture texture = texSupplier.get();
					if (texture != null) {
						location = texture.getResourceLocation();
					}
				}
				if (location == null) location = ITexture.MISSING_TEXTURE_LOCATION;
				RenderSystem.setShaderTexture(0, location);
				bufferBuilder.method_1328(class_293.class_5596.field_27382, class_290.field_1575);
				int o = Math.round(255.0F * panoramaAlpha) / (j + 1);
				if (n == 0) {
					bufferBuilder.method_22912(-1.0, -1.0, 1.0).method_22913(0.0F, 0.0F).method_1336(255, 255, 255, o).method_1344();
					bufferBuilder.method_22912(-1.0, 1.0, 1.0).method_22913(0.0F, 1.0F).method_1336(255, 255, 255, o).method_1344();
					bufferBuilder.method_22912(1.0, 1.0, 1.0).method_22913(1.0F, 1.0F).method_1336(255, 255, 255, o).method_1344();
					bufferBuilder.method_22912(1.0, -1.0, 1.0).method_22913(1.0F, 0.0F).method_1336(255, 255, 255, o).method_1344();
				}
				if (n == 1) {
					bufferBuilder.method_22912(1.0, -1.0, 1.0).method_22913(0.0F, 0.0F).method_1336(255, 255, 255, o).method_1344();
					bufferBuilder.method_22912(1.0, 1.0, 1.0).method_22913(0.0F, 1.0F).method_1336(255, 255, 255, o).method_1344();
					bufferBuilder.method_22912(1.0, 1.0, -1.0).method_22913(1.0F, 1.0F).method_1336(255, 255, 255, o).method_1344();
					bufferBuilder.method_22912(1.0, -1.0, -1.0).method_22913(1.0F, 0.0F).method_1336(255, 255, 255, o).method_1344();
				}
				if (n == 2) {
					bufferBuilder.method_22912(1.0, -1.0, -1.0).method_22913(0.0F, 0.0F).method_1336(255, 255, 255, o).method_1344();
					bufferBuilder.method_22912(1.0, 1.0, -1.0).method_22913(0.0F, 1.0F).method_1336(255, 255, 255, o).method_1344();
					bufferBuilder.method_22912(-1.0, 1.0, -1.0).method_22913(1.0F, 1.0F).method_1336(255, 255, 255, o).method_1344();
					bufferBuilder.method_22912(-1.0, -1.0, -1.0).method_22913(1.0F, 0.0F).method_1336(255, 255, 255, o).method_1344();
				}
				if (n == 3) {
					bufferBuilder.method_22912(-1.0, -1.0, -1.0).method_22913(0.0F, 0.0F).method_1336(255, 255, 255, o).method_1344();
					bufferBuilder.method_22912(-1.0, 1.0, -1.0).method_22913(0.0F, 1.0F).method_1336(255, 255, 255, o).method_1344();
					bufferBuilder.method_22912(-1.0, 1.0, 1.0).method_22913(1.0F, 1.0F).method_1336(255, 255, 255, o).method_1344();
					bufferBuilder.method_22912(-1.0, -1.0, 1.0).method_22913(1.0F, 0.0F).method_1336(255, 255, 255, o).method_1344();
				}
				if (n == 4) {
					bufferBuilder.method_22912(-1.0, -1.0, -1.0).method_22913(0.0F, 0.0F).method_1336(255, 255, 255, o).method_1344();
					bufferBuilder.method_22912(-1.0, -1.0, 1.0).method_22913(0.0F, 1.0F).method_1336(255, 255, 255, o).method_1344();
					bufferBuilder.method_22912(1.0, -1.0, 1.0).method_22913(1.0F, 1.0F).method_1336(255, 255, 255, o).method_1344();
					bufferBuilder.method_22912(1.0, -1.0, -1.0).method_22913(1.0F, 0.0F).method_1336(255, 255, 255, o).method_1344();
				}
				if (n == 5) {
					bufferBuilder.method_22912(-1.0, 1.0, 1.0).method_22913(0.0F, 0.0F).method_1336(255, 255, 255, o).method_1344();
					bufferBuilder.method_22912(-1.0, 1.0, -1.0).method_22913(0.0F, 1.0F).method_1336(255, 255, 255, o).method_1344();
					bufferBuilder.method_22912(1.0, 1.0, -1.0).method_22913(1.0F, 1.0F).method_1336(255, 255, 255, o).method_1344();
					bufferBuilder.method_22912(1.0, 1.0, 1.0).method_22913(1.0F, 0.0F).method_1336(255, 255, 255, o).method_1344();
				}
				tesselator.method_1350();
			}
			modelViewStack.method_22909();
			RenderSystem.applyModelViewMatrix();
			RenderSystem.colorMask(true, true, true, false);
		}

		RenderSystem.colorMask(true, true, true, true);
		RenderSystem.restoreProjectionMatrix();
		modelViewStack.method_22909();
		RenderSystem.applyModelViewMatrix();
		RenderSystem.depthMask(true);
		RenderSystem.enableCull();
		RenderSystem.enableDepthTest();

		GuiGraphics graphics = GuiGraphics.currentGraphics();

		if (this.overlayTextureSupplier != null) {
			ITexture texture = this.overlayTextureSupplier.get();
			if (texture != null) {
				class_2960 location = texture.getResourceLocation();
				if (location != null) {
					graphics.setColor(1.0F, 1.0F, 1.0F, this.opacity);
					RenderSystem.enableBlend();
					graphics.blit(location, 0, 0, 0.0F, 0.0F, screenW, screenH, screenW, screenH);
				}
			}
		}

		RenderingUtils.resetShaderColor(graphics);

	}

	public String getName() {
		return this.name;
	}

	public void setSpeed(float speed) {
		if (speed < 0.0F) {
			speed = 0.0F;
		}
		this.speed = speed;
	}

	public void setFov(double fov) {
		if (fov > 179.0D) {
			fov = 179.0D;
		}
		this.fov = fov;
	}

	public void setAngle(float angle) {
		this.angle = angle;
	}

}
