package de.keksuccino.fancymenu.networking.packets.placeholders.nbt;

import com.mojang.brigadier.StringReader;
import com.mojang.brigadier.exceptions.CommandSyntaxException;
import de.keksuccino.fancymenu.networking.PacketHandler;
import de.keksuccino.fancymenu.networking.packets.placeholders.nbt.ServerNbtDataResponsePacket.ResultType;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.List;
import java.util.Locale;
import net.minecraft.class_2105;
import net.minecraft.class_2168;
import net.minecraft.class_2203;
import net.minecraft.class_2262;
import net.minecraft.class_2267;
import net.minecraft.class_2300;
import net.minecraft.class_2303;
import net.minecraft.class_2338;
import net.minecraft.class_2487;
import net.minecraft.class_2512;
import net.minecraft.class_2514;
import net.minecraft.class_2520;
import net.minecraft.class_2561;
import net.minecraft.class_2586;
import net.minecraft.class_2960;
import net.minecraft.class_3218;
import net.minecraft.class_3222;
import net.minecraft.class_4565;

public class ServerSideServerNbtDataRequestPacketLogic {

    private static final Logger LOGGER = LogManager.getLogger();

    protected static boolean handle(@NotNull class_3222 sender, @NotNull ServerNbtDataRequestPacket packet) {
        if (packet.placeholder == null || packet.placeholder.isEmpty()) {
            LOGGER.warn("[FANCYMENU] Received malformed server NBT placeholder request without placeholder string.");
            return false;
        }

        String result = "";
        ResultType resultType = ResultType.EMPTY;

        try {
            CommandContextData context = buildContext(sender, packet);
            if (context != null) {
                class_2520 tag = resolveTag(context, packet);
                if (tag != null) {
                    result = convertResult(tag, packet, context);
                    resultType = ResultType.SUCCESS;
                }
            }
        } catch (Exception ex) {
            LOGGER.error("[FANCYMENU] Failed to resolve server-side NBT placeholder.", ex);
        }

        PacketHandler.sendToClient(sender, new ServerNbtDataResponsePacket(packet.placeholder, result, resultType));
        return true;
    }

    @Nullable
    private static CommandContextData buildContext(@NotNull class_3222 sender, @NotNull ServerNbtDataRequestPacket packet) throws CommandSyntaxException {
        if (packet.source_type == null) {
            return null;
        }
        String sourceType = packet.source_type.toLowerCase(Locale.ROOT);
        return switch (sourceType) {
            case "entity" -> resolveEntity(sender, packet.entity_selector);
            case "block" -> resolveBlock(sender, packet.block_pos);
            case "storage" -> resolveStorage(sender, packet.storage_id);
            default -> null;
        };
    }

    @Nullable
    private static CommandContextData resolveEntity(@NotNull class_3222 sender, @Nullable String selectorString) throws CommandSyntaxException {
        if ((selectorString == null) || selectorString.isEmpty()) {
            return null;
        }
        class_2303 parser = new class_2303(new StringReader(selectorString), true);
        class_2300 selector = parser.method_9882();
        class_2168 source = sender.method_5671();
        class_2487 tag = class_2105.method_9076(selector.method_9809(source));
        return new CommandContextData(tag, source);
    }

    @Nullable
    private static CommandContextData resolveBlock(@NotNull class_3222 sender, @Nullable String positionString) throws CommandSyntaxException {
        if ((positionString == null) || positionString.isEmpty()) {
            return null;
        }
        class_3218 level = (sender.field_6002 instanceof class_3218 l) ? l : null;
        if (level == null) return null;
        class_2168 source = sender.method_5671();
        class_2267 coordinates = class_2262.method_9698().method_9699(new StringReader(positionString));
        class_2338 pos = coordinates.method_9704(source);
        if (!level.method_22340(pos) || !level.method_24794(pos)) {
            return null;
        }
        class_2586 be = level.method_8321(pos);
        if (be == null) {
            return null;
        }
        class_2487 tag = be.method_38242();
        return new CommandContextData(tag, source);
    }

    @Nullable
    private static CommandContextData resolveStorage(@NotNull class_3222 sender, @Nullable String storageId) {
        if ((storageId == null) || storageId.isEmpty()) {
            return null;
        }
        try {
            class_2960 id = class_2960.method_12829(storageId);
            if (id == null) {
                return null;
            }
            class_2168 source = sender.method_5671();
            class_4565 storage = sender.method_5682().method_22827();
            class_2487 tag = storage.method_22546(id);
            return new CommandContextData(tag, source);
        } catch (Exception ex) {
            LOGGER.error("[FANCYMENU] Failed to resolve command storage for '{}'.", storageId, ex);
            return null;
        }
    }

    @Nullable
    private static class_2520 resolveTag(@NotNull CommandContextData context, @NotNull ServerNbtDataRequestPacket packet) throws CommandSyntaxException {
        if (context.baseTag() == null) {
            return null;
        }

        if ((packet.nbt_path == null) || packet.nbt_path.isEmpty()) {
            return context.baseTag();
        }

        class_2203.class_2209 path = class_2203.method_9360().method_9362(new StringReader(packet.nbt_path));
        List<class_2520> tags = path.method_9366(context.baseTag());
        if (tags.isEmpty()) {
            return null;
        }
        return tags.get(0);
    }

    @NotNull
    private static String convertResult(@NotNull class_2520 tag, @NotNull ServerNbtDataRequestPacket packet, @NotNull CommandContextData context) throws CommandSyntaxException {
        String returnType = packet.return_type == null ? "value" : packet.return_type.toLowerCase(Locale.ROOT);
        double scale = packet.scale == null ? 1.0D : packet.scale;

        switch (returnType) {
            case "string" -> {
                return tag.method_10714();
            }
            case "snbt" -> {
                return tag.toString();
            }
            case "json" -> {
                if (tag instanceof class_2487) {
                    class_2561 component = class_2512.method_32270(tag);
                    String json = class_2561.class_2562.method_10867(component);
                    if (json.startsWith("\"") && json.endsWith("\"") && (json.length() >= 2)) {
                        return json.substring(1, json.length() - 1);
                    }
                    return json;
                }
                return tag.toString();
            }
            default -> {
                if (tag instanceof class_2514 numericTag) {
                    return formatScaledNumeric(numericTag, scale);
                }
                return tag.method_10714();
            }
        }
    }

    @NotNull
    private static String formatScaledNumeric(@NotNull class_2514 tag, double scale) {
        if (scale != 1.0D) {
            return formatScaled(tag, scale);
        }
        return tag.method_10714();
    }

    @NotNull
    private static String formatScaled(@NotNull class_2514 tag, double scale) {
        double scaled = tag.method_10697() * scale;
        if (tag instanceof net.minecraft.class_2494) {
            return Float.toString((float) scaled) + "f";
        }
        if (tag instanceof net.minecraft.class_2489) {
            return Double.toString(scaled) + "d";
        }
        long rounded = Math.round(scaled);
        if (tag instanceof net.minecraft.class_2481) {
            return Byte.toString((byte) rounded) + "b";
        }
        if (tag instanceof net.minecraft.class_2516) {
            return Short.toString((short) rounded) + "s";
        }
        if (tag instanceof net.minecraft.class_2497) {
            return Integer.toString((int) rounded);
        }
        if (tag instanceof net.minecraft.class_2503) {
            return Long.toString(rounded) + "L";
        }
        return Long.toString(rounded);
    }

    private record CommandContextData(class_2520 baseTag, class_2168 source) {
    }

}
