package com.dks.client;

import com.google.common.collect.ImmutableSet;
import com.kbp.client.impl.IKeyMappingImpl;
import com.mojang.blaze3d.platform.InputConstants;
import com.mojang.blaze3d.platform.InputConstants.Key;
import com.mojang.datafixers.util.Pair;
import net.minecraft.client.KeyMapping;
import net.minecraft.client.Minecraft;
import net.minecraft.client.Options;
import net.minecraft.client.gui.screens.ConfirmScreen;
import net.minecraft.network.chat.TranslatableComponent;
import net.minecraftforge.client.ConfigGuiHandler.ConfigGuiFactory;
import net.minecraftforge.client.event.ScreenOpenEvent;
import net.minecraftforge.client.extensions.IForgeKeyMapping;
import net.minecraftforge.client.settings.KeyModifier;
import net.minecraftforge.common.MinecraftForge;
import net.minecraftforge.eventbus.api.SubscribeEvent;
import net.minecraftforge.fml.IExtensionPoint.DisplayTest;
import net.minecraftforge.fml.ModContainer;
import net.minecraftforge.fml.ModList;
import net.minecraftforge.fml.ModLoadingContext;
import net.minecraftforge.fml.common.Mod;
import net.minecraftforge.fml.common.Mod.EventBusSubscriber;
import net.minecraftforge.fml.config.IConfigEvent;
import net.minecraftforge.fml.config.ModConfig;
import net.minecraftforge.fml.config.ModConfig.Type;
import net.minecraftforge.fml.event.config.ModConfigEvent;
import net.minecraftforge.fml.javafmlmod.FMLModContainer;
import net.minecraftforge.fml.util.ObfuscationReflectionHelper;

import java.io.File;
import java.util.Arrays;
import java.util.EnumMap;
import java.util.Objects;
import java.util.function.BiConsumer;
import java.util.stream.Collectors;

@Mod( "default_key_setup" )
@EventBusSubscriber
public final class DKSMod
{
	private static void __saveDefaultKeySetup()
	{
		final var has_kbp_mod = ModList.get().isLoaded( "key_binding_patch" );
		final BiConsumer< KeyMapping, StringBuilder > append_value;
		if ( has_kbp_mod ) {
			append_value = ( km, builder ) -> builder.append( km.saveString() );
		}
		else
		{
			append_value = ( km, builder ) -> {
				builder.append( km.saveString() );
				builder.append( ':' );
				builder.append( km.getKeyModifier() );
			};
		}
		final var mc = Minecraft.getInstance();
		DKSModConfig.DEFAULT_KEY_SETUP.set(
			Arrays.stream( mc.options.keyMappings )
			.map( km -> {
				final StringBuilder builder = new StringBuilder( km.getName() );
				builder.append( '=' );
				append_value.accept( km, builder );
				return builder.toString();
			} )
			.toList()
		);
	}
	
	private static void __applyDefaultKeySetup()
	{
		final var has_kbp_mod = ModList.get().isLoaded( "key_binding_patch" );
		final var data = (
			DKSModConfig.DEFAULT_KEY_SETUP.get().stream()
			.map( s -> {
				final var idx = s.indexOf( '=' );
				final var name = s.substring( 0, idx );
				final var value = s.substring( idx + 1 );
				return Pair.of( name, value );
			} )
			.collect( Collectors.toMap( Pair::getFirst, Pair::getSecond ) )
		);
		
		final var mc = Minecraft.getInstance();
		for ( var km : mc.options.keyMappings )
		{
			final var value = data.get( km.getName() );
			if ( value == null ) {
				continue;
			}
			
			final var split = value.split( ":" );
			final var key = InputConstants.getKey( split[ 0 ] );
			final var modifier = KeyModifier.valueFromString( split[ 1 ] );
			KeyMappingAccess._setDefaultKey( km, key );
			KeyMappingAccess._setKeyModifierDefault( km, modifier );
			if ( has_kbp_mod )
			{
				final ImmutableSet< Key > cmb_keys;
				if ( split.length > 2 )
				{
					cmb_keys = (
						Arrays.stream( split[ 2 ].split( "\\+" ) )
						.map( InputConstants::getKey )
						.collect( ImmutableSet.toImmutableSet() )
					);
				}
				else {
					cmb_keys = IKeyMappingImpl.toCmbKeySet( modifier );
				}
				KeyMappingAccess._setDefaultCmbKeys( km, cmb_keys );
			}
		}
	}
	
	private final ModConfig config;
	public DKSMod()
	{
		// Make sure the mod being absent on the other network side does not
		// cause the client to display the server as incompatible.
		// Make sure the mod being absent on the other network side does not
		// cause the client to display the server as incompatible.
		final var load_ctx = ModLoadingContext.get();
		load_ctx.registerExtensionPoint(
			DisplayTest.class,
			() -> new DisplayTest(
				() -> "This is a client only mod.",
				( remote_version_string, network_bool ) -> network_bool
			)
		);
		
		// Setup mod config settings.
		load_ctx.registerConfig( Type.CLIENT, DKSModConfig.CONFIG_SPEC );
		load_ctx.registerExtensionPoint(
			ConfigGuiFactory.class,
			() -> new ConfigGuiFactory( ( mc, screen ) -> new ConfirmScreen(
				result -> {
					if ( result )
					{
						__saveDefaultKeySetup();
						
						// I think this is a bug in Forge. Saving the config this way may not trigger the reload event.
						final var container = ModList.get().getModContainerById( "default_key_setup" ).orElseThrow();
						container.dispatchConfigEvent( IConfigEvent.reloading( DKSMod.this.config ) );
					}
					mc.setScreen( screen );
				},
				new TranslatableComponent( "dks.gui.warning_title" ),
				new TranslatableComponent( "dks.gui.warning_msg" )
			) )
		);
		
		MinecraftForge.EVENT_BUS.register( new Object() {
			@SubscribeEvent
			void onOpenGui( ScreenOpenEvent evt )
			{
				// This event somehow could be fired during the mod loading phase, which is pointless
				// to process. So defer the event subscription after the game load complete.
				final var container = ( FMLModContainer ) ModList.get().getModContainerById( "default_key_setup" ).orElseThrow();
				container.getEventBus().addListener( ( ModConfigEvent.Reloading e ) -> __applyDefaultKeySetup() );
				
				final boolean should_reset_km;
				if ( DKSModConfig.FORCE_KEY_RESET.get() )
				{
					DKSModConfig.FORCE_KEY_RESET.set( false );
					should_reset_km = true;
					
					container.dispatchConfigEvent( IConfigEvent.reloading( DKSMod.this.config ) );
				}
				else
				{
					final var mc = Minecraft.getInstance();
					final File file = ObfuscationReflectionHelper.getPrivateValue( Options.class, mc.options, "f_92110_" );
					should_reset_km = !Objects.requireNonNull( file ).exists();
					
					__applyDefaultKeySetup();
				}
				
				if ( should_reset_km )
				{
					final var mc = Minecraft.getInstance();
					Arrays.stream( mc.options.keyMappings ).forEachOrdered( IForgeKeyMapping::setToDefault );
					KeyMapping.resetMapping();
				}
				
				MinecraftForge.EVENT_BUS.unregister( this );
			}
		} );
		
		final var container = ( FMLModContainer ) load_ctx.getActiveContainer();
		EnumMap< Type, ModConfig > configs = ObfuscationReflectionHelper.getPrivateValue( ModContainer.class, container, "configs" );
		assert configs != null;
		this.config = configs.get( Type.CLIENT );
	}
}
