package fr.aeldit.cyanlib.lib.config;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.reflect.TypeToken;
import com.mojang.brigadier.suggestion.Suggestions;
import com.mojang.brigadier.suggestion.SuggestionsBuilder;
import fr.aeldit.cyanlib.lib.utils.RULES;
import net.fabricmc.api.EnvType;
import net.fabricmc.api.Environment;
import net.fabricmc.loader.api.FabricLoader;
import net.minecraft.class_2172;
import net.minecraft.class_7172;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.io.IOException;
import java.io.Reader;
import java.io.Writer;
import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

public class CyanLibOptionsStorage
{
    private final String modid;
    private final ICyanLibConfig cyanLibConfigClass;
    // We use a synchronized list because 2 players can edit the config at the same time when in multiplayer
    private final List<IOption<?>> optionsList = Collections.synchronizedList(new ArrayList<>());

    public CyanLibOptionsStorage(String modid, ICyanLibConfig configClass)
    {
        this.modid              = modid;
        this.cyanLibConfigClass = configClass;
        readConfig();
    }

    public ICyanLibConfig getConfigClass()
    {
        return cyanLibConfigClass;
    }

    public List<String> getOptionsNames()
    {
        return optionsList.stream().map(IOption::getName).toList();
    }

    @Environment(EnvType.CLIENT)
    public static class_7172<?> @NotNull [] asConfigOptions(@NotNull ICyanLibConfig configClass)
    {
        ArrayList<class_7172<?>> options = new ArrayList<>(configClass.getClass().getDeclaredFields().length);

        for (Field field : configClass.getClass().getDeclaredFields())
        {
            try
            {
                options.add(((IOption<?>) field.get(null)).asConfigOption());
            }
            catch (IllegalAccessException e)
            {
                throw new RuntimeException(e);
            }
        }
        return options.toArray(class_7172[]::new);
    }

    public @Nullable Object getOptionValue(String optionName)
    {
        return optionsList.stream()
                          .filter(option -> optionName.equals(option.getName()))
                          .map(IOption::getValue)
                          .findFirst()
                          .orElse(null);
    }

    public boolean setOption(String optionName, Object value, boolean save)
    {
        IOption<?> option = optionsList.stream().filter(opt -> opt.getName().equals(optionName))
                                       .findFirst().orElse(null);
        if (option != null)
        {
            boolean success = option.setValue(value);
            if (save)
            {
                writeConfig();
            }
            return success;
        }
        return false;
    }

    public void resetOptions()
    {
        optionsList.forEach(IOption::reset);
    }

    public boolean optionExists(String optionName)
    {
        return optionsList.stream().anyMatch(option -> optionName.equals(option.getName()));
    }

    /**
     * Called for the command {@code /modid config <optionName>}
     *
     * @return a suggestion with the available options
     */
    public static CompletableFuture<Suggestions> getOptionsSuggestions(
            @NotNull SuggestionsBuilder builder,
            @NotNull CyanLibOptionsStorage optionsStorage
    )
    {
        return class_2172.method_9265(optionsStorage.getOptionsNames(), builder);
    }

    public boolean hasRule(String optionName, RULES rule)
    {
        return optionsList.stream().anyMatch(option -> option.getName().equals(optionName) && option.getRule() == rule);
    }

    private void readConfig()
    {
        Path path = FabricLoader.getInstance().getConfigDir().resolve("%s.json".formatted(modid));

        // If the file does not exist, we simply load the class in memory
        if (!Files.exists(path))
        {
            for (Field field : cyanLibConfigClass.getClass().getDeclaredFields())
            {
                if (Modifier.isPublic(field.getModifiers()) && Modifier.isStatic(field.getModifiers())
                    && Modifier.isFinal(field.getModifiers()))
                {
                    if (BooleanOption.class.isAssignableFrom(field.getType()))
                    {
                        try
                        {
                            BooleanOption booleanOption = (BooleanOption) field.get(null);
                            optionsList.add(booleanOption);
                        }
                        catch (IllegalAccessException e)
                        {
                            throw new RuntimeException(e);
                        }
                    }
                    else if (IntegerOption.class.isAssignableFrom(field.getType()))
                    {
                        try
                        {
                            IntegerOption integerOption = (IntegerOption) field.get(null);
                            optionsList.add(integerOption);
                        }
                        catch (IllegalAccessException e)
                        {
                            throw new RuntimeException(e);
                        }
                    }
                }
            }
        }
        // Otherwise, we load the config from the file
        else
        {
            Map<String, Object> config;

            try
            {
                Gson gson = new Gson();
                Reader reader = Files.newBufferedReader(path);
                TypeToken<Map<String, Object>> mapType = new TypeToken<>()
                {
                };
                config = gson.fromJson(reader, mapType);
                reader.close();
            }
            catch (IOException e)
            {
                throw new RuntimeException(e);
            }

            if (config != null && !config.isEmpty())
            {
                // If there are options present in teh code but not in the config file, we need to save the new options
                boolean fileNeedsUpdate = false;

                // Puts doubles with 0 as decimal as integers
                for (Map.Entry<String, Object> entry : config.entrySet())
                {
                    if (entry.getValue() instanceof Double)
                    {
                        // Integer values are stored as double in the gson file, so by doing this we can put them back
                        // to an int
                        if (((Double) entry.getValue()).intValue() == (Double) entry.getValue())
                        {
                            config.put(entry.getKey(), ((Double) entry.getValue()).intValue());
                        }
                    }
                }

                // Remove options present in the config file but not in the code
                ArrayList<String> toRemove = new ArrayList<>();
                for (String option : config.keySet())
                {
                    boolean exists = false;

                    for (Field field : cyanLibConfigClass.getClass().getDeclaredFields())
                    {
                        if (Modifier.isPublic(field.getModifiers()) && Modifier.isStatic(field.getModifiers())
                            && Modifier.isFinal(field.getModifiers()))
                        {
                            if (BooleanOption.class.isAssignableFrom(field.getType()))
                            {
                                try
                                {
                                    if (((BooleanOption) field.get(null)).getName().equals(option))
                                    {
                                        exists = true;
                                        break;
                                    }
                                }
                                catch (IllegalAccessException e)
                                {
                                    throw new RuntimeException(e);
                                }
                            }
                            else if (IntegerOption.class.isAssignableFrom(field.getType()))
                            {
                                try
                                {
                                    if (((IntegerOption) field.get(null)).getName().equals(option))
                                    {
                                        exists = true;
                                        break;
                                    }
                                }
                                catch (IllegalAccessException e)
                                {
                                    throw new RuntimeException(e);
                                }
                            }
                        }
                    }

                    if (!exists)
                    {
                        toRemove.add(option);
                    }
                }

                for (String option : toRemove)
                {
                    config.remove(option);
                }

                if (!toRemove.isEmpty())
                {
                    fileNeedsUpdate = true;
                    toRemove.clear();
                }

                // For each option found in the config file, update the value of the option object
                // If an option object is not present in the file, it is added to the options and the file is updated
                for (Field field : cyanLibConfigClass.getClass().getDeclaredFields())
                {
                    if (Modifier.isPublic(field.getModifiers()) && Modifier.isStatic(field.getModifiers())
                        && Modifier.isFinal(field.getModifiers()))
                    {
                        if (BooleanOption.class.isAssignableFrom(field.getType()))
                        {
                            try
                            {
                                BooleanOption booleanOption = (BooleanOption) field.get(null);

                                if (config.containsKey(booleanOption.getName()))
                                {
                                    boolean configFileValue = (Boolean) config.get(booleanOption.getName());
                                    // If the value in the config file is different from the default one, we change
                                    // its value in the class
                                    if (configFileValue != booleanOption.getValue())
                                    {
                                        booleanOption.setValue(configFileValue);
                                    }
                                }
                                else
                                {
                                    fileNeedsUpdate = true;
                                }
                                optionsList.add(booleanOption);
                            }
                            catch (IllegalAccessException e)
                            {
                                throw new RuntimeException(e);
                            }
                        }
                        else if (IntegerOption.class.isAssignableFrom(field.getType()))
                        {
                            try
                            {
                                IntegerOption integerOption = (IntegerOption) field.get(null);

                                if (config.containsKey(integerOption.getName()))
                                {
                                    int configFileValue = (Integer) config.get(integerOption.getName());
                                    // If the value in the config file is different from the default one, we change
                                    // its value in the class
                                    if (configFileValue != integerOption.getValue())
                                    {
                                        integerOption.setValue(configFileValue);
                                    }
                                }
                                else
                                {
                                    fileNeedsUpdate = true;
                                }
                                optionsList.add(integerOption);
                            }
                            catch (IllegalAccessException e)
                            {
                                throw new RuntimeException(e);
                            }
                        }
                    }
                }

                if (fileNeedsUpdate)
                {
                    writeConfig();
                }
            }
        }
    }

    public void writeConfig()
    {
        Map<String, Object> config = optionsList.stream()
                                                .collect(Collectors.toMap(IOption::getName, IOption::getValue));

        Path path = FabricLoader.getInstance().getConfigDir().resolve("%s.json".formatted(modid));
        if (!Files.exists(path))
        {
            try
            {
                Files.createFile(path);
            }
            catch (IOException e)
            {
                throw new RuntimeException(e);
            }
        }

        try
        {
            Gson gson = new GsonBuilder().setPrettyPrinting().create();
            Writer writer = Files.newBufferedWriter(path);
            gson.toJson(config, writer);
            writer.close();
        }
        catch (IOException e)
        {
            throw new RuntimeException(e);
        }
    }
}
