package dev.anvilcraft.ping.network;

import dev.anvilcraft.ping.MiddleKeyPing;
import dev.anvilcraft.ping.client.MiddleKeyPingClient;
import dev.anvilcraft.ping.util.ByteBufCodecUtils;
import dev.anvilcraft.ping.util.PingType;
import io.netty.buffer.ByteBuf;
import net.minecraft.ChatFormatting;
import net.minecraft.core.registries.Registries;
import net.minecraft.network.chat.Component;
import net.minecraft.network.codec.StreamCodec;
import net.minecraft.network.protocol.common.custom.CustomPacketPayload;
import net.minecraft.resources.ResourceKey;
import net.minecraft.server.MinecraftServer;
import net.minecraft.server.level.ServerPlayer;
import net.minecraft.world.level.Level;
import net.minecraft.world.phys.Vec3;
import net.minecraft.world.scores.PlayerTeam;
import net.neoforged.neoforge.network.handling.IPayloadContext;

import java.util.List;

public record PingPayload(ResourceKey<Level> dimension, Vec3 position, PingType pingType) implements CustomPacketPayload {
    public static final Type<PingPayload> TYPE = new Type<>(MiddleKeyPing.of("inspection_state"));

    public static final StreamCodec<ByteBuf, PingPayload> STREAM_CODEC = StreamCodec.composite(
        ResourceKey.streamCodec(Registries.DIMENSION),
        PingPayload::dimension,
        ByteBufCodecUtils.VEC3,
        PingPayload::position,
        ByteBufCodecUtils.enumCodec(PingType.class),
        PingPayload::pingType,
        PingPayload::new
    );

    @Override
    public Type<? extends CustomPacketPayload> type() {
        return PingPayload.TYPE;
    }

    public void clientHandle(IPayloadContext context) {
        context.enqueueWork(() -> MiddleKeyPingClient.handle(this));
    }

    @SuppressWarnings("resource")
    public void serverHandle(IPayloadContext context) {
        final ServerPlayer player = (ServerPlayer) context.player();
        context.enqueueWork(() -> {
            MinecraftServer server = player.getServer();
            if (server == null) return;
            List<ServerPlayer> players = server.getPlayerList().getPlayers();
            for (ServerPlayer other : players) {
                PlayerTeam team = player.getTeam();
                PlayerTeam team1 = other.getTeam();
                if (team != null && !team.equals(team1)) return;
                ResourceKey<Level> dimension = player.level().dimension();
                ResourceKey<Level> dimension1 = other.level().dimension();
                if (!dimension.equals(dimension1)) return;
                other.connection.send(new PingPayload(this.dimension, this.position, this.pingType));
                other.sendSystemMessage(Component.translatable(
                    this.pingType.description(),
                    Component.translatable("chat.middle_key_ping.player", player.getDisplayName()).withStyle(ChatFormatting.WHITE)
                ).withColor(this.pingType.color()));
            }
        });
    }
}
