/*
 * Decompiled with CFR 0.152.
 */
package dan200.computercraft.core.filesystem;

import com.google.common.collect.Sets;
import dan200.computercraft.ComputerCraft;
import dan200.computercraft.api.filesystem.FileOperationException;
import dan200.computercraft.api.filesystem.IWritableMount;
import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.FileChannel;
import java.nio.channels.NonReadableChannelException;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.SeekableByteChannel;
import java.nio.channels.WritableByteChannel;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.util.Collections;
import java.util.List;
import java.util.OptionalLong;
import java.util.Set;
import javax.annotation.Nonnull;

public class FileMount
implements IWritableMount {
    private static final int MINIMUM_FILE_SIZE = 500;
    private static final Set<OpenOption> READ_OPTIONS = Collections.singleton(StandardOpenOption.READ);
    private static final Set<OpenOption> WRITE_OPTIONS = Sets.newHashSet(StandardOpenOption.WRITE, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING);
    private static final Set<OpenOption> APPEND_OPTIONS = Sets.newHashSet(StandardOpenOption.WRITE, StandardOpenOption.CREATE, StandardOpenOption.APPEND);
    private final File rootPath;
    private final long capacity;
    private long usedSpace;

    public FileMount(File rootPath, long capacity) {
        this.rootPath = rootPath;
        this.capacity = capacity + 500L;
        this.usedSpace = this.created() ? FileMount.measureUsedSpace(this.rootPath) : 500L;
    }

    private static long measureUsedSpace(File file) {
        if (!file.exists()) {
            return 0L;
        }
        try {
            Visitor visitor = new Visitor();
            Files.walkFileTree(file.toPath(), visitor);
            return visitor.size;
        }
        catch (IOException e) {
            ComputerCraft.log.error("Error computing file size for {}", (Object)file, (Object)e);
            return 0L;
        }
    }

    @Override
    public boolean exists(@Nonnull String path) {
        if (!this.created()) {
            return path.isEmpty();
        }
        File file = this.getRealPath(path);
        return file.exists();
    }

    @Override
    public boolean isDirectory(@Nonnull String path) {
        if (!this.created()) {
            return path.isEmpty();
        }
        File file = this.getRealPath(path);
        return file.exists() && file.isDirectory();
    }

    @Override
    public void list(@Nonnull String path, @Nonnull List<String> contents) throws IOException {
        String[] paths;
        if (!this.created()) {
            if (!path.isEmpty()) {
                throw new FileOperationException(path, "Not a directory");
            }
            return;
        }
        File file = this.getRealPath(path);
        if (!file.exists() || !file.isDirectory()) {
            throw new FileOperationException(path, "Not a directory");
        }
        for (String subPath : paths = file.list()) {
            if (!new File(file, subPath).exists()) continue;
            contents.add(subPath);
        }
    }

    @Override
    public long getSize(@Nonnull String path) throws IOException {
        if (!this.created()) {
            if (path.isEmpty()) {
                return 0L;
            }
        } else {
            File file = this.getRealPath(path);
            if (file.exists()) {
                return file.isDirectory() ? 0L : file.length();
            }
        }
        throw new FileOperationException(path, "No such file");
    }

    @Override
    @Nonnull
    public ReadableByteChannel openForRead(@Nonnull String path) throws IOException {
        File file;
        if (this.created() && (file = this.getRealPath(path)).exists() && !file.isDirectory()) {
            return FileChannel.open(file.toPath(), READ_OPTIONS, new FileAttribute[0]);
        }
        throw new FileOperationException(path, "No such file");
    }

    @Override
    @Nonnull
    public BasicFileAttributes getAttributes(@Nonnull String path) throws IOException {
        File file;
        if (this.created() && (file = this.getRealPath(path)).exists()) {
            return Files.readAttributes(file.toPath(), BasicFileAttributes.class, new LinkOption[0]);
        }
        throw new FileOperationException(path, "No such file");
    }

    @Override
    public void makeDirectory(@Nonnull String path) throws IOException {
        this.create();
        File file = this.getRealPath(path);
        if (file.exists()) {
            if (!file.isDirectory()) {
                throw new FileOperationException(path, "File exists");
            }
            return;
        }
        int dirsToCreate = 1;
        File parent = file.getParentFile();
        while (!parent.exists()) {
            ++dirsToCreate;
            parent = parent.getParentFile();
        }
        if (this.getRemainingSpace() < (long)dirsToCreate * 500L) {
            throw new FileOperationException(path, "Out of space");
        }
        if (file.mkdirs()) {
            this.usedSpace += (long)dirsToCreate * 500L;
        } else {
            throw new FileOperationException(path, "Access denied");
        }
    }

    @Override
    public void delete(@Nonnull String path) throws IOException {
        File file;
        if (path.isEmpty()) {
            throw new FileOperationException(path, "Access denied");
        }
        if (this.created() && (file = this.getRealPath(path)).exists()) {
            this.deleteRecursively(file);
        }
    }

    private void deleteRecursively(File file) throws IOException {
        if (file.isDirectory()) {
            String[] children;
            for (String aChildren : children = file.list()) {
                this.deleteRecursively(new File(file, aChildren));
            }
        }
        long fileSize = file.isDirectory() ? 0L : file.length();
        boolean success = file.delete();
        if (success) {
            this.usedSpace -= Math.max(500L, fileSize);
        } else {
            throw new IOException("Access denied");
        }
    }

    @Override
    @Nonnull
    public WritableByteChannel openForWrite(@Nonnull String path) throws IOException {
        this.create();
        File file = this.getRealPath(path);
        if (file.exists() && file.isDirectory()) {
            throw new FileOperationException(path, "Cannot write to directory");
        }
        if (file.exists()) {
            this.usedSpace -= Math.max(file.length(), 500L);
        } else if (this.getRemainingSpace() < 500L) {
            throw new FileOperationException(path, "Out of space");
        }
        this.usedSpace += 500L;
        return new SeekableCountingChannel(Files.newByteChannel(file.toPath(), WRITE_OPTIONS, new FileAttribute[0]), 500L);
    }

    @Override
    @Nonnull
    public WritableByteChannel openForAppend(@Nonnull String path) throws IOException {
        if (!this.created()) {
            throw new FileOperationException(path, "No such file");
        }
        File file = this.getRealPath(path);
        if (!file.exists()) {
            throw new FileOperationException(path, "No such file");
        }
        if (file.isDirectory()) {
            throw new FileOperationException(path, "Cannot write to directory");
        }
        return new WritableCountingChannel(Files.newByteChannel(file.toPath(), APPEND_OPTIONS, new FileAttribute[0]), Math.max(500L - file.length(), 0L));
    }

    @Override
    public long getRemainingSpace() {
        return Math.max(this.capacity - this.usedSpace, 0L);
    }

    @Override
    @Nonnull
    public OptionalLong getCapacity() {
        return OptionalLong.of(this.capacity - 500L);
    }

    private File getRealPath(String path) {
        return new File(this.rootPath, path);
    }

    private boolean created() {
        return this.rootPath.exists();
    }

    private void create() throws IOException {
        boolean success;
        if (!this.rootPath.exists() && !(success = this.rootPath.mkdirs())) {
            throw new IOException("Access denied");
        }
    }

    private static class Visitor
    extends SimpleFileVisitor<Path> {
        long size;

        private Visitor() {
        }

        @Override
        public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) {
            this.size += 500L;
            return FileVisitResult.CONTINUE;
        }

        @Override
        public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) {
            this.size += Math.max(attrs.size(), 500L);
            return FileVisitResult.CONTINUE;
        }

        @Override
        public FileVisitResult visitFileFailed(Path file, IOException exc) {
            ComputerCraft.log.error("Error computing file size for {}", (Object)file, (Object)exc);
            return FileVisitResult.CONTINUE;
        }
    }

    private class SeekableCountingChannel
    extends WritableCountingChannel
    implements SeekableByteChannel {
        private final SeekableByteChannel inner;

        SeekableCountingChannel(SeekableByteChannel inner, long bytesToIgnore) {
            super(inner, bytesToIgnore);
            this.inner = inner;
        }

        @Override
        public SeekableByteChannel position(long newPosition) throws IOException {
            if (!this.isOpen()) {
                throw new ClosedChannelException();
            }
            if (newPosition < 0L) {
                throw new IllegalArgumentException("Cannot seek before the beginning of the stream");
            }
            long delta = newPosition - this.inner.position();
            if (delta < 0L) {
                this.ignoredBytesLeft -= delta;
            } else {
                this.count(delta);
            }
            return this.inner.position(newPosition);
        }

        @Override
        public SeekableByteChannel truncate(long size) throws IOException {
            throw new IOException("Not yet implemented");
        }

        @Override
        public int read(ByteBuffer dst) throws ClosedChannelException {
            if (!this.inner.isOpen()) {
                throw new ClosedChannelException();
            }
            throw new NonReadableChannelException();
        }

        @Override
        public long position() throws IOException {
            return this.inner.position();
        }

        @Override
        public long size() throws IOException {
            return this.inner.size();
        }
    }

    private class WritableCountingChannel
    implements WritableByteChannel {
        private final WritableByteChannel inner;
        long ignoredBytesLeft;

        WritableCountingChannel(WritableByteChannel inner, long bytesToIgnore) {
            this.inner = inner;
            this.ignoredBytesLeft = bytesToIgnore;
        }

        @Override
        public int write(@Nonnull ByteBuffer b) throws IOException {
            this.count(b.remaining());
            return this.inner.write(b);
        }

        void count(long n) throws IOException {
            this.ignoredBytesLeft -= n;
            if (this.ignoredBytesLeft < 0L) {
                long newBytes = -this.ignoredBytesLeft;
                this.ignoredBytesLeft = 0L;
                long bytesLeft = FileMount.this.capacity - FileMount.this.usedSpace;
                if (newBytes > bytesLeft) {
                    throw new IOException("Out of space");
                }
                FileMount.this.usedSpace += newBytes;
            }
        }

        @Override
        public boolean isOpen() {
            return this.inner.isOpen();
        }

        @Override
        public void close() throws IOException {
            this.inner.close();
        }
    }
}

