#version 120

uniform sampler2D colortex0;
uniform sampler2D colortex1; // Lightmap ve normal bilgileri için
uniform sampler2D depthtex0; // Derinlik bilgisi için gölge
uniform float frameTimeCounter;
uniform float viewWidth;
uniform float viewHeight;
uniform vec3 sunPosition;
uniform vec3 shadowLightPosition;
uniform int worldTime;
uniform float rainStrength;


varying vec2 texcoord;

// Basit noise fonksiyonu
float noise(vec2 co) {
    return fract(sin(dot(co.xy, vec2(12.9898, 78.233))) * 43758.5453);
}

// Hash fonksiyonu - daha iyi random için
float hash(float n) {
    return fract(sin(n) * 43758.5453);
}

// 2D hash fonksiyonu
vec2 hash2(vec2 p) {
    p = vec2(dot(p, vec2(127.1, 311.7)), dot(p, vec2(269.5, 183.3)));
    return -1.0 + 2.0 * fract(sin(p) * 43758.5453123);
}

// Gelişmiş VHS bozulması
vec2 vhsDistortion(vec2 uv, float time) {
    // Ana dalga bozulması
    float wave1 = sin(uv.y * 8.0 + time * 2.0) * 0.001;
    float wave2 = sin(uv.y * 15.0 + time * 3.5) * 0.001;
    
    // Rastgele glitch çizgileri
    float glitchLine = floor(uv.y * 200.0);
    float glitchRandom = hash(glitchLine + floor(time * 5.0));
    float glitchStrength = step(0.98, glitchRandom) * 0.02;
    
    // Horizontal shift
    float horizontalShift = glitchStrength * (hash(glitchLine * 13.7 + time) - 0.5);
    
    return uv + vec2(wave1 + wave2 + horizontalShift, 0.0);
}

// RGB kaydırması (chromatic aberration)
vec3 chromaShift(sampler2D tex, vec2 uv, float time) {
    float shift = 0.002 + sin(time * 0.5) * 0.001;
    
    vec2 redUV = uv + vec2(shift, 0.0);
    vec2 greenUV = uv;
    vec2 blueUV = uv - vec2(shift, 0.0);
    
    float r = texture2D(tex, redUV).r;
    float g = texture2D(tex, greenUV).g;
    float b = texture2D(tex, blueUV).b;
    
    return vec3(r, g, b);
}

// Tarama çizgileri
float scanlines(vec2 uv) {
    float line = sin(uv.y * viewHeight * 0.7) * 0.08;
    return 1.0 - abs(line);
}

// Film tanecikleri (gelişmiş)
float filmGrain(vec2 uv, float time) {
    vec2 grainUV = uv * 300.0 + time * 50.0;
    return (noise(grainUV) - 0.5) * 0.25;
}

// Tracking çizgileri
float trackingLines(vec2 uv, float time) {
    float tracking = 0.0;
    float speed = time * 0.1;
    
    // Ana tracking çizgisi
    float trackY = fract(speed) * 2.0 - 1.0;
    float trackDist = abs(uv.y - trackY);
    tracking += smoothstep(0.01, 0.0, trackDist) * 0.3;
    
    // İkincil çizgiler
    float track2Y = fract(speed * 0.7 + 0.3) * 2.0 - 1.0;
    float track2Dist = abs(uv.y - track2Y);
    tracking += smoothstep(0.005, 0.0, track2Dist) * 0.3;
    
    return tracking;
}

// VHS renk degradasyonu
vec3 vhsColor(vec3 color, float time) {
    // Hafif desaturasyon
    float gray = dot(color, vec3(0.299, 0.587, 0.114));
    color = mix(color, vec3(gray), 0.3);
    
    // VHS'e özgü renk kayması
    color.r *= 1.1;
    color.g *= 0.95;
    color.b *= 0.9;
    
    // Kontrast ve parlaklık ayarı
    color = (color - 0.5) * 1.3 + 0.6;
    
    // Hafif sarımsı ton
    color += vec3(0.05, 0.03, 0.0);
    
    return color;
}

// Glitch blokları
vec3 glitchBlocks(vec3 color, vec2 uv, float time) {
    vec2 blockUV = floor(uv * vec2(20.0, 15.0));
    float blockRandom = hash(blockUV.x + blockUV.y * 13.7 + floor(time * 3.0));
    
    if (blockRandom > 0.95) {
        // Renk inversiyonu
        //color = 1.0 - color;
    } else if (blockRandom > 0.92) {
        // Renk kayması
        //color = color.brg;
    } else if (blockRandom > 0.90) {
        // Parlaklık artışı
        //color *= 1.5;
    }
    
    return color;
}

// VHS UI öğeleri çizme fonksiyonu
float drawRect(vec2 uv, vec2 pos, vec2 size) {
    vec2 d = abs(uv - pos) - size;
    return 1.0 - smoothstep(0.0, 0.002, max(d.x, d.y));
}

// Basit karakter çizimi (sadece dikdörtgenler kullanarak)
float drawChar(vec2 uv, vec2 pos, int charCode) {
    vec2 localUV = (uv - pos) / vec2(0.015, 0.025);
    
    if (localUV.x < 0.0 || localUV.x > 1.0 || localUV.y < 0.0 || localUV.y > 1.0) {
        return 0.0;
    }
    
    // Basit 5x7 bitmap font yaklaşımı
    float pixel = 0.0;
    vec2 pixelPos = floor(localUV * vec2(5.0, 7.0));
    
    // Sayılar için basit pattern'ler (güncellenmiş versiyonunuz)
    if (charCode == 0) { // '0'
        if ((pixelPos.x == 0.0 || pixelPos.x == 4.0) && pixelPos.y > 0.0 && pixelPos.y < 6.0) pixel = 1.0;
        if ((pixelPos.y == 0.0 || pixelPos.y == 6.0) && pixelPos.x > 0.0 && pixelPos.x < 4.0) pixel = 1.0;
    } else if (charCode == 1) { // '1'
        if (pixelPos.x == 3.0) pixel = 1.0;
        if (pixelPos.x == 2.0 && pixelPos.y < 1.0) pixel = 1.0;
        if (pixelPos.x == 3.0 && pixelPos.y < 1.0) pixel = 1.0;
        if (pixelPos.x == 4.0 && pixelPos.y < 1.0) pixel = 1.0;
        if (pixelPos.y == 5.0 && pixelPos.x >= 2.0 && pixelPos.x <= 3.0) pixel = 1.0;
    } else if (charCode == 2) { // '2'
        if (pixelPos.y == 0.0 || pixelPos.y == 3.0 || pixelPos.y == 6.0) pixel = 1.0;
        if (pixelPos.x == 0.0 && pixelPos.y < 3.0) pixel = 1.0;
        if (pixelPos.x == 4.0 && pixelPos.y > 3.0) pixel = 1.0;
    } else if (charCode == 3) { // '3'
        if (pixelPos.y == 0.0 || pixelPos.y == 3.0 || pixelPos.y == 6.0) pixel = 1.0;
        if (pixelPos.x == 4.0 && pixelPos.y != 3.0) pixel = 1.0;
    } else if (charCode == 4) { // '4'
        if (pixelPos.x == 0.0 && pixelPos.y > 3.0) pixel = 1.0;
        if (pixelPos.x == 4.0) pixel = 1.0;
        if (pixelPos.y == 3.0) pixel = 1.0;
    } else if (charCode == 5) { // '5'
        if (pixelPos.y == 0.0 || pixelPos.y == 3.0 || pixelPos.y == 6.0) pixel = 1.0;
        if (pixelPos.x == 0.0 && pixelPos.y > 3.0) pixel = 1.0;
        if (pixelPos.x == 4.0 && pixelPos.y < 3.0) pixel = 1.0;
    } else if (charCode == 6) { // '6'
        if (pixelPos.y == 0.0 || pixelPos.y == 3.0 || pixelPos.y == 6.0) pixel = 1.0;
        if (pixelPos.x == 0.0) pixel = 1.0;
        if (pixelPos.x == 4.0 && pixelPos.y < 4.0) pixel = 1.0;
    } else if (charCode == 7) { // '7'
        if (pixelPos.y == 6.0) pixel = 1.0;
        if (pixelPos.x == 4.0) pixel = 1.0;
    } else if (charCode == 8) { // '8'
        if (pixelPos.y == 0.0 || pixelPos.y == 3.0 || pixelPos.y == 6.0) pixel = 1.0;
        if (pixelPos.x == 0.0 || pixelPos.x == 4.0) pixel = 1.0;
    } else if (charCode == 9) { // '9'
        if (pixelPos.y == 0.0 || pixelPos.y == 3.0 || pixelPos.y == 6.0) pixel = 1.0;
        if (pixelPos.x == 0.0 && pixelPos.y > 3.0) pixel = 1.0;
        if (pixelPos.x == 4.0) pixel = 1.0;
    } else if (charCode == 10) { // ':'
        if (pixelPos.x == 2.0 && (pixelPos.y == 2.0 || pixelPos.y == 4.0)) pixel = 1.0;
    } else if (charCode == 11) { // 'R'
        if (pixelPos.x == 0.0) pixel = 1.0;
        if (pixelPos.y == 6.0 || pixelPos.y == 3.0) pixel = 1.0;
        if (pixelPos.x == 2.0 && (pixelPos.y > 5.0 || pixelPos.y == 2.0)) pixel = 1.0;
        if (pixelPos.x == 3.0 && (pixelPos.y > 5.0 || pixelPos.y == 1.0)) pixel = 1.0;
        if (pixelPos.x == 4.0 && (pixelPos.y > 3.0 || pixelPos.y == 0.0)) pixel = 1.0;
    } else if (charCode == 12) { // 'E'
        if (pixelPos.x == 0.0) pixel = 1.0;
        if (pixelPos.y == 0.0 || pixelPos.y == 3.0 || pixelPos.y == 6.0) pixel = 1.0;
    } else if (charCode == 13) { // 'C'
        if (pixelPos.x == 0.0 && pixelPos.y > 0.0 && pixelPos.y < 6.0) pixel = 1.0;
        if ((pixelPos.y == 0.0 || pixelPos.y == 6.0) && pixelPos.x > 0.0) pixel = 1.0;
    } else if (charCode == 14) { // '-'
        if (pixelPos.x == 0.0 && pixelPos.y > 0.0 && pixelPos.y < 1.0) pixel = 1.0;
        if ((pixelPos.y == 1.0 || pixelPos.y == 0.0) && pixelPos.x > 2.0) pixel = 1.0;
    }
    
    return pixel;
}


// UI çizimi
vec3 drawVHSUI(vec3 color, vec2 uv, float time) {
    vec3 uiColor = color;
    float ui = 0.0;
    
    // Letterbox alanı tanımları
    float letterboxHeight = 0.15;
    float activeAreaStart = letterboxHeight;
    float activeAreaEnd = 1.0 - letterboxHeight;
    float activeAreaHeight = activeAreaEnd - activeAreaStart;
    
    // REC indicator (sol üst - üst letterbox'ta)
    if (uv.x < 0.55 && uv.y > 0.85 && uv.y < 1.0) {
        // REC yazısı - üst letterbox alanında
        ui += drawChar(uv, vec2(0.17, 0.92), 11); // R
        ui += drawChar(uv, vec2(0.19, 0.92), 12); // E
        ui += drawChar(uv, vec2(0.21, 0.92), 13); // C
        
        // Yanıp sönen kırmızı nokta
        float blink = step(0.5, fract(time * 0.5));
        if (uv.x > 0.235 && uv.x < 0.245 && uv.y > 0.915 && uv.y < 0.925) {
            uiColor = mix(uiColor, vec3(1.0, 0.0, 0.0), blink);
        }
    }
    
    // Zaman göstergesi (sağ üst - üst letterbox'ta)
    if (uv.x > 0.6 && uv.y > 0.85 && uv.y < 1.0) {
        // Başlangıç saati 00:15:00 olarak varsayalım ve gerçek zamanlı ilerlesin
        float totalSeconds = time + 01.0 * 3600.0 + 15.0 * 60.0;
        
        int hours = int(mod(totalSeconds / 3600.0, 24.0));
        int minutes = int(mod(totalSeconds / 60.0, 60.0));
        int seconds = int(mod(totalSeconds, 60.0));
        
        // Saat - onlar basamağı
        int hourTens = hours / 10;
        int hourOnes = hours - hourTens * 10;
        ui += drawChar(uv, vec2(0.73, 0.92), hourTens);
        ui += drawChar(uv, vec2(0.75, 0.92), hourOnes);
        ui += drawChar(uv, vec2(0.77, 0.92), 10); // :
        
        // Dakika
        int minuteTens = minutes / 10;
        int minuteOnes = minutes - minuteTens * 10;
        ui += drawChar(uv, vec2(0.79, 0.92), minuteTens);
        ui += drawChar(uv, vec2(0.81, 0.92), minuteOnes);
        //ui += drawChar(uv, vec2(0.95, 0.92), 10); // :
        
        // Saniye
        int secondTens = seconds / 10;
        int secondOnes = seconds - secondTens * 10;
        //ui += drawChar(uv, vec2(0.93, 0.92), secondTens);
        //ui += drawChar(uv, vec2(0.95, 0.92), secondOnes);
    }
    
    // Tarih (sol alt - alt letterbox'ta)
    if (uv.x < 0.55 && uv.y > 0.0 && uv.y < letterboxHeight) {
        // Başlangıç tarihi: 25 Aralık 2024 ve gün bazında ilerle
        float daysSinceStart = floor(time / 86400.0);
        
        // Basit tarih hesabı
        int currentDay = int(mod(25.0 + daysSinceStart, 31.0));
        if (currentDay == 0) currentDay = 31;
        
        // Tarih formatı: DD/MM/YYYY
        int dayTens = currentDay / 10;
        int dayOnes = currentDay - dayTens * 10;
        ui += drawChar(uv, vec2(0.17, 0.08), dayTens);
        ui += drawChar(uv, vec2(0.19, 0.08), dayOnes);
        ui += drawChar(uv, vec2(0.21, 0.08), 14); // /
        
        ui += drawChar(uv, vec2(0.23, 0.08), 1); // 1 (Aralık = 12)
        ui += drawChar(uv, vec2(0.25, 0.08), 2); // 2
        ui += drawChar(uv, vec2(0.27, 0.08), 14); // /
        
        ui += drawChar(uv, vec2(0.29, 0.08), 1); // 1
        ui += drawChar(uv, vec2(0.31, 0.08), 9); // 9
        ui += drawChar(uv, vec2(0.33, 0.08), 9); // 9
        ui += drawChar(uv, vec2(0.35, 0.08), 8); // 8
    }
    
    // Kayıt süresi (sağ alt - alt letterbox'ta)
    if (uv.x > 0.6 && uv.y > 0.0 && uv.y < letterboxHeight) {
        // Kayıt süresini saniye cinsinden hesapla
        float recordingTime = time;
        int recordHours = int(recordingTime / 3600.0);
        int recordMinutes = int(mod(recordingTime / 60.0, 60.0));
        int recordSeconds = int(mod(recordingTime, 60.0));
        
        // Saat formatı: HH:MM:SS
        int hourTens = recordHours / 10;
        int hourOnes = recordHours - hourTens * 10;
        ui += drawChar(uv, vec2(0.67, 0.08), hourTens);
        ui += drawChar(uv, vec2(0.69, 0.08), hourOnes);
        ui += drawChar(uv, vec2(0.71, 0.08), 10); // :
        
        int minuteTens = recordMinutes / 10;
        int minuteOnes = recordMinutes - minuteTens * 10;
        ui += drawChar(uv, vec2(0.73, 0.08), minuteTens);
        ui += drawChar(uv, vec2(0.75, 0.08), minuteOnes);
        ui += drawChar(uv, vec2(0.77, 0.08), 10); // :
        
        int secondTens = recordSeconds / 10;
        int secondOnes = recordSeconds - secondTens * 10;
        ui += drawChar(uv, vec2(0.79, 0.08), secondTens);
        ui += drawChar(uv, vec2(0.81, 0.08), secondOnes);
    }
    
    // UI overlay
    if (ui > 0.0) {
        uiColor = mix(uiColor, vec3(1.0, 1.0, 1.0), ui * 0.8);
    }
    
    return uiColor;
}

void main() {
    float time = frameTimeCounter;
    vec2 uv = texcoord;
    
    // Aspect ratio hesaplama
    float aspectRatio = viewWidth / viewHeight;
    
    // Letterbox efekti için kare alan belirleme
    float letterboxHeight = 0.15;
    
    // Kare alan sınırları
    bool inLetterbox = (uv.x < letterboxHeight || uv.x > (1.0 - letterboxHeight));
    
    // Ana görüntü alanı için UV koordinatları
    vec2 mainUV = uv;
    if (!inLetterbox) {
        // Ana görüntüyü kare alana sığdır
        mainUV.x = (uv.x - letterboxHeight) / (1.0 - 2.0 * letterboxHeight);
    }
    
    vec3 color = vec3(0.0);
    
    if (inLetterbox) {
        // Letterbox alanı - siyah arka plan
        color = vec3(0.0, 0.0, 0.0);
    } else {
        // Ana görüntü alanı
        
        // VHS bozulması uygula
        vec2 distortedUV = vhsDistortion(mainUV, time);
        
        // Chromatic aberration ile rengi al
        color = chromaShift(colortex0, distortedUV, time);
        
        // Güvenlik kontrolü
        if (length(color) < 0.05) {
            color = texture2D(colortex0, mainUV).rgb;
        }
        
        // Glitch blokları ekle
        color = glitchBlocks(color, mainUV, time);
        
        // Tarama çizgileri
        color *= scanlines(mainUV);
        
        // Film tanecikleri
        color += filmGrain(mainUV, time);
        
        // Tracking çizgileri
        color += trackingLines(mainUV, time);
        
        // VHS renk stili
        color = vhsColor(color, time);
        
        // Vignette (sadece ana görüntü alanında)
        vec2 center = mainUV - 0.5;
        float vignette = 1.0 - dot(center, center) * 0.3;
        color *= vignette;
        
        // Renkleri sınırla
        color = clamp(color, 0.0, 1.0);
    }
    
    // VHS UI elementleri (tüm ekran için, letterbox dahil)
    color = drawVHSUI(color, uv, time);
    
    gl_FragColor = vec4(color, 1.0);
}