/*
 * Decompiled with CFR 0.152.
 */
package eclipse.euphoriacompanion.report;

import eclipse.euphoriacompanion.analyzer.ShaderAnalyzer;
import eclipse.euphoriacompanion.report.AnalysisReport;
import java.io.BufferedWriter;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

public class ReportGenerator {
    private static final DateTimeFormatter TIMESTAMP_FORMAT = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");

    public static void generateReport(AnalysisReport report, Path outputPath) throws IOException {
        if (outputPath.getParent() == null) {
            throw new IOException("Output path has no parent directory: " + String.valueOf(outputPath));
        }
        Files.createDirectories(outputPath.getParent(), new FileAttribute[0]);
        Path tempPath = outputPath.getParent().resolve(String.valueOf(outputPath.getFileName()) + ".tmp");
        try (BufferedWriter writer = Files.newBufferedWriter(tempPath, new OpenOption[0]);){
            ReportGenerator.writeHeader(writer, report);
            ReportGenerator.writeMissingBlocks(writer, report);
            ReportGenerator.writeTagCoverage(writer, report);
            ReportGenerator.writeUnusedTags(writer, report);
            ReportGenerator.writeIncompleteBlockStates(writer, report);
            ReportGenerator.writeDuplicateDefinitions(writer, report);
            ReportGenerator.writeRenderLayerMismatches(writer, report);
        }
        Files.move(tempPath, outputPath, StandardCopyOption.REPLACE_EXISTING, StandardCopyOption.ATOMIC_MOVE);
    }

    private static void writeHeader(BufferedWriter writer, AnalysisReport report) throws IOException {
        writer.write("=== SHADER ANALYSIS: " + report.getShaderpackName() + " ===\n");
        writer.write("Generated: " + LocalDateTime.now().format(TIMESTAMP_FORMAT) + "\n\n");
        int totalInGame = report.getTotalBlocksInGame();
        int totalInShader = report.getTotalBlocksInShader();
        int totalMissing = report.getTotalMissingBlocks();
        writer.write("STATISTICS:\n");
        writer.write("  Total blocks in game: " + totalInGame + "\n");
        writer.write("  Blocks defined in shader: " + totalInShader + "\n");
        writer.write("  Missing blocks: " + totalMissing + "\n");
        double coverage = totalInGame > 0 ? 100.0 * (1.0 - (double)totalMissing / (double)totalInGame) : 0.0;
        writer.write(String.format("  Coverage: %.2f%%\n\n", coverage));
    }

    private static void writeMissingBlocks(BufferedWriter writer, AnalysisReport report) throws IOException {
        writer.write("----------------------------------------\n");
        writer.write("MISSING BLOCKS BY MOD:\n\n");
        Map<String, Map<String, List<String>>> missingByMod = report.getMissingBlocksByMod();
        if (missingByMod.isEmpty()) {
            writer.write("No missing blocks found.\n\n");
            return;
        }
        for (Map.Entry<String, Map<String, List<String>>> modEntry : missingByMod.entrySet()) {
            String modName = modEntry.getKey();
            Map<String, List<String>> categories = modEntry.getValue();
            int totalBlocks = categories.values().stream().mapToInt(List::size).sum();
            writer.write(modName + " (" + totalBlocks + " blocks):\n");
            for (Map.Entry<String, List<String>> categoryEntry : categories.entrySet()) {
                String category = categoryEntry.getKey();
                List<String> blocks = categoryEntry.getValue();
                writer.write("  " + category + " (" + blocks.size() + "):\n");
                Collections.sort(blocks);
                for (String block : blocks) {
                    writer.write(" " + block + "\n");
                }
                writer.write("\n");
            }
        }
    }

    private static void writeTagCoverage(BufferedWriter writer, AnalysisReport report) throws IOException {
        writer.write("----------------------------------------\n");
        writer.write("COVERED BY TAGS:\n\n");
        Map<String, Set<String>> tagCoverage = report.getTagCoverage();
        Map<String, String> tagDefinitions = report.getTagDefinitions();
        Map<String, Integer> tagToProperty = report.getTagToProperty();
        if (tagCoverage.isEmpty()) {
            if (report.isTagSupportEnabled()) {
                writer.write("No tag coverage.\n\n");
            } else {
                writer.write("Tag support is disabled.\n\n");
            }
            return;
        }
        for (Map.Entry<String, Set<String>> entry : tagCoverage.entrySet()) {
            String tagIdentifier = entry.getKey();
            Set<String> blocks = entry.getValue();
            String tagValue = tagDefinitions.getOrDefault(tagIdentifier, "unknown");
            Integer propertyId = tagToProperty.get(tagIdentifier);
            Object propertyStr = propertyId != null ? "block." + propertyId : "unused";
            writer.write("Tag: " + tagIdentifier + " = " + tagValue + " (" + (String)propertyStr + ") - " + blocks.size() + " blocks\n");
            ArrayList<String> sortedBlocks = new ArrayList<String>(blocks);
            Collections.sort(sortedBlocks);
            for (String block : sortedBlocks) {
                writer.write(block + "\n");
            }
            writer.write("\n");
        }
    }

    private static void writeUnusedTags(BufferedWriter writer, AnalysisReport report) throws IOException {
        Map<String, String> tagDefinitions = report.getTagDefinitions();
        Map<String, Integer> tagToProperty = report.getTagToProperty();
        ArrayList<String> unusedTags = new ArrayList<String>();
        for (String tagIdentifier : tagDefinitions.keySet()) {
            if (tagToProperty.containsKey(tagIdentifier)) continue;
            unusedTags.add(tagIdentifier);
        }
        if (unusedTags.isEmpty()) {
            return;
        }
        writer.write("----------------------------------------\n");
        writer.write("UNUSED TAG DEFINITIONS:\n\n");
        writer.write("WARNING: The following tags are defined but not assigned to any block.XX property.\n");
        writer.write("These tags will not affect shader behavior.\n\n");
        Collections.sort(unusedTags);
        for (String tagIdentifier : unusedTags) {
            String tagValue = tagDefinitions.get(tagIdentifier);
            writer.write("  " + tagIdentifier + " = " + tagValue + "\n");
        }
        writer.write("\n");
    }

    private static void writeIncompleteBlockStates(BufferedWriter writer, AnalysisReport report) throws IOException {
        Map<String, Map<String, List<String>>> incompleteBlockStates = report.getIncompleteBlockStates();
        writer.write("----------------------------------------\n");
        writer.write("INCOMPLETE BLOCKSTATE DEFINITIONS:\n\n");
        if (incompleteBlockStates.isEmpty()) {
            writer.write("All blockstate definitions are complete.\n\n");
            return;
        }
        ArrayList<Map.Entry<String, Map<String, List<String>>>> sortedBlocks = new ArrayList<Map.Entry<String, Map<String, List<String>>>>(incompleteBlockStates.entrySet());
        sortedBlocks.sort(Map.Entry.comparingByKey());
        for (Map.Entry entry : sortedBlocks) {
            String blockId = (String)entry.getKey();
            Map missingByProperty = (Map)entry.getValue();
            writer.write(blockId + ":\n");
            for (Map.Entry propEntry : missingByProperty.entrySet()) {
                String propertyName = (String)propEntry.getKey();
                List missingValues = (List)propEntry.getValue();
                writer.write("  " + propertyName + " - Missing values: " + String.join((CharSequence)", ", missingValues) + "\n");
            }
            writer.write("\n");
        }
    }

    private static void writeDuplicateDefinitions(BufferedWriter writer, AnalysisReport report) throws IOException {
        Map<String, List<Integer>> duplicates = report.getDuplicateDefinitions();
        writer.write("----------------------------------------\n");
        writer.write("DUPLICATE DEFINITIONS:\n\n");
        if (duplicates.isEmpty()) {
            writer.write("No duplicate definitions found.\n\n");
            return;
        }
        ArrayList<Map.Entry<String, List<Integer>>> sortedDuplicates = new ArrayList<Map.Entry<String, List<Integer>>>(duplicates.entrySet());
        sortedDuplicates.sort(Map.Entry.comparingByKey());
        for (Map.Entry entry : sortedDuplicates) {
            String blockState = (String)entry.getKey();
            List propertyIds = (List)entry.getValue();
            Collections.sort(propertyIds);
            String propertyIdsStr = propertyIds.stream().map(id -> "block." + id).reduce((a, b) -> a + ", " + b).orElse("");
            writer.write(blockState + " is defined multiple times:\n");
            writer.write("  Properties: " + propertyIdsStr + "\n\n");
        }
    }

    private static void writeRenderLayerMismatches(BufferedWriter writer, AnalysisReport report) throws IOException {
        Map<String, ShaderAnalyzer.RenderLayerMismatch> mismatches = report.getRenderLayerMismatches();
        writer.write("----------------------------------------\n");
        writer.write("RENDER LAYER MISMATCHES (" + mismatches.size() + "):\n\n");
        writer.write("NOTE: The shader pack must be actively loaded in your shader loader\n");
        writer.write("(Iris, OptiFine, Oculus, etc.) at the time this report is generated\n");
        writer.write("for accurate render layer validation. If the shader is not loaded,\n");
        writer.write("mismatches may be incorrectly reported.\n\n");
        if (mismatches.isEmpty()) {
            writer.write("No render layer mismatches found.\n\n");
            return;
        }
        ArrayList<Map.Entry<String, ShaderAnalyzer.RenderLayerMismatch>> sortedMismatches = new ArrayList<Map.Entry<String, ShaderAnalyzer.RenderLayerMismatch>>(mismatches.entrySet());
        sortedMismatches.sort(Map.Entry.comparingByKey());
        for (Map.Entry entry : sortedMismatches) {
            String blockId = (String)entry.getKey();
            ShaderAnalyzer.RenderLayerMismatch mismatch = (ShaderAnalyzer.RenderLayerMismatch)entry.getValue();
            writer.write(blockId + "\n");
            writer.write("Expected: " + mismatch.expected() + " | Actual: " + mismatch.actual() + "\n\n");
        }
    }
}

