package com.iafenvoy.jupiter.config.entry;

import com.iafenvoy.jupiter.config.interfaces.ConfigBuilder;
import com.iafenvoy.jupiter.config.interfaces.ValueChangeCallback;
import com.iafenvoy.jupiter.interfaces.IConfigEntry;
import com.iafenvoy.jupiter.util.Comment;
import com.iafenvoy.jupiter.util.TextUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.function.Consumer;
import net.minecraft.class_2561;

public abstract class BaseEntry<T> implements IConfigEntry<T> {
    protected final class_2561 name;
    @Nullable
    protected String jsonKey;
    @Nullable
    protected class_2561 tooltip = null;
    protected boolean visible;
    protected final T defaultValue;
    protected T value;
    protected boolean restartRequired;
    protected final List<ValueChangeCallback<T>> callbacks = new ArrayList<>();

    protected BaseEntry(Builder<T, ?, ?> builder) {
        this.name = builder.name;
        this.defaultValue = builder.defaultValue;
        this.jsonKey = builder.jsonKey;
        this.tooltip = builder.tooltip;
        this.visible = builder.visible;
        this.restartRequired = builder.restartRequired;
        this.callbacks.addAll(builder.callbacks);
        this.value = this.newDefaultValue();
    }

    @Comment("Use builder instead")
    @Deprecated(forRemoval = true)
    public BaseEntry(@NotNull String nameKey, T defaultValue) {
        this.name = TextUtil.translatable(nameKey);
        this.jsonKey = nameKey;
        this.defaultValue = defaultValue;
        this.value = this.newDefaultValue();
    }

    @Comment("Use builder instead")
    @Deprecated(forRemoval = true)
    public BaseEntry<T> visible(boolean visible) {
        this.visible = visible;
        return this;
    }

    @Comment("Use builder instead")
    @Deprecated(forRemoval = true)
    public BaseEntry<T> json(String jsonKey) {
        this.jsonKey = jsonKey;
        return this;
    }

    @Comment("Use builder instead")
    @Deprecated(forRemoval = true)
    public BaseEntry<T> callback(Consumer<T> callback) {
        this.callbacks.add((v2, b1, b2) -> callback.accept(v2));
        return this;
    }

    @Comment("Use builder instead")
    @Deprecated(forRemoval = true)
    public BaseEntry<T> restartRequired() {
        this.restartRequired = true;
        return this;
    }

    @Override
    public void registerCallback(ValueChangeCallback<T> callback) {
        this.callbacks.add(callback);
    }

    @Override
    public void setValue(T value) {
        T oldValue = this.value;
        this.value = value;
        this.callbacks.forEach(x -> x.onValueChange(this.value, false, Objects.equals(this.value, this.defaultValue)));
    }

    @Override
    public @Nullable String getJsonKey() {
        return this.jsonKey;
    }

    @Override
    public class_2561 getName() {
        return this.restartRequired ? this.name.method_27661().method_27693(" ").method_10852(TextUtil.translatable("jupiter.screen.restart_required")) : this.name;
    }

    @Override
    public @Nullable class_2561 getTooltip() {
        return this.tooltip;
    }

    @Override
    public T getDefaultValue() {
        return this.defaultValue;
    }

    @Override
    public T getValue() {
        return this.value;
    }

    @Override
    public void reset() {
        this.value = this.newDefaultValue();
        this.callbacks.forEach(x -> x.onValueChange(this.value, true, true));
    }

    protected T newDefaultValue() {
        return this.defaultValue;
    }

    public static abstract class Builder<T, E extends BaseEntry<T>, B extends Builder<T, E, B>> implements ConfigBuilder<T, E, B> {
        protected final class_2561 name;
        protected final T defaultValue;
        protected T value;
        @Nullable
        protected String jsonKey;
        @Nullable
        protected class_2561 tooltip;
        protected boolean visible = true;
        protected boolean restartRequired;
        protected final List<ValueChangeCallback<T>> callbacks = new ArrayList<>();

        public Builder(String nameKey, T defaultValue) {
            this(TextUtil.translatable(nameKey), defaultValue);
            this.json(nameKey);
        }

        public Builder(class_2561 name, T defaultValue) {
            this.name = name;
            this.defaultValue = this.value = defaultValue;
        }

        public Builder(E parent) {
            this.name = parent.name;
            this.defaultValue = this.value = parent.defaultValue;
            this.jsonKey = parent.jsonKey;
            this.tooltip = parent.tooltip;
            this.visible = parent.visible;
            this.restartRequired = parent.restartRequired;
            this.callbacks.addAll(parent.callbacks);
        }

        public B visible(boolean visible) {
            this.visible = visible;
            return this.self();
        }

        public B json(String jsonKey) {
            this.jsonKey = jsonKey;
            return this.self();
        }

        public B restartRequired() {
            this.restartRequired = true;
            return this.self();
        }

        @Override
        public B tooltip(String tooltipKey) {
            return this.tooltip(TextUtil.translatable(tooltipKey));
        }

        @Override
        public B tooltip(class_2561 tooltipKey) {
            this.tooltip = tooltipKey;
            return this.self();
        }

        @Override
        public B callback(ValueChangeCallback<T> callback) {
            this.callbacks.add(callback);
            return this.self();
        }

        @Override
        public B value(T value) {
            this.value = value;
            return this.self();
        }

        public abstract B self();

        protected abstract E buildInternal();

        @Override
        public E build() {
            E e = this.buildInternal();
            if (this.value != null) e.value = this.value;
            return e;
        }
    }
}
