/*
 * Decompiled with CFR 0.152.
 */
package net.cufufy.pronouns.common.store;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import net.cufufy.pronouns.api.PronounParser;
import net.cufufy.pronouns.api.set.PronounSet;
import net.cufufy.pronouns.api.supplier.PronounSupplier;
import net.cufufy.pronouns.common.ProNouns;
import net.cufufy.pronouns.common.platform.config.Config;
import net.cufufy.pronouns.common.store.CachedPronounStore;
import net.cufufy.pronouns.common.util.UuidUtil;
import net.cufufy.pronouns.shadow.hikari.HikariDataSource;
import org.jetbrains.annotations.NotNull;

public class MySqlPronounStore
implements CachedPronounStore,
AutoCloseable {
    private final HikariDataSource dataSource;
    private final Map<UUID, List<PronounSet>> cache = new ConcurrentHashMap<UUID, List<PronounSet>>();
    private final ProNouns plugin;
    private static final PronounParser parser = new PronounParser(PronounSet.builtins);
    private Instant lastTimestamp = Instant.now();

    public MySqlPronounStore(ProNouns plugin, Config.MySqlConnectionInfo connectionInfo) {
        try {
            Class.forName("com.mysql.cj.jdbc.Driver");
        }
        catch (ClassNotFoundException e) {
            throw new RuntimeException(e);
        }
        this.plugin = plugin;
        this.dataSource = new HikariDataSource();
        this.dataSource.setJdbcUrl(connectionInfo.jdbcUrl());
        this.dataSource.setUsername(connectionInfo.username());
        this.dataSource.setPassword(connectionInfo.password());
        this.dataSource.addDataSourceProperty("cachePrepStmts", "true");
        this.dataSource.addDataSourceProperty("prepStmtCacheSize", "250");
        this.dataSource.addDataSourceProperty("prepStmtCacheSqlLimit", "2048");
        this.dataSource.addDataSourceProperty("useServerPrepStmts ", "true");
        this.dataSource.addDataSourceProperty("rewriteBatchedStatements", "true");
        try (Connection con = this.dataSource.getConnection();){
            con.prepareStatement("CREATE TABLE IF NOT EXISTS pronouns (\n    player BINARY(16) PRIMARY KEY,\n    pronouns TEXT NOT NULL,\n    last_updated_from TEXT NOT NULL,\n    last_updated_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP\n)\n").execute();
        }
        catch (SQLException e) {
            throw new RuntimeException(e);
        }
        plugin.platform().logger().info("Connected to MySQL");
        plugin.executorService().scheduleWithFixedDelay(this::poll, 10L, 10L, TimeUnit.SECONDS);
    }

    private void push(UUID uuid, List<PronounSet> sets) {
        try (Connection con = this.dataSource.getConnection();){
            if (sets.size() == 0) {
                PreparedStatement stmt = con.prepareStatement("DELETE FROM pronouns WHERE player=?");
                stmt.setBytes(1, UuidUtil.toBytes(uuid));
                stmt.execute();
                return;
            }
            PreparedStatement stmt = con.prepareStatement("REPLACE INTO pronouns (player, pronouns, last_updated_from) VALUES (?, ?, ?)");
            stmt.setBytes(1, UuidUtil.toBytes(uuid));
            stmt.setString(2, sets.stream().map(PronounSet::toFullString).collect(Collectors.joining(";")));
            stmt.setString(3, "pronouns_plugin_instance");
            stmt.execute();
        }
        catch (SQLException e) {
            this.plugin.platform().logger().error("Failed to write pronouns to MySQL: " + e.getMessage());
        }
    }

    private void pushAll(Map<UUID, List<PronounSet>> sets) {
        try (Connection con = this.dataSource.getConnection();){
            PreparedStatement stmt = con.prepareStatement("REPLACE INTO pronouns (player, pronouns, last_updated_from) VALUES (?, ?, ?)");
            for (Map.Entry<UUID, List<PronounSet>> entry : sets.entrySet()) {
                stmt.setBytes(1, UuidUtil.toBytes(entry.getKey()));
                stmt.setString(2, entry.getValue().stream().map(PronounSet::toFullString).collect(Collectors.joining(";")));
                stmt.setString(3, "pronouns_plugin_instance");
                stmt.addBatch();
            }
            stmt.executeBatch();
        }
        catch (SQLException e) {
            this.plugin.platform().logger().error("Failed to write pronouns to MySQL: " + e.getMessage());
        }
    }

    private void poll() {
        try (Connection con = this.dataSource.getConnection();){
            if (this.cache.size() == 0) {
                return;
            }
            PreparedStatement stmt = con.prepareStatement("SELECT * FROM pronouns WHERE last_updated_at > ? AND last_updated_from != ?");
            stmt.setTimestamp(1, Timestamp.from(this.lastTimestamp));
            stmt.setString(2, "pronouns_plugin_instance");
            ResultSet results = stmt.executeQuery();
            while (results.next()) {
                UUID uuid = UuidUtil.fromBytes(results.getBytes("player"));
                if (!this.cache.containsKey(uuid)) continue;
                List<PronounSet> newSets = parser.parse(results.getString("pronouns"));
                this.cache.put(uuid, newSets);
                this.plugin.platform().logger().info("Player " + this.plugin.platform().getPlayer(uuid).get().name() + " changed pronouns to " + PronounSet.format(newSets) + " on another server");
            }
            this.lastTimestamp = Instant.now();
        }
        catch (Exception e) {
            this.plugin.platform().logger().error("Failed to update pronoun cache from MySQL: " + e.getMessage());
        }
    }

    @Override
    public void addPronouns(UUID player, @NotNull List<PronounSet> pronounsToAdd) {
        if (pronounsToAdd.isEmpty()) {
            return;
        }
        ArrayList<PronounSet> currentSets = new ArrayList<PronounSet>(this.sets(player));
        if (currentSets.size() == 1 && ((PronounSet)currentSets.get(0)).equals(PronounSet.Builtins.UNSET)) {
            if (pronounsToAdd.size() == 1 && pronounsToAdd.get(0).equals(PronounSet.Builtins.UNSET)) {
                return;
            }
            currentSets.clear();
        }
        for (PronounSet toAdd : pronounsToAdd) {
            if (currentSets.contains(toAdd)) continue;
            currentSets.add(toAdd);
        }
        if (currentSets.isEmpty()) {
            this.cache.remove(player);
        } else {
            this.cache.put(player, currentSets);
        }
        List<PronounSet> setsToPush = List.copyOf(currentSets);
        this.plugin.executorService().submit(() -> this.push(player, setsToPush));
    }

    @Override
    public void removePronouns(UUID player, @NotNull List<PronounSet> pronounsToRemove) {
        if (pronounsToRemove.isEmpty()) {
            return;
        }
        ArrayList<PronounSet> currentSets = new ArrayList<PronounSet>(this.sets(player));
        if (currentSets.size() == 1 && ((PronounSet)currentSets.get(0)).equals(PronounSet.Builtins.UNSET)) {
            if (pronounsToRemove.contains(PronounSet.Builtins.UNSET)) {
                this.cache.remove(player);
                this.plugin.executorService().submit(() -> this.push(player, Collections.emptyList()));
            }
            return;
        }
        boolean changed = currentSets.removeAll(pronounsToRemove);
        if (changed) {
            if (currentSets.isEmpty()) {
                this.cache.remove(player);
            } else {
                this.cache.put(player, currentSets);
            }
            List<PronounSet> setsToPush = List.copyOf(currentSets);
            this.plugin.executorService().submit(() -> this.push(player, setsToPush));
        }
    }

    @Override
    public PronounSupplier predefined() {
        return PronounSet.builtins;
    }

    @Override
    public List<PronounSet> sets(UUID player) {
        return this.cache.getOrDefault(player, UNSET_LIST);
    }

    @Override
    public void set(UUID player, @NotNull List<PronounSet> sets) {
        if (sets.size() == 0) {
            this.cache.remove(player);
        } else {
            this.cache.put(player, sets);
        }
        this.plugin.executorService().submit(() -> this.push(player, sets));
    }

    @Override
    public void setAll(Map<UUID, List<PronounSet>> sets) {
        sets.forEach(this.cache::putIfAbsent);
        this.plugin.executorService().submit(() -> this.pushAll(sets));
    }

    @Override
    public Map<UUID, List<PronounSet>> dump() {
        throw new RuntimeException("L + ratio + get better");
    }

    @Override
    public void onPlayerJoin(UUID uuid) {
        try (Connection con = this.dataSource.getConnection();){
            PreparedStatement stmt = con.prepareStatement("SELECT pronouns FROM pronouns WHERE player=?");
            stmt.setBytes(1, UuidUtil.toBytes(uuid));
            ResultSet resultSet = stmt.executeQuery();
            if (!resultSet.next()) {
                return;
            }
            this.cache.put(uuid, parser.parse(resultSet.getString("pronouns")));
        }
        catch (SQLException e) {
            this.plugin.platform().logger().error("Failed to fetch pronouns from MySQL: " + e.getMessage());
        }
    }

    @Override
    public void onPlayerLeave(UUID uuid) {
        this.cache.remove(uuid);
    }

    @Override
    public void close() {
        this.dataSource.close();
    }
}

