#version 330 core

#import <sodium:include/fog.glsl>
#import <sodium:include/chunk_vertex.glsl>
#import <sodium:include/chunk_matrices.glsl>

out vec4 v_Color;
out vec2 v_TexCoord;
out vec2 v_LightCoord;

flat out uint v_Material;

out vec3 inChunkPos;
out vec4 inWorldPos;
out vec4 inScreenPos;
out float fadeFactor;

uniform int u_FogShape;
uniform vec3 u_RegionOffset;
uniform vec2 u_TexCoordShrink;
uniform int u_CurrentTime;
uniform float u_FadePeriodInv;

layout(std140) uniform ChunkData {
    ivec4 u_chunkFades[64]; // Packing into ivec4 is needed to avoid wasting 3KB...
};

uniform sampler2D u_LightTex; // The light map texture sampler

uvec3 _get_relative_chunk_coord(uint pos) {
    // Packing scheme is defined by LocalSectionIndex
    return uvec3(pos) >> uvec3(5u, 0u, 2u) & uvec3(7u, 3u, 7u);
}

vec3 _get_draw_translation(uint pos) {
    return _get_relative_chunk_coord(pos) * vec3(16.0);
}

void main() {
    _vert_init();

    inChunkPos = _vert_position.xyz;
    inWorldPos = vec4(_vert_position + u_RegionOffset + _get_draw_translation(_draw_id), 1.0);
    inScreenPos = u_ProjectionMatrix * u_ModelViewMatrix * inWorldPos;

    #ifdef USE_FOG
    int chunkId = int(_draw_id);
    int chunkFade = u_chunkFades[chunkId >> 2][chunkId & 3];
    int fadeTime = u_CurrentTime - chunkFade;
    float elapsed = float(fadeTime);
    float fade = clamp(float(u_CurrentTime - chunkFade) * u_FadePeriodInv, 0.0, 1.0);
    fadeFactor = (chunkFade < 0) ? 1.0 : fade;
    #endif

    // Transform the vertex position into model-view-projection space
    gl_Position = inScreenPos;

    // Add the light color to the vertex color, and pass the texture coordinates to the fragment shader
    v_Color = _vert_color;
    v_TexCoord = (_vert_tex_diffuse_coord_bias * u_TexCoordShrink) + _vert_tex_diffuse_coord;
    v_LightCoord = _vert_tex_light_coord;

    v_Material = _material_params;
}