#version 150

in vec2 pass_textureCoords;

out vec4 out_color;

uniform vec2 offset;
uniform vec2 texelSize;
uniform sampler2D imageMap;
uniform float near;
uniform float far;
uniform float p00;
uniform float blurWorldDistance;

// formula by x.com/FreyaHolmer/status/1820157167682388210
float screenSpaceRadius(float depth, int imageWidth) {
    return abs((imageWidth * p00) / (2.0 * depth));
}

float linearEyeDepth(float depth) {
    float z = depth * 2.0 - 1.0;
    return (2.0 * near * far) / (far + near - z * (far - near));
}

float gaussianKernel(int x, float sigma) {
    float c = 2 * sigma * sigma;
    return exp(-x * x / c);
}

void gaussian1DBlur(vec2 offset, float depthCenter, inout float depthBlurred, inout float kernelAmount) {
	int imageWidth = textureSize(imageMap, 0).x;
	float depthCenterLinear = linearEyeDepth(depthCenter);
	// adjusts the point at which particles can merge in the depth buffer
	float depthStrength = 6.0;
	// max blur radius for very close particles
	int maxRadius = imageWidth / 10;
	
	float screenRadius = screenSpaceRadius(depthCenterLinear, imageWidth) * blurWorldDistance;
	int radius = min(maxRadius, int(round(screenRadius)));
	float radiusFraction = max(0.0, radius - screenRadius);
	
	float sigmaStrength = 1.0;
	float sigma = max(0.0000001, (radius - radiusFraction) / (6.0 * sigmaStrength));

	for (int i = -radius; i <= radius; i++) {
		vec2 texelOffset = offset * texelSize * i;
		float depthOffset = texture(imageMap, pass_textureCoords + texelOffset).r;
		
		float depthOffsetLinear = linearEyeDepth(depthOffset);
		float depthDiff = depthCenterLinear - depthOffsetLinear;
		float weightDepth = exp(-(depthDiff * depthDiff * depthStrength));
		
		float kernelWeight = gaussianKernel(i, sigma);
		float weight = kernelWeight * weightDepth;
		depthBlurred += depthOffset * weight;
		kernelAmount += weight;
	}
}

void main(void) {
	float depthCenter = texture(imageMap, pass_textureCoords).r;
	float depthBlurred = 0.0;
	float kernelAmount = 0.0;
	
	if (depthCenter != 1.0) {
		gaussian1DBlur(offset, depthCenter, depthBlurred, kernelAmount);
	}
	
	if (kernelAmount == 0.0) {
		out_color = vec4(depthCenter, 1.0, 1.0, 1.0);
	} else {
		out_color = vec4(depthBlurred / kernelAmount, 1.0, 1.0, 1.0);
	}
}