package net.xun.lib.common.api.item.armor;

import net.minecraft.core.Holder;
import net.minecraft.world.item.ArmorItem;
import net.minecraft.world.item.ArmorMaterial;
import net.minecraft.world.item.Item;
import net.xun.lib.common.api.registries.LazyItemReference;
import net.xun.lib.common.internal.item.ItemRegistrar;
import net.xun.lib.common.internal.platform.RegistrationServices;

import java.util.EnumMap;
import java.util.List;
import java.util.Map;
import java.util.function.Supplier;
import java.util.stream.Collectors;

public class ArmorSet {

    private final String name;
    private final Holder<ArmorMaterial> material;
    private final int durabilityFactor;
    private final Item.Properties properties;
    private final ArmorConfigurator configuration;
    private final Map<ArmorType, LazyItemReference<? extends Item>> armors = new EnumMap<>(ArmorType.class);
    private final ItemRegistrar registrar;

    public ArmorSet(String name, Holder<ArmorMaterial> material, int durabilityFactor, Item.Properties properties, ArmorConfigurator configuration, ItemRegistrar registrar) {
        this.name = name;
        this.material = material;
        this.durabilityFactor = durabilityFactor;
        this.properties = properties;
        this.configuration = configuration;
        this.registrar = registrar;
    }

    public static class Builder {
        private final String name;
        private final Holder<ArmorMaterial> material;
        private int durabilityFactor;
        private Item.Properties properties;
        private ArmorConfigurator configuration = ArmorConfigurator.DEFAULT;
        private final ItemRegistrar registrar;

        public Builder(String name, Holder<ArmorMaterial> material) {
            this.name = name;
            this.material = material;
            this.registrar = RegistrationServices.getItemRegistrar();
        }

        public Builder withDurabilityFactor(int durabilityFactor) {
            this.durabilityFactor = durabilityFactor;
            return this;
        }

        public Builder withItemProperties(Item.Properties properties) {
            this.properties = properties;
            return this;
        }

        public Builder withConfiguration(ArmorConfigurator configuration) {
            this.configuration = configuration;
            return this;
        }

        public ArmorSet build() {
            return new ArmorSet(this.name, this.material, this.durabilityFactor, this.properties, this.configuration, this.registrar);
        }
    }

    public ArmorSet registerAll() {
        for (ArmorType type : ArmorType.values()) {
            final String fullName = name + type.getNameSuffix();

            LazyItemReference<Item> reference = new LazyItemReference<>(fullName, () -> configuration.createArmor(type, material, durabilityFactor, properties));

            registrar.registerItem(fullName, reference::get);
            armors.put(type, reference);
        }
        return this;
    }

    public Supplier<ArmorItem> helmet() {
        return getArmor(ArmorType.HELMET);
    }

    public Supplier<ArmorItem> chestplate() {
        return getArmor(ArmorType.CHESTPLATE);
    }

    public Supplier<ArmorItem> leggings() {
        return getArmor(ArmorType.LEGGINGS);
    }

    public Supplier<ArmorItem> boots() {
        return getArmor(ArmorType.BOOTS);
    }

    @SuppressWarnings("unchecked")
    private Supplier<ArmorItem> getArmor(ArmorType type) {
        return (Supplier<ArmorItem>) armors.get(type);
    }

    public List<Item> getAll() {
        return armors.values().stream().map(LazyItemReference::get).collect(Collectors.toList());
    }
}
