/*
 * Decompiled with CFR 0.152.
 */
package dev.ultreon.devices.core;

import com.google.common.base.Preconditions;
import dev.ultreon.devices.core.FS;
import dev.ultreon.devices.core.LockKey;
import dev.ultreon.devices.core.VirtualBlockDevice;
import dev.ultreon.devices.core.io.Path;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.lang.ref.Cleaner;
import java.nio.ByteBuffer;
import java.nio.file.DirectoryNotEmptyException;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.Files;
import java.nio.file.NoSuchFileException;
import java.nio.file.NotDirectoryException;
import java.nio.file.OpenOption;
import java.nio.file.StandardOpenOption;
import java.util.Collections;
import java.util.Iterator;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import net.minecraft.util.Unit;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.jnode.driver.Device;
import org.jnode.driver.DeviceAPI;
import org.jnode.driver.block.BlockDeviceAPI;
import org.jnode.driver.virtual.VirtualDevice;
import org.jnode.fs.FSDirectory;
import org.jnode.fs.FSEntry;
import org.jnode.fs.FSFile;
import org.jnode.fs.FileSystem;
import org.jnode.fs.FileSystemException;
import org.jnode.fs.ext2.BlockSize;
import org.jnode.fs.ext2.Ext2Directory;
import org.jnode.fs.ext2.Ext2Entry;
import org.jnode.fs.ext2.Ext2FileSystem;
import org.jnode.fs.ext2.Ext2FileSystemFormatter;
import org.jnode.fs.ext2.Ext2FileSystemType;
import org.jnode.fs.spi.AbstractFileSystem;
import org.slf4j.LoggerFactory;

public class Ext2FS
implements FS {
    private final FileSystem<?> fs;
    private final ConcurrentMap<Path, Unit> locks = new ConcurrentHashMap<Path, Unit>();

    private Ext2FS(Ext2FileSystem fs) {
        this.fs = fs;
        Cleaner cleaner = Cleaner.create();
        cleaner.register(this, () -> {
            try {
                fs.close();
            }
            catch (IOException e) {
                LoggerFactory.getLogger(Ext2FS.class).error("Failed to close filesystem", (Throwable)e);
            }
        });
    }

    public static Ext2FS open(java.nio.file.Path filePath) throws IOException, FileSystemException {
        return Ext2FS.open(false, filePath);
    }

    public static Ext2FS open(boolean readOnly, java.nio.file.Path filePath) throws IOException, FileSystemException {
        VirtualDevice device = new VirtualDevice("MineDisk");
        VirtualBlockDevice blockDevice = new VirtualBlockDevice(filePath.toFile().getAbsolutePath(), Files.size(filePath));
        device.registerAPI(BlockDeviceAPI.class, (DeviceAPI)blockDevice);
        Ext2FileSystemType type = new Ext2FileSystemType();
        Ext2FileSystem fs = new Ext2FileSystem((Device)device, readOnly, type);
        fs.read();
        return new Ext2FS(fs);
    }

    public static Ext2FS openForced(java.nio.file.Path filePath) throws IOException, FileSystemException {
        VirtualDevice device = new VirtualDevice("MineDisk");
        VirtualBlockDevice blockDevice = new VirtualBlockDevice(filePath.toFile().getAbsolutePath(), Files.size(filePath));
        device.registerAPI(BlockDeviceAPI.class, (DeviceAPI)blockDevice);
        Ext2FileSystemType type = new Ext2FileSystemType();
        Ext2FileSystem fs = new Ext2FileSystem((Device)device, false, type);
        fs.getSuperblock().setState(1);
        fs.flush();
        fs.read();
        return new Ext2FS(fs);
    }

    public static Ext2FS format(java.nio.file.Path filePath, long diskSize) throws IOException, FileSystemException {
        if (diskSize <= 16384L) {
            throw new IllegalArgumentException("Disk size must be greater than 16 KiB");
        }
        VirtualDevice device = new VirtualDevice("MineDisk");
        device.registerAPI(BlockDeviceAPI.class, (DeviceAPI)new VirtualBlockDevice(filePath.toFile().getAbsolutePath(), diskSize));
        VirtualBlockDevice blockDevice = new VirtualBlockDevice(filePath.toFile().getAbsolutePath(), diskSize);
        Ext2FileSystemFormatter formatter = new Ext2FileSystemFormatter(BlockSize._1Kb);
        Ext2FileSystem fs = formatter.format((Device)device);
        return new Ext2FS(fs);
    }

    @Override
    public void close() throws IOException {
        this.fs.close();
    }

    @Override
    public InputStream read(Path path, OpenOption ... options) throws IOException {
        FSFile file = this.getFileAt(path);
        if (file == null) {
            throw new NoSuchFileException("File not found: " + String.valueOf(path));
        }
        return new FSInputStream(file, options);
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    @Override
    public OutputStream write(Path path, OpenOption ... options) throws IOException {
        boolean create = false;
        boolean createNew = false;
        for (OpenOption option : options) {
            if (option == StandardOpenOption.CREATE) {
                create = true;
                continue;
            }
            if (option != StandardOpenOption.CREATE_NEW) continue;
            createNew = true;
        }
        if (create && createNew) {
            throw new IllegalArgumentException("Cannot create and createNew");
        }
        if (create) {
            Ext2Directory parentDir = this.getDirectoryAt(path.getParent());
            if (parentDir == null) {
                throw new NoSuchFileException(path.getParent().toString(), null, "parent directory not found");
            }
            Ext2Entry entry = (Ext2Entry)parentDir.getEntry(path.getFileName().toString());
            if (entry == null) {
                file = parentDir.addFile(path.getFileName().toString()).getFile();
                return new FSOutputStream(file, options);
            } else {
                if (!entry.isFile()) throw new FileAlreadyExistsException("Directory already exists: " + String.valueOf(path));
                file = entry.getFile();
            }
            return new FSOutputStream(file, options);
        } else if (createNew) {
            Ext2Directory parentDir = this.getDirectoryAt(path.getParent());
            if (parentDir == null) {
                throw new NoSuchFileException(path.getParent().toString(), null, "parent directory not found");
            }
            if (parentDir.getEntry(path.getFileName().toString()) != null) {
                throw new FileAlreadyExistsException("File already exists: " + String.valueOf(path));
            }
            file = parentDir.addFile(path.getFileName().toString()).getFile();
            return new FSOutputStream(file, options);
        } else {
            file = this.getFileAt(path);
            if (file != null) return new FSOutputStream(file, options);
            throw new NoSuchFileException(path.toString(), null, "file not found");
        }
    }

    @Override
    public boolean exists(Path path) {
        try {
            return this.getFsEntry(path) != null;
        }
        catch (IOException e) {
            return false;
        }
    }

    private FSFile getFileAt(Path path) throws IOException {
        Ext2Entry entry = this.getFsEntry(path);
        if (entry == null) {
            return null;
        }
        return entry.getFile();
    }

    private Ext2Directory getDirectoryAt(Path path) throws IOException {
        Preconditions.checkNotNull((Object)path, (Object)"path");
        Ext2Entry fsEntry = this.getFsEntry(path);
        if (fsEntry == null) {
            return null;
        }
        if (!fsEntry.isDirectory()) {
            throw new IllegalArgumentException("Path is not a directory: " + String.valueOf(path));
        }
        return (Ext2Directory)fsEntry.getDirectory();
    }

    @Nullable
    private Ext2Entry getFsEntry(Path path) throws IOException {
        String string = path.toString().replace("\\", "/");
        if (!string.startsWith("/")) {
            path = Path.of("/" + String.valueOf(path));
        }
        if (path.getParent() == null) {
            return (Ext2Entry)this.fs.getRootEntry();
        }
        Ext2Directory root = (Ext2Directory)this.fs.getRootEntry().getDirectory();
        for (String s : path.getParent()) {
            Ext2Entry entry = (Ext2Entry)root.getEntry(s.toString());
            if (!entry.isDirectory()) {
                return null;
            }
            root = (Ext2Directory)entry.getDirectory();
            if (root != null) continue;
            return null;
        }
        return (Ext2Entry)root.getEntry(path.getFileName().toString());
    }

    @Override
    public void flush() throws IOException {
        if (this.fs instanceof AbstractFileSystem) {
            ((AbstractFileSystem)this.fs).flush();
        }
    }

    @Override
    public void createFile(Path path, byte[] data) throws IOException {
        Ext2Directory parentDir;
        String string = path.toString().replace("\\", "/");
        if (!string.startsWith("/")) {
            path = Path.of("/" + String.valueOf(path));
        } else if (string.equals("/")) {
            throw new IOException("Invalid path for file: " + String.valueOf(path));
        }
        Ext2Directory ext2Directory = parentDir = path.getParent() == null ? (Ext2Directory)this.fs.getRootEntry().getDirectory() : this.getDirectoryAt(path.getParent());
        if (parentDir == null) {
            throw new NoSuchFileException(path.getParent().toString(), null, "parent directory not found");
        }
        FSFile file = parentDir.addFile(path.getFileName().toString()).getFile();
        file.flush();
        ByteBuffer buffer = ByteBuffer.allocate(1024);
        int i = 0;
        while ((double)i < Math.ceil((double)data.length / 1024.0)) {
            buffer.clear();
            buffer.put(data, i * 1024, Math.min(1024, data.length - i * 1024));
            buffer.flip();
            file.write(0L, buffer);
            file.flush();
            ++i;
        }
        FileSystem<?> fileSystem = this.fs;
        if (fileSystem instanceof AbstractFileSystem) {
            AbstractFileSystem absFs = (AbstractFileSystem)fileSystem;
            absFs.flush();
        }
    }

    @Override
    public void createDirectory(Path path) throws IOException {
        Ext2Directory parentDir;
        String string = path.toString().replace("\\", "/");
        if (!string.startsWith("/")) {
            path = Path.of("/" + String.valueOf(path));
        }
        Ext2Directory ext2Directory = parentDir = path.getParent() == null ? (Ext2Directory)this.fs.getRootEntry().getDirectory() : this.getDirectoryAt(path.getParent());
        if (path.getParent() != null && parentDir != null) {
            parentDir.addDirectory(path.getFileName().toString());
            parentDir.flush();
        } else {
            FSDirectory directory = this.fs.getRootEntry().getDirectory();
            directory.addDirectory(path.getFileName().toString());
            directory.flush();
        }
        FileSystem<?> fileSystem = this.fs;
        if (fileSystem instanceof AbstractFileSystem) {
            AbstractFileSystem absFs = (AbstractFileSystem)fileSystem;
            absFs.flush();
        }
    }

    @Override
    public Iterator<String> listDirectory(Path of) throws IOException {
        Ext2Directory dir = this.getDirectoryAt(of);
        if (dir == null) {
            return Collections.emptyIterator();
        }
        return new DirNameIterator(dir);
    }

    @Override
    public void delete(Path path) throws IOException {
        Ext2Entry parent;
        Ext2Entry ext2Entry = parent = path.getParent() == null ? (Ext2Entry)this.fs.getRootEntry() : this.getFsEntry(path.getParent());
        if (parent == null) {
            return;
        }
        if (!parent.isDirectory()) {
            throw new NotDirectoryException("Path is not a directory: " + String.valueOf(path));
        }
        Ext2Entry entry = (Ext2Entry)parent.getDirectory().getEntry(path.getFileName().toString());
        if (entry.isDirectory()) {
            if (entry.getDirectory().iterator().hasNext()) {
                throw new DirectoryNotEmptyException("Directory is not empty: " + String.valueOf(path));
            }
            parent.getDirectory().remove(path.getFileName().toString());
            return;
        }
        if (!entry.isFile()) {
            throw new IOException("Path is not a file: " + String.valueOf(path));
        }
        parent.getDirectory().remove(path.getFileName().toString());
    }

    @Override
    public long size(Path path) throws IOException {
        Ext2Entry entry = this.getFsEntry(path);
        if (entry == null) {
            return -1L;
        }
        if (!entry.isFile()) {
            return -1L;
        }
        return entry.getFile().getLength();
    }

    @Override
    public void rename(Path from, String name) throws IOException {
        Ext2Entry parent;
        if (name.contains("/")) {
            throw new IOException("Invalid name: " + name);
        }
        Ext2Entry ext2Entry = parent = from.getParent() == null ? (Ext2Entry)this.fs.getRootEntry() : this.getFsEntry(from.getParent());
        if (parent == null) {
            return;
        }
        if (!parent.isDirectory()) {
            throw new NotDirectoryException("Path is not a directory: " + String.valueOf(from));
        }
        Ext2Entry entry = (Ext2Entry)parent.getDirectory().getEntry(from.getFileName().toString());
        if (entry == null) {
            return;
        }
        entry.setName(name);
        parent.getDirectory().flush();
    }

    @Override
    public boolean isFolder(Path path) throws IOException {
        Ext2Entry entry = this.getFsEntry(path);
        return entry != null && entry.isDirectory();
    }

    @Override
    public boolean isFile(Path path) throws IOException {
        Ext2Entry entry = this.getFsEntry(path);
        return entry != null && entry.isFile();
    }

    @Override
    public boolean isSymbolicLink(Path path) {
        return false;
    }

    @Override
    @Nullable
    public LockKey lock(String path) throws IOException {
        if (path == null) {
            throw new IllegalArgumentException("Path must not be null");
        }
        Path pathObj = Path.of(path);
        if (!this.exists(pathObj)) {
            return null;
        }
        if (this.locks.containsKey(pathObj)) {
            throw new IOException("Path is already locked: " + String.valueOf(pathObj));
        }
        this.locks.put(pathObj, Unit.INSTANCE);
        return new LockKey(pathObj);
    }

    @Override
    public void unlock(String directory) {
        if (directory == null) {
            throw new IllegalArgumentException("Path must not be null");
        }
        this.locks.remove(Path.of(directory));
    }

    @Override
    public boolean isLocked(String directory) {
        return this.locks.containsKey(Path.of(directory));
    }

    @Override
    public void setReadOnly(Path of, boolean b) throws IOException {
        @Nullable Ext2Entry dir = this.getFsEntry(of);
        if (dir == null) {
            return;
        }
        dir.getAccessRights().setReadable(true, false);
        dir.getAccessRights().setWritable(!b, false);
        dir.getParent().flush();
    }

    @Override
    public void setExecutable(Path of, boolean b) throws IOException {
        @Nullable Ext2Entry dir = this.getFsEntry(of);
        if (dir == null) {
            return;
        }
        if (!dir.isFile()) {
            throw new IOException("Path is not a file: " + String.valueOf(of));
        }
        dir.getAccessRights().setExecutable(b, false);
        dir.getParent().flush();
    }

    @Override
    public boolean isExecutable(Path of) throws IOException {
        Ext2Entry entry = this.getFsEntry(of);
        if (entry == null) {
            return false;
        }
        return entry.isFile() && entry.getAccessRights().canExecute();
    }

    @Override
    public boolean isWritable(Path of) throws IOException {
        Ext2Entry entry = this.getFsEntry(of);
        if (entry == null) {
            return false;
        }
        return entry.isFile() && entry.getAccessRights().canWrite();
    }

    @Override
    public boolean isReadable(Path of) throws IOException {
        Ext2Entry entry = this.getFsEntry(of);
        if (entry == null) {
            return false;
        }
        return entry.isFile() && entry.getAccessRights().canRead();
    }

    @Override
    public int getOwner(Path of) throws IOException {
        Ext2Entry entry = this.getFsEntry(of);
        if (entry == null) {
            throw new IOException("Path does not exist: " + String.valueOf(of));
        }
        return entry.getINode().getUid();
    }

    @Override
    public int getGroup(Path of) throws IOException {
        Ext2Entry entry = this.getFsEntry(of);
        if (entry == null) {
            throw new IOException("Path does not exist: " + String.valueOf(of));
        }
        return entry.getINode().getGid();
    }

    @Override
    public int getPermissions(Path of) throws IOException {
        Ext2Entry entry = this.getFsEntry(of);
        if (entry == null) {
            throw new IOException("Path does not exist: " + String.valueOf(of));
        }
        return entry.getINode().getMode();
    }

    @Override
    public void setPermissions(Path of, int mode) throws IOException {
        Ext2Entry entry = this.getFsEntry(of);
        if (entry == null) {
            throw new IOException("Path does not exist: " + String.valueOf(of));
        }
        entry.getINode().setMode(mode);
        entry.getParent().flush();
    }

    @Override
    public void setOwner(Path of, int uid, int gid) throws IOException {
        Ext2Entry entry = this.getFsEntry(of);
        if (entry == null) {
            throw new IOException("Path does not exist: " + String.valueOf(of));
        }
        entry.getINode().setUid(uid);
        entry.getINode().setGid(gid);
        entry.getParent().flush();
    }

    @Override
    public void setGroup(Path of, int gid) throws IOException {
        Ext2Entry entry = this.getFsEntry(of);
        if (entry == null) {
            throw new IOException("Path does not exist: " + String.valueOf(of));
        }
        entry.getINode().setGid(gid);
        entry.getParent().flush();
    }

    @Override
    public void setOwner(Path of, int uid) throws IOException {
        Ext2Entry entry = this.getFsEntry(of);
        if (entry == null) {
            throw new IOException("Path does not exist: " + String.valueOf(of));
        }
        entry.getINode().setUid(uid);
        entry.getParent().flush();
    }

    @Override
    public long getGeneration(Path of) throws IOException {
        Ext2Entry entry = this.getFsEntry(of);
        if (entry == null) {
            throw new IOException("Path does not exist: " + String.valueOf(of));
        }
        return entry.getINode().getGeneration();
    }

    @Override
    public void setGeneration(Path of, long generation) throws IOException {
        Ext2Entry entry = this.getFsEntry(of);
        if (entry == null) {
            throw new IOException("Path does not exist: " + String.valueOf(of));
        }
        entry.getINode().setGeneration(generation);
        entry.getParent().flush();
    }

    @Override
    public boolean isReadOnly(Path of) throws IOException {
        Ext2Entry entry = this.getFsEntry(of);
        if (entry == null) {
            throw new IOException("Path does not exist: " + String.valueOf(of));
        }
        return entry.getAccessRights().canRead() && !entry.getAccessRights().canWrite();
    }

    @Override
    public boolean canWrite(Path of) throws IOException {
        return this.isWritable(of);
    }

    @Override
    public boolean canRead(Path of) throws IOException {
        return this.isReadable(of);
    }

    @Override
    public boolean canExecute(Path of) throws IOException {
        Ext2Entry entry = this.getFsEntry(of);
        if (entry == null) {
            return false;
        }
        return entry.getAccessRights().canExecute();
    }

    @Override
    public long lastModified(Path path) throws IOException {
        Ext2Entry entry = this.getFsEntry(path);
        if (entry == null) {
            return 0L;
        }
        return entry.getLastModified();
    }

    @Override
    public long lastAccessed(Path path) throws IOException {
        Ext2Entry entry = this.getFsEntry(path);
        if (entry == null) {
            return 0L;
        }
        return entry.getLastAccessed();
    }

    @Override
    public long creationTime(Path path) throws IOException {
        Ext2Entry entry = this.getFsEntry(path);
        if (entry == null) {
            return 0L;
        }
        return entry.getINode().getCtime();
    }

    @Override
    public void setLastAccessed(Path path, long time) throws IOException {
        Ext2Entry entry = this.getFsEntry(path);
        if (entry == null) {
            return;
        }
        entry.setLastAccessed(time);
        entry.getParent().flush();
    }

    @Override
    public void setLastModified(Path path, long time) throws IOException {
        Ext2Entry entry = this.getFsEntry(path);
        if (entry == null) {
            return;
        }
        entry.setLastModified(time);
        entry.getParent().flush();
    }

    @Override
    public void setCreationTime(Path path, long time) throws IOException {
        Ext2Entry entry = this.getFsEntry(path);
        if (entry == null) {
            return;
        }
        entry.getINode().setCtime(time);
        entry.getParent().flush();
    }

    @Override
    public long getTotalSpace() throws IOException {
        return this.fs.getTotalSpace();
    }

    @Override
    public long getUsableSpace() throws IOException {
        return this.fs.getUsableSpace();
    }

    @Override
    public long getFreeSpace() throws IOException {
        return this.fs.getFreeSpace();
    }

    @Override
    public void move(Path source, Path destination) throws IOException {
        Ext2Entry sourceEntry = this.getFsEntry(source);
        Ext2Entry destinationEntry = this.getFsEntry(destination);
        if (sourceEntry == null || destinationEntry == null) {
            return;
        }
        try (InputStream in = this.read(source, new OpenOption[0]);){
            this.createFile(destination, in.readAllBytes());
        }
        sourceEntry.getParent().remove(sourceEntry.getName());
        sourceEntry.getParent().flush();
        destinationEntry.getParent().flush();
    }

    @Override
    public void copy(Path source, Path destination) throws IOException {
        Ext2Entry sourceEntry = this.getFsEntry(source);
        Ext2Entry destinationEntry = this.getFsEntry(destination);
        if (sourceEntry == null || destinationEntry == null) {
            return;
        }
        try (InputStream in = this.read(source, new OpenOption[0]);){
            this.createFile(destination, in.readAllBytes());
        }
        sourceEntry.getParent().flush();
        destinationEntry.getParent().flush();
    }

    @Override
    public void write(Path path, long offset, byte[] dataBytes) throws IOException {
        FSFile file;
        Ext2Entry entry = this.getFsEntry(path);
        if (entry == null) {
            return;
        }
        if (entry.isFile()) {
            file = entry.getFile();
            long length = file.getLength();
            if (offset > length) {
                throw new IOException("Offset out of range: " + offset);
            }
            if (offset + (long)dataBytes.length > length) {
                file.setLength(offset + (long)dataBytes.length);
            }
        } else {
            throw new IOException("Path is not a file: " + String.valueOf(path));
        }
        file.write(offset, ByteBuffer.wrap(dataBytes));
        file.flush();
        entry.getParent().flush();
    }

    @Override
    public void write(Path path, byte[] dataBytes) throws IOException {
        this.write(path, 0L, dataBytes);
    }

    @Override
    public void truncate(Path path, long size) throws IOException {
        Ext2Entry entry = this.getFsEntry(path);
        if (entry == null) {
            return;
        }
        if (entry.isFile()) {
            entry.getFile().setLength(size);
        }
    }

    @Override
    public void read(Path path, ByteBuffer buffer, long offset) throws IOException {
        Ext2Entry entry = this.getFsEntry(path);
        if (entry == null) {
            return;
        }
        if (entry.isFile()) {
            entry.getFile().read(offset, buffer);
        }
    }

    public FileSystem<?> getFileSystem() {
        return this.fs;
    }

    private static class FSInputStream
    extends InputStream {
        private final FSFile file;
        private final ByteBuffer buffer = ByteBuffer.allocate(1024);
        private long bufferOffset;
        private long fileOffset = 0L;
        private int markLimit;

        public FSInputStream(FSFile file, OpenOption[] options) throws IOException {
            this.file = file;
            for (OpenOption option : options) {
                if (option == StandardOpenOption.TRUNCATE_EXISTING) {
                    file.setLength(0L);
                    continue;
                }
                if (option == StandardOpenOption.READ) continue;
                throw new UnsupportedOperationException("Option not supported: " + String.valueOf(option));
            }
            this.bufferOffset = 0L;
            this.buffer.clear();
            if (this.fileOffset + (long)this.buffer.capacity() > file.getLength()) {
                this.buffer.limit((int)(file.getLength() - this.fileOffset));
            }
            file.read(this.fileOffset, this.buffer);
            this.buffer.flip();
        }

        @Override
        public int read() throws IOException {
            if (this.bufferOffset == (long)this.buffer.limit()) {
                this.bufferOffset = 0L;
                this.buffer.clear();
                if (this.fileOffset + (long)this.buffer.capacity() > this.file.getLength()) {
                    this.buffer.limit((int)(this.file.getLength() - this.fileOffset));
                }
                this.file.read(this.fileOffset, this.buffer);
                this.fileOffset += (long)this.buffer.capacity();
            }
            return this.buffer.get() & 0xFF;
        }

        @Override
        public int read(byte @NotNull [] b, int off, int len) throws IOException {
            int bytesRead = 0;
            if (this.fileOffset >= this.file.getLength()) {
                return -1;
            }
            while (bytesRead < len) {
                if (this.fileOffset >= this.file.getLength()) {
                    return bytesRead;
                }
                int remaining = this.buffer.remaining();
                int len0 = Math.min(len - bytesRead, remaining);
                if (len0 == 0) {
                    return -1;
                }
                int fileRemaining = (int)(this.file.getLength() - this.fileOffset);
                int len1 = Math.min(len0, fileRemaining);
                this.buffer.get(b, off + bytesRead, len1);
                this.bufferOffset += (long)len1;
                bytesRead += len1;
                this.fileOffset += (long)len1;
                if (this.bufferOffset == (long)this.buffer.capacity()) {
                    this.bufferOffset = 0L;
                    this.buffer.clear();
                    if (this.fileOffset + (long)this.buffer.capacity() > this.file.getLength()) {
                        this.buffer.limit((int)(this.file.getLength() - this.fileOffset));
                    }
                    this.file.read(this.fileOffset, this.buffer);
                }
                if (len0 >= remaining) continue;
                break;
            }
            return bytesRead;
        }

        @Override
        public int read(byte @NotNull [] b) throws IOException {
            return this.read(b, 0, b.length);
        }
    }

    private class FSOutputStream
    extends OutputStream {
        private final FSFile file;
        private final ByteBuffer buffer = ByteBuffer.allocate(1024);
        private boolean sync = false;
        private long bufferOffset = 0L;
        private long fileOffset = 0L;

        public FSOutputStream(FSFile file, OpenOption[] options) throws IOException {
            this.file = file;
            for (OpenOption option : options) {
                if (option == StandardOpenOption.TRUNCATE_EXISTING) {
                    file.setLength(0L);
                    continue;
                }
                if (option == StandardOpenOption.APPEND) {
                    this.fileOffset = file.getLength();
                    continue;
                }
                if (option == StandardOpenOption.SYNC) {
                    this.sync = true;
                    continue;
                }
                if (option == StandardOpenOption.WRITE || option == StandardOpenOption.CREATE || option == StandardOpenOption.CREATE_NEW) continue;
                throw new UnsupportedOperationException("Option not supported: " + String.valueOf(option));
            }
        }

        @Override
        public void write(int b) throws IOException {
            this.buffer.put((byte)b);
            ++this.bufferOffset;
            if (this.bufferOffset == (long)this.buffer.capacity()) {
                this.buffer.flip();
                this.file.write(this.fileOffset, this.buffer);
                this.buffer.clear();
                this.bufferOffset = 0L;
                this.fileOffset += (long)this.buffer.capacity();
            }
        }

        @Override
        public void write(byte @NotNull [] b) throws IOException {
            this.write(b, 0, b.length);
        }

        @Override
        public void flush() throws IOException {
            this.file.flush();
            FileSystem<?> fileSystem = Ext2FS.this.fs;
            if (fileSystem instanceof AbstractFileSystem) {
                AbstractFileSystem absFs = (AbstractFileSystem)fileSystem;
                absFs.flush();
            }
        }

        @Override
        public void close() throws IOException {
            this.file.flush();
            FileSystem<?> fileSystem = Ext2FS.this.fs;
            if (fileSystem instanceof AbstractFileSystem) {
                AbstractFileSystem absFs = (AbstractFileSystem)fileSystem;
                absFs.flush();
            }
        }
    }

    private static class DirNameIterator
    implements Iterator<String> {
        private final Iterator<FSEntry> iterator;

        public DirNameIterator(Ext2Directory dir) throws IOException {
            this.iterator = dir.iterator();
        }

        @Override
        public boolean hasNext() {
            return this.iterator.hasNext();
        }

        @Override
        public String next() {
            return this.iterator.next().getName();
        }
    }
}

